<?php

namespace Drupal\dkan_dataset_archiver_remote_storage\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\dkan_dataset_archiver\Event\ArchivePostSaveEvent;
use Drupal\dkan_dataset_archiver_remote_storage\Service\RemoteStorageService;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Subscriber to manage remote storage related events.
 */
class StorageSubscriber implements EventSubscriberInterface {

  /**
   * The remote storage service.
   *
   * @var \Drupal\dkan_dataset_archiver_remote_storage\Service\RemoteStorageService
   */
  protected $remoteStorageService;

  /**
   * The archiver settings.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $archiverSettings;


  /**
   * The dkan_dataset_archiver_remote_storage logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * {@inheritDoc}
   */
  public static function getSubscribedEvents(): array {
    $events = [];
    $events[ArchivePostSaveEvent::EVENT_NAME][] = ['addArchiveToFileMover', 10];

    return $events;
  }

  /**
   * Constructor.
   *
   * @param \Psr\Log\LoggerInterface $logger
   *   The dkan_dataset_archiver_remote_storage logger channel.
   * @param \Drupal\dkan_dataset_archiver_remote_storage\Service\RemoteStorageService $remoteStorageService
   *   The archive service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(
    LoggerInterface $logger,
    RemoteStorageService $remoteStorageService,
    ConfigFactoryInterface $config_factory,
  ) {
    $this->logger = $logger;
    $this->remoteStorageService = $remoteStorageService;
    $this->archiverSettings = $config_factory->get('dkan_dataset_archiver.settings');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('logger.channel.dkan_dataset_archiver_remote_storage'),
      $container->get('dkan_dataset_archiver_remote_storage.remote_storage_service'),
      $container->get('config.factory'),
    );
  }

  /**
   * Add the archive to the file mover service.
   *
   * @param \Drupal\dkan_dataset_archiver\Event\ArchivePostSaveEvent $event
   *   The archive post save event.
   */
  public function addArchiveToFileMover(ArchivePostSaveEvent $event): void {
    $archive = $event->getArchive();
    // Check if the archive has already remote stored (bypass recursion).
    if (($archive->isNew() || $archive->localFilesChanged()) && !$archive->hasRemoteStored()) {
      $this->remoteStorageService->addToRemoteFileSyncQueue($archive);
    }
  }

}
