<?php

namespace Drupal\dkan_dataset_archiver\Entity;

use Drupal\Core\Entity\EditorialContentEntityBase;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\File\FileExists;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\dkan_dataset_archiver\Event\ArchivePostSaveEvent;
use Drupal\dkan_dataset_archiver\Event\ArchivePreSaveEvent;
use Drupal\dkan_dataset_archiver\Service\Util;
use Drupal\link\Plugin\Field\FieldType\LinkItem;
use Drupal\path_alias\Entity\PathAlias;
use Drupal\user\UserInterface;

/**
 * Defines the DKAN Dataset Archive entity.
 *
 * @ingroup dda_archive
 *
 * @ContentEntityType(
 *   id = "dda_archive",
 *   label = @Translation("Dataset Archive"),
 *   label_plural = @Translation("Dataset Archives"),
 *   label_collection = @Translation("Dataset Archives"),
 *   handlers = {
 *     "storage" = "\Drupal\Core\Entity\Sql\SqlContentEntityStorage",
 *     "view_builder" = "Drupal\dkan_dataset_archiver\Entity\DdaArchiveViewBuilder",
 *     "views_data" = "Drupal\dkan_dataset_archiver\Entity\DdaArchiveViewsData",
 *     "form" = {
 *       "default" = "Drupal\dkan_dataset_archiver\Form\DdaArchiveForm",
 *       "add" = "Drupal\dkan_dataset_archiver\Form\DdaArchiveForm",
 *       "edit" = "Drupal\dkan_dataset_archiver\Form\DdaArchiveForm",
 *       "delete" = "Drupal\dkan_dataset_archiver\Form\DdaArchiveDeleteForm",
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\dkan_dataset_archiver\Routing\DdaArchiveHtmlRouteProvider",
 *     },
 *     "access" = "Drupal\dkan_dataset_archiver\DdaArchiveAccessControlHandler",
 *   },
 *   base_table = "dda_archive",
 *   data_table = "dda_archive_field_data",
 *   revision_table = "dda_archive_revision",
 *   revision_data_table = "dda_archive_field_revision",
 *   show_revision_ui = TRUE,
 *   translatable = TRUE,
 *   admin_permission = "administer dkan dataset archiver settings",
 *   entity_keys = {
 *     "id" = "id",
 *     "revision" = "vid",
 *     "label" = "name",
 *     "uuid" = "uuid",
 *     "uid" = "user_id",
 *     "langcode" = "langcode",
 *     "published" = "status",
 *   },
 *   revision_metadata_keys = {
 *     "revision_user" = "revision_user",
 *     "revision_created" = "revision_created",
 *     "revision_log_message" = "revision_log_message",
 *   },
 *   links = {
 *     "canonical" = "/admin/content/archive/{dda_archive}",
 *     "add-form" = "/admin/content/archive/add",
 *     "edit-form" = "/admin/content/archive/{dda_archive}/edit",
 *     "delete-form" = "/admin/content/archive/{dda_archive}/delete",
 *     "version-history" = "/admin/content/archive/{dda_archive}/revisions",
 *     "revision" = "/admin/content/archive/{dda_archive}/revisions/{dda_archive_revision}/view",
 *     "revision_revert" = "/admin/content/archive/{dda_archive}/revisions/{dda_archive_revision}/revert",
 *     "revision_delete" = "/admin/content/archive/{dda_archive}/revisions/{dda_archive_revision}/delete",
 *     "collection" = "/admin/content/archive",
 *   },
 *   field_ui_base_route = "entity.dda_archive.config_form",
 *   constraints = {
 *   }
 * )
 */
class DdaArchive extends EditorialContentEntityBase implements DdaArchiveInterface {

  /**
   * The archive service.
   *
   * @var \Drupal\dkan_dataset_archiver\Service\ArchiveService
   */
  protected $archiveService;

  /**
   * Configuration settings.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $archiverSettings;

  /**
   * Cache tag invalidator service.
   *
   * @var \Drupal\Core\Cache\CacheTagsInvalidatorInterface
   */
  protected $cacheInvalidator;

  /**
   * Flag indicating local files have been moved to remote storage this session.
   *
   * @var bool
   */
  protected $hasRemoteStored = FALSE;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  public $entityFieldManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  public $entityTypeManager;

  /**
   * The event dispatcher.
   *
   * @var \Symfony\Contracts\EventDispatcher\EventDispatcherInterface
   */
  public $eventDispatcher;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  public $fileSystem;

  /**
   * An array of datasets that have been loaded. Keyed by dataset ID.
   *
   * @var array
   */
  protected $loadedDatasets = [];

  /**
   * The metastore service.
   *
   * @var \Drupal\metastore\MetastoreService
   */
  public $metastoreService;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $pathAliasStorage;

  /**
   * {@inheritdoc}
   */
  public function __construct(array $values, $entity_type, $bundle = FALSE, $translations = []) {
    parent::__construct($values, $entity_type, $bundle, $translations);
    // Can't use Dependency Injection on a @ContentEntityType.
    // https://www.drupal.org/project/drupal/issues/2142515 provides history.
    $this->archiverSettings = \Drupal::config('dkan_dataset_archiver.settings');
    $this->archiveService = \Drupal::service('dkan_dataset_archiver.archive_service');
    $this->cacheInvalidator = \Drupal::service('cache_tags.invalidator');
    $this->entityTypeManager = \Drupal::service('entity_type.manager');
    $this->entityFieldManager = \Drupal::service('entity_field.manager');
    $this->fileSystem = \Drupal::service('file_system');
    $this->metastoreService = \Drupal::service('dkan.metastore.service');
    $this->pathAliasStorage = $this->entityTypeManager->getStorage('path_alias');
    $this->eventDispatcher = \Drupal::service('event_dispatcher');
  }

  /**
   * {@inheritdoc}
   */
  public function setHasRemoteStored(bool $hasRemoteStored): void {
    $this->hasRemoteStored = $hasRemoteStored;
  }

  /**
   * {@inheritdoc}
   */
  public function hasRemoteStored(): bool {
    return $this->hasRemoteStored;
  }

  /**
   * {@inheritdoc}
   */
  public static function preCreate(EntityStorageInterface $storage, array &$values): void {
    parent::preCreate($storage, $values);
    $values += [
      'user_id' => \Drupal::currentUser()->id(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  protected function urlRouteParameters($rel): array {
    $uri_route_parameters = parent::urlRouteParameters($rel);

    if ($rel === 'revision_revert') {
      $uri_route_parameters[$this->getEntityTypeId() . '_revision'] = $this->getRevisionId();
    }
    elseif ($rel === 'revision_delete') {
      $uri_route_parameters[$this->getEntityTypeId() . '_revision'] = $this->getRevisionId();
    }

    return $uri_route_parameters;
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage): void {
    parent::preSave($storage);

    // If no revision author has been set explicitly,
    // make the archive owner the revision author.
    if (!$this->getRevisionUser()) {
      $this->setRevisionUserId($this->getOwnerId());
    }

    if ($this->isNew() || empty($this->getName())) {
      // We only want to force the name on creation.  After that it can be
      // changed by an editor.
      // Blanking it out will also cause it to be rebuilt on save.
      $this->setName($this->buildArchiveName());
    }
    // Make sure we have a dataset modified date.
    if (empty($this->get('dataset_modified')->value)) {
      $this->set('dataset_modified', date('Y-m-d'));
    }
    // Make sure we have an archive_type.
    if (empty($this->get('archive_type')->value)) {
      $this->set('archive_type', 'individual');
    }
    if ($this->getArchiveType() === 'individual') {
      $level = $this->values["access level"] ?? $this->getDataset()->{'$.accessLevel'} ?? 'public';
      // Account for 'private' and map it to 'non-public'.
      $level = ($level === 'private') ? 'non-public' : $level;
      $this->set('access_level', $level);
    }
    $this->modifyArchiveByType();
    $this->matchStatusToDataset();
    if (!$this->isSyncing()) {
      // This is being created by hand for the first time.
      // Juggle the file location.
      $total_file_size = 0;
      $local_files = $this->getResourceFileItems();
      foreach ($local_files as $file) {
        /** @var \Drupal\file\Entity\File $file */
        $total_file_size += $file->getSize();
        $original_file_name = $file->getFileName();
        $original_uri = $file->getFileUri();

        $new_path = Util::createArchiveFilePath(
          $this->getArchiveType(),
          $this->get('aggregate_on')->value ?? '',
          $this->get('dataset_id')->value ?? '',
          $this->get('dataset_modified')->value ?? ''
        );
        $new_file_name = Util::createArchiveFilename(
          $this->getArchiveType(),
          $this->get('aggregate_of')->value ?? '',
          $this->get('aggregate_on')->value ?? '',
          $this->get('dataset_id')->value ?? '',
          $this->get('dataset_modified')->value ?? '',
          $original_file_name
        );
        // Do we need to move anything? If the paths match, then the file was
        // already processed.
        if (!str_contains($original_uri, $new_path)) {
          $new_uri = $this->adjustStorageLocation("{$new_path}/{$new_file_name}");
          $file->setFileUri($new_uri);
          $file->setFilename($new_file_name);
          $new_dir = pathinfo($new_uri, PATHINFO_DIRNAME);
          Util::prepareDirectory($new_dir);
          // Move the file.
          $this->fileSystem->move($original_uri, $new_uri, FileExists::Replace);
          $file->save();
        }
      }
      $this->set('size', $total_file_size);
    }

    // Trigger an event that can be used to perform actions pre save.
    $event = new ArchivePreSaveEvent($this);
    $this->eventDispatcher->dispatch($event, ArchivePreSaveEvent::EVENT_NAME);
  }

  /**
   * {@inheritdoc}
   */
  public function postSave(EntityStorageInterface $storage, $update = TRUE): void {
    parent::postSave($storage, $update);

    if ($update) {
      // This is an update, so we have to delete aliases first.
      $exists = $this->removeExistingAlias();
      if (!$exists) {
        $this->setAlias();
      }
    }
    else {
      $this->setAlias();
    }

    // Clear relevant caches.
    $tags = [];
    $tags[] = 'dda_archive_list';
    $tags[] = "dda_archive:{$this->id()}";
    $api_tags = Util::getAggregationTagsToClear($this->getArchiveType(), $this->get('aggregate_of')->value, $this->get('aggregate_on')->value ?? '');
    $tags = array_filter(array_merge($tags, $api_tags));

    $this->cacheInvalidator->invalidateTags($tags);
    // Trigger an event that can be used to perform actions after save.
    $event = new ArchivePostSaveEvent($this);
    $this->eventDispatcher->dispatch($event, ArchivePostSaveEvent::EVENT_NAME);
    $year = date('Y', strtotime($this->get('dataset_modified')->value));

    $types = ['aggregate', 'theme', 'keyword'];
    // If this is a aggregate we want to create or update the annual.
    if (in_array($this->getArchiveType(), $types) && !empty($this->get('aggregate_of')->value) && !empty($this->get('aggregate_on')->value)) {
      $this->archiveService->queueAnnualArchives($year, 'aggregate', $this->get('aggregate_of')->value, $this->get('aggregate_on')->value);
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function postDelete(EntityStorageInterface $storage, array $entities) {
    parent::postDelete($storage, $entities);
    foreach ($entities as $dda_archive) {
      /** @var \Drupal\dkan_dataset_archiver\Entity\DdaArchive $dda_archive */
      $aliases = $dda_archive->getExistingAliases();
      foreach ($aliases as $alias) {
        $alias->delete();
      }
    }
    // @todo this should also remove any related archive files from
    // both local and remote storage.
  }

  /**
   * {@inheritdoc}
   */
  public function getName(): string {
    return $this->get('name')->value ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function setName(string $name): DdaArchiveInterface {
    $this->set('name', $name);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime(): int {
    return $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setCreatedTime(int $timestamp): DdaArchiveInterface {
    $this->set('created', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwner(): UserInterface {
    /** @var \Drupal\user\UserInterface $user */
    $user = $this->get('user_id')->entity;
    return $user;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwnerId(): int {
    return (int) $this->get('user_id')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwnerId($uid) {
    $this->set('user_id', $uid);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwner(UserInterface $account): self {
    $this->set('user_id', $account->id());
    return $this;
  }

  /**
   * Remove existing aliases that do not match new & return remaining count.
   *
   * @return int
   *   The number of aliases remaining.  Expecting either 0 or 1.
   */
  protected function removeExistingAlias(): int {
    $existing_aliases = $this->getExistingAliases();
    $new_alias = $this->getAliasPattern();
    $existing_count = count($existing_aliases);
    $deleted_count = 0;
    foreach ($existing_aliases as $existing_alias) {
      // Check to see if this matches the new one to create.
      if ($existing_alias->get('alias')->value !== $new_alias) {
        // It does not match the new one, so get rid of it.
        $existing_alias->delete();
        $deleted_count++;
      }
    }
    $aliases_remaining = $existing_count - $deleted_count;
    return $aliases_remaining;
  }

  /**
   * {@inheritdoc}
   */
  public function getArchiveType(): string {
    $archive_type = $this->get('archive_type')->value ?? '';
    $aggregate_of = $this->get('aggregate_of')->value ?? '';
    if ($archive_type === 'aggregate') {
      $constructed_type = $aggregate_of;
    }
    if ($archive_type === 'annual' && !empty($aggregate_of)) {
      $constructed_type = "annual_{$aggregate_of}";
    }
    return $constructed_type ?? $archive_type;
  }

  /**
   * {@inheritdoc}
   */
  public function isPrivate(): bool {
    $access_level = $this->get('access_level')->value ?? 'public';
    $private = Util::isConsideredPrivate($access_level);

    return $private;
  }

  /**
   * {@inheritdoc}
   */
  public function adjustStorageLocation(string $url): string {
    $is_private = $this->isPrivate();
    return Util::adjustStorageLocation($url, $is_private);
  }

  /**
   * {@inheritdoc}
   */
  public function getResourceFileItems(): array {
    /** @var \Drupal\file\Plugin\Field\FieldType\FileFieldItemList $file_list */
    $file_list = $this->get('resource_files');
    return $file_list->referencedEntities();
  }

  /**
   * Sets the alias for the Archive.
   */
  protected function setAlias(): void {
    $alias = $this->getAliasPattern();
    $path = $this->getUri();
    $new_alias = PathAlias::Create([
      'path' => $path,
      'alias' => $alias,
      'langcode' => !empty($this->language()->getId()) ? $this->language()->getId() : 'en',
    ]);
    // @todo prevent duplicate alias creation.
    $new_alias->save();

  }

  /**
   * {@inheritdoc}
   */
  public function getAliasPattern(): string {
    // Get the type and build the alias based on type.
    $type = $this->getArchiveType();
    $dataset_id = $this->get('dataset_id')->value ?? '';
    $aggregate_of = $this->get('aggregate_of')->value ?? '';
    $aggregated_on = $this->get('aggregate_on')->value ?? '';
    $created = $this->getCreatedTime();
    $created_date = date('Y-m-d', $created);
    $year = date('Y', $created);
    $private_text = $this->isPrivate() ? '/private' : '';

    $aliases = [
      'individual' => "/dataset-archives/{$dataset_id}/{$created_date}{$private_text}",
      'keyword' => "/dataset-archives/{$aggregate_of}/{$aggregated_on}{$created_date}{$private_text}",
      'theme' => "/dataset-archives/{$aggregate_of}/{$aggregated_on}/{$created_date}{$private_text}",
      'annual' => "/dataset-archives/annual/{$year}{$private_text}",
      'annual_keyword' => "/dataset-archives/{$aggregate_of}/{$aggregated_on}/annual/{$year}{$private_text}",
      'annual_theme' => "/dataset-archives/{$aggregate_of}/{$aggregated_on}/annual/{$year}{$private_text}",
      'current' => "/dataset-archives/current/{$aggregate_of}/{$aggregated_on}{$private_text}",
    ];

    return $aliases[$type];
  }

  /**
   * Get any aliases that exist for this archive.
   *
   * @return array
   *   An array of existing Path alias objects for this archive.
   */
  protected function getExistingAliases(): array {
    // Retrieve existing aliases for this CM Document.
    $path = $this->getUri();
    $existing_aliases = $this->pathAliasStorage->loadByProperties([
      'path' => $path,
      'langcode' => 'en',
    ]);

    return $existing_aliases;
  }

  /**
   * {@inheritdoc}
   */
  public function getUri(): string {
    $uri = '/admin/content/archive/' . $this->id();

    return $uri;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Name'))
      ->setDescription(t('The name for the archive. Automatically generated on save.'))
      ->setRevisionable(TRUE)
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
      ])
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'string',
      ])
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', FALSE)
      ->setRevisionable(TRUE)
      ->setTranslatable(FALSE)
      ->setRequired(FALSE);

    $fields['archive_type'] = BaseFieldDefinition::create('list_string')
      ->setLabel(new TranslatableMarkup('Archive type'))
      ->setDescription(new TranslatableMarkup('Choose the type of thing being archived.'))
      ->setTranslatable(FALSE)
      ->setSettings([
        'allowed_values' => [
          'individual' => t('Individual dataset'),
          'aggregate' => t('Aggregate'),
          'annual' => t('Annual'),
          'current' => t('Current Download'),
        ],
      ])
      ->setDisplayOptions('form', [
        'type' => 'options_select',
      ])
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'list_default',
      ])
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', FALSE)
      ->setRequired(TRUE);

    $fields['access_level'] = BaseFieldDefinition::create('list_string')
      ->setLabel(new TranslatableMarkup('Access level'))
      ->setDescription(new TranslatableMarkup("Choose the dataset's accessLevel."))
      ->setTranslatable(FALSE)
      ->setSettings([
        'allowed_values' => [
          'public' => t('public'),
          'restricted public' => t('restricted public'),
          'non-public' => t('non-public'),
        ],
      ])
      ->setDisplayOptions('form', [
        'type' => 'options_select',
      ])
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'list_default',
      ])
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', FALSE)
      ->setRequired(TRUE);

    $fields['dataset_id'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Dataset ID'))
      ->setDescription(t('The ID of the dataset being archived.'))
      ->setRevisionable(TRUE)
      ->setSettings([
        'max_length' => 300,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
      ])
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'string',
      ])
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', FALSE)
      ->setRevisionable(TRUE)
      ->setTranslatable(FALSE)
      ->setRequired(FALSE);

    $fields['dataset_modified'] = BaseFieldDefinition::create('datetime')
      ->setLabel(t('Dataset modified'))
      ->setDescription(t('The date that the dataset in this archive was modified.'))
      ->setRequired(TRUE)
      ->setSettings([
        'datetime_type' => 'date',
        'time_type' => 'none',
      ])
      ->setDisplayOptions('view', [
        'type' => 'date',
      ])
      ->setDisplayOptions('form', [
        'type' => 'date',
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    // This should have been a entity reference to data:theme but does not work
    // due to data not having hash ids as a title and theme being in json.
    $fields['themes'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Themes'))
      ->setDescription(t('Themes associated with this archive.'))
      ->setRevisionable(TRUE)
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
      ])
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'string',
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRevisionable(TRUE)
      ->setTranslatable(FALSE)
      ->setCardinality(BaseFieldDefinition::CARDINALITY_UNLIMITED)
      ->setRequired(FALSE);

    // This should have been a entity reference to data:keyword but doesn't work
    // due to data not having hash ids as a title and theme being in json.
    $fields['keywords'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Keywords'))
      ->setDescription(t('Keywords associated with this archive.'))
      ->setRevisionable(TRUE)
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
      ])
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'string',
      ])
      ->setCardinality(BaseFieldDefinition::CARDINALITY_UNLIMITED)
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRevisionable(TRUE)
      ->setTranslatable(FALSE)
      ->setRequired(FALSE);

    $fields['aggregate_of'] = BaseFieldDefinition::create('list_string')
      ->setLabel(new TranslatableMarkup('Aggregation of'))
      ->setDescription(new TranslatableMarkup("What is the base of this aggregation?"))
      ->setTranslatable(FALSE)
      ->setSettings([
        'allowed_values' => [
          'theme' => t('theme'),
          'keyword' => t('keyword'),
        ],
      ])
      ->setDisplayOptions('form', [
        'type' => 'options_select',
      ])
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'list_default',
      ])
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', FALSE)
      ->setRequired(FALSE);

    $fields['aggregate_on'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Aggregate on'))
      ->setDescription(t('The theme or keyword this is aggregated on.'))
      ->setRevisionable(TRUE)
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
      ])
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'string',
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRevisionable(TRUE)
      ->setTranslatable(FALSE)
      ->setRequired(FALSE);

    $fields['resource_files'] = BaseFieldDefinition::create('file')
      ->setLabel(t('Local archive files'))
      ->setDescription(t('Files from the dataset being archived.'))
      ->setCardinality(BaseFieldDefinition::CARDINALITY_UNLIMITED)
      ->setSettings([
        'file_extensions' => 'csv gz pdf rar tar tsv txt zip 7z',
        'uri_scheme' => 'public',
        'file_directory' => 'dataset-archives',
        'target_type' => 'file',
        'handlers' => ['file'],
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'file_default',
      ])
      ->setDisplayOptions('form', [
        'type' => 'file_generic',
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['remote_url'] = BaseFieldDefinition::create('link')
      ->setLabel(t('Remote archive file URLs'))
      ->setDescription(t('The url of the remote archive zip file.'))
      ->setCardinality(BaseFieldDefinition::CARDINALITY_UNLIMITED)
      ->setDefaultValue('')
      ->setSettings([
        'link_type' => LinkItem::LINK_GENERIC,
        'title' => DRUPAL_DISABLED,
      ])
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'link',
        'settings' => [
          'text' => 'Remote archive',
        ],
      ])
      ->setDisplayOptions('form', [
        'type' => 'link_default',
      ])
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', FALSE)
      ->setRevisionable(TRUE)
      ->setTranslatable(FALSE)
      ->setRequired(FALSE);

    $fields['size'] = BaseFieldDefinition::create('integer')
      ->setSettings([
        'unsigned' => TRUE,
        'size' => 'big',
      ])
      ->setLabel(t('File size'))
      ->setDescription(t('The size of the files (in bytes)'))
      ->setDisplayOptions('form', [
        'type' => 'number',
        'size' => 14,
      ])
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'file_size',
        'settings' => [
          'thousand_separator' => ',',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE)
      ->setRequired(FALSE)
      ->setDefaultValue(0);

    $fields['source_archives'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Source archives'))
      ->setDescription(t('The archives that are present in this aggregated archive.'))
      ->setRevisionable(TRUE)
      ->setSetting('target_type', 'dda_archive')
      ->setSetting('selection_handler', 'default')
      ->setSetting('tags', FALSE)
      ->setTranslatable(FALSE)
      ->setCardinality(BaseFieldDefinition::CARDINALITY_UNLIMITED)
      ->setDisplayOptions('form', [
        'type' => 'entity_autocomplete',
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'placeholder' => t('Type the dataset name'),
          'autocomplete_type' => 'tags',
        ],
      ])
      ->setDisplayOptions('view', [
        'label' => 'inline',
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Published'))
      ->setDescription(t('Archive is published.'))
      ->setDefaultValue(FALSE)
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE)
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'settings' => [
          '#weight' => 60,
        ],
      ]);

    $fields['revision_translation_affected'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Revision translation affected'))
      ->setDescription(t('Indicates if the last edit of a translation belongs to current revision.'))
      ->setReadOnly(TRUE)
      ->setRevisionable(TRUE)
      ->setTranslatable(FALSE)
      ->setInitialValue(TRUE);

    $fields['user_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Created by'))
      ->setDescription(t('The user ID of the creator of this Archive.'))
      ->setRevisionable(TRUE)
      ->setSetting('target_type', 'user')
      ->setSetting('handler', 'default')
      ->setTranslatable(FALSE)
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => t('type archive name'),
        ],
      ])
      ->setDisplayOptions('view', [
        'region' => 'hidden',
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDisplayOptions('view', [
        'label' => 'inline',
      ])
      ->setDescription(t('The date-time that the archive was created.'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDisplayOptions('view', [
        'label' => 'inline',
      ])
      ->setDescription(t('The date-time that the archive was updated.'));

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function getDataset($dataset_id = NULL): object|null {
    $dataset_id = $dataset_id ?? $this->get('dataset_id')->value;
    if (empty($dataset_id)) {
      return NULL;
    }
    if (!isset($this->loadedDatasets[$dataset_id])) {
      // Get the entity from the metastore.
      // Depending on workflow, the archive may be created before the dataset is
      // published, so we load it either way.
      try {
        $dataset = $this->metastoreService->get('dataset', $dataset_id, FALSE);
        $this->loadedDatasets[$dataset_id] = $dataset;
      }
      catch (\Throwable $e) {
        \Drupal::logger('dkan_dataset_archiver')->error('Failed to load dataset with ID @id: @message', [
          '@id' => $dataset_id,
          '@message' => $e->getMessage(),
        ]);
        $dataset = NULL;
      }
    }
    else {
      $dataset = $this->loadedDatasets[$dataset_id];
    }
    return $dataset;
  }

  /**
   * {@inheritdoc}
   */
  public function getDatasetTitle(): string {
    $dataset = $this->getDataset();
    if ($dataset) {
      return $dataset->{'$.title'} ?? $this->values["name"]["x-default"][0]["value"] ?? '';
    }
    return '';
  }

  /**
   * {@inheritdoc}
   */
  public function matchStatusToDataset(): void {
    $dataset_id = $this->get('dataset_id')->value;
    if ($dataset_id && $this->isNew()) {
      // Only on the creation of the archive, should the status match the status
      // of the dataset.
      // @todo this will need to be reworked around moderation states.
      // Depending on workflow, the archive may be created before the dataset is
      // published.
      $dataset_status = $this->metastoreService->isPublished('dataset', $dataset_id);
      $this->set('status', $dataset_status);
    }
  }

  /**
   * Build the archive name dynamically based on archive type.
   *
   * @return string
   *   The name of the archive
   */
  protected function buildArchiveName(): string {
    $date = $this->get('dataset_modified')->value;
    $year = date('Y', strtotime($date));
    $title = $this->getDatasetTitle();
    $aggregate_of = $this->get('aggregate_of')->value ?? '';
    $aggregate_on = $this->get('aggregate_on')->value ?? '';
    $title = !empty($title) ? $title : t('Dataset not found');
    $theme_text = t('Theme');
    $keyword_text = t('Keyword');
    $annual_text = t('Annual');
    $current_text = t('Current Download');
    $private_text = $this->isPrivate() ? t('private') . ' ' : '';

    $names = [
      'individual' => "{$title} ({$date})",
      'keyword' => "{$keyword_text}: {$aggregate_on} {$private_text}({$date})",
      'theme' => "{$theme_text}: {$aggregate_on} {$private_text}({$date})",
      'annual' => "{$annual_text} {$private_text}{$year}",
      'annual_keyword' => "{$annual_text}: {$keyword_text} {$aggregate_on} {$private_text}({$year})",
      'annual_theme' => "{$annual_text}: {$theme_text} {$aggregate_on} {$private_text}({$year})",
      'current' => "{$current_text}: {$private_text} {$aggregate_of} {$aggregate_on}",
    ];

    return $names[$this->getArchiveType()];
  }

  /**
   * Modify the archive based on its type.
   */
  protected function modifyArchiveByType(): void {
    match ($this->getArchiveType()) {
      'individual' => $this->modifyIndividualArchive(),
      'keyword' => $this->modifyAggregatedArchive(),
      'theme' => $this->modifyAggregatedArchive(),
      'annual' => $this->modifyAnnualArchive(),
      'annual_keyword' => $this->modifyAnnualAggregatedArchive(),
      'annual_theme' => $this->modifyAnnualAggregatedArchive(),
      default => NULL,
    };
  }

  /**
   * Modify an individual archive.
   */
  protected function modifyIndividualArchive(): void {
    // Blank out aggregate_on and of field.
    $this->set('aggregate_of', NULL);
    $this->set('aggregate_on', NULL);
  }

  /**
   * Modify an aggregated archive.
   */
  protected function modifyAggregatedArchive(): void {
    $type = $this->getArchiveType();
    // Based on type of archive, clear out either themes or keywords, opposite.
    if (in_array($type, ['keyword'])) {
      $this->set('themes', NULL);
      $this->set('aggregate_of', 'keyword');
    }
    if (in_array($type, ['theme'])) {
      $this->set('keywords', NULL);
      $this->set('aggregate_of', 'theme');
    }
  }

  /**
   * Modify an annual archive.
   */
  protected function modifyAnnualArchive(): void {
    // Based on type of archive, clear out either themes or keywords, opposite.
    if (in_array($this->getArchiveType(), ['annual_keyword'])) {
      $this->set('themes', NULL);
      $this->set('aggregate_of', 'keyword');
    }
    if (in_array($this->getArchiveType(), ['annual_theme'])) {
      $this->set('keywords', NULL);
      $this->set('aggregate_of', 'theme');
    }
    if (in_array($this->getArchiveType(), ['annual'])) {
      $this->set('keywords', NULL);
      $this->set('themes', NULL);
    }
  }

  /**
   * Modify an annual aggregated archive.
   */
  protected function modifyAnnualAggregatedArchive(): void {
    // @todo Implement annual aggregated archive handling.
    $this->set('themes', NULL);
    $this->set('keywords', NULL);
  }

  /**
   * {@inheritdoc}
   */
  public function localFilesChanged(): bool {
    $original = $this->original;
    if (empty($original)) {
      // No original, so this is new.
      return TRUE;
    }
    /** @var \Drupal\dkan_dataset_archiver\Entity\DdaArchive $original */
    $original_files = $original->getResourceFileItems();
    $current_files = $this->getResourceFileItems();
    // @todo create some hashes of filename, sizes and dates.
    // This is forced to always return true for now. MUST FIX!
    $original_hash = 'A';
    $current_hash = 'B';
    return $original_hash !== $current_hash;
  }

}
