<?php

namespace Drupal\dkan_dataset_archiver\Entity;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityChangedInterface;
use Drupal\Core\Entity\EntityPublishedInterface;
use Drupal\Core\Entity\RevisionLogInterface;
use Drupal\user\EntityOwnerInterface;

/**
 * Provides an interface for defining Archive entities.
 *
 * @ingroup cm_document
 */
interface DdaArchiveInterface extends ContentEntityInterface, RevisionLogInterface, EntityChangedInterface, EntityPublishedInterface, EntityOwnerInterface {

  /**
   * Gets the DdaArchive name.
   *
   * @return string
   *   Name of the DdaArchive.
   */
  public function getName(): string;

  /**
   * Sets the DdaArchive name.
   *
   * @param string $name
   *   The DdaArchive name.
   *
   * @return \Drupal\dkan_dataset_archiver\Entity\DdaArchiveInterface
   *   The called DdaArchive entity.
   */
  public function setName(string $name): DdaArchiveInterface;

  /**
   * Returns whether the DdaArchive is treated as private.
   *
   * @return bool
   *   TRUE if the DdaArchive is private, FALSE otherwise.
   */
  public function isPrivate(): bool;

  /**
   * Adjusts a URL to be stored in the correct file system.
   *
   * @param string $url
   *   The original URL.
   *
   * @return string
   *   The adjusted URL with stream wrapper.
   */
  public function adjustStorageLocation(string $url): string;

  /**
   * Get the resource_files.
   *
   * @return array
   *   Array of file entities.
   */
  public function getResourceFileItems(): array;

  /**
   * Whether this session of the archive has sent files to remote storage.
   *
   * @return bool
   *   TRUE if files have been sent to remote storage, FALSE otherwise.
   */
  public function hasRemoteStored(): bool;

  /**
   * Sets whether this session of the archive has sent files to remote storage.
   *
   * @param bool $hasRemoteStored
   *   TRUE if files have been sent to remote storage, FALSE otherwise.
   */
  public function setHasRemoteStored(bool $hasRemoteStored): void;

  /**
   * Gets the DdaArchive creation timestamp.
   *
   * @return int
   *   Creation timestamp of the DdaArchive.
   */
  public function getCreatedTime();

  /**
   * Sets the DdaArchive creation timestamp.
   *
   * @param int $timestamp
   *   The DdaArchive creation timestamp.
   *
   * @return \Drupal\dkan_dataset_archiver\Entity\DdaArchiveInterface
   *   The called DdaArchive entity.
   */
  public function setCreatedTime(int $timestamp): DdaArchiveInterface;

  /**
   * Reads type field and returns type.
   *
   * @return string
   *   The value of the type, bundle or field. Empty string otherwise.
   */
  public function getArchiveType(): string;

  /**
   * Loads, caches, and returns the dataset that is being archived.
   *
   * @param string|null $dataset_id
   *   The ID of the dataset to retrieve. If null, uses the dataset_id field.
   *
   * @return object|null
   *   The dataset being archived.
   */
  public function getDataset($dataset_id = NULL): object|null;

  /**
   * Gets the title of the dataset being archived.
   *
   * @return string
   *   The title of the dataset being archived.
   */
  public function getDatasetTitle(): string;

  /**
   * Force the archive to have the same status as the dataset.
   */
  public function matchStatusToDataset(): void;

  /**
   * Determine if local files have changed since last archive save.
   *
   * @return bool
   *   TRUE if local files have changed, FALSE otherwise.
   */
  public function localFilesChanged(): bool;

}
