<?php

namespace Drupal\dkan_dataset_archiver\Drush\Commands;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Utility\Token;
use Drupal\dkan_dataset_archiver\Service\ArchiveService;
use Drupal\dkan_dataset_archiver\Service\Util;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A Drush command file.
 */
class DkanDatasetArchiverCommands extends DrushCommands {

  /**
   * The archive service.
   *
   * @var \Drupal\dkan_dataset_archiver\Service\ArchiveService
   */
  protected $archiveService;

  /**
   * The module's settings.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $archiverSettings;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $myLogger;

  /**
   * Constructs a DkanDatasetArchiverCommands object.
   */
  public function __construct(
    private Token $token,
    private ArchiveService $dkanDatasetArchiverArchiveService,
    LoggerInterface $logger,
    ConfigFactoryInterface $config_factory,
  ) {
    parent::__construct();
    $this->archiveService = $dkanDatasetArchiverArchiveService;
    $this->myLogger = $logger;
    $this->configFactory = $config_factory;
    $this->archiverSettings = $this->configFactory->get('dkan_dataset_archiver.settings');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('token'),
      $container->get('dkan_dataset_archiver.archive_service'),
      $container->get('logger.channel.dkan_dataset_archiver'),
      $container->get('config.factory'),
    );
  }

  /**
   * Create or update the annual archive un-naturally for the current year.
   */
  #[CLI\Command(name: 'dkan_dataset_archiver:create-annual', aliases: ['dkan_dataset_archiver:refresh-annual'])]
  #[CLI\Usage(name: 'dkan_dataset_archiver:create-annual', description: "Generates or updates this year's annual archives.")]
  #[CLI\Argument(name: 'year', description: 'The year for which to create or update annual archives. Defaults to current year.')]
  public function createAnnual(?int $year = NULL) {
    /** @var \Drush\Style\DrushStyle $io */
    $io = $this->io();
    $year = $year ?? (int) Util::date()->format('Y');
    $years_to_keep = (int) $this->archiverSettings->get('archive_years_retained');
    $earliest_year = (int) Util::date()->format('Y') - $years_to_keep;
    if ($years_to_keep > 0 && $year < $earliest_year) {
      $io->error(dt('Year @year is older than the configured number of years to keep (@years). The earliest year allowed is @earliest_year. Annual archive creation skipped.', [
        '@year' => $year,
        '@years' => $years_to_keep,
        '@earliest_year' => $earliest_year,
      ]));
      return;
    }

    if ($io->confirm(dt('Generate or update annual dataset archives?'))) {
      // User confirmed, proceed with archiving logic.
      if (!$this->archiverSettings->get('create_annual_archives')) {
        $this->myLogger->warning(dt('Annual archive creation is disabled in settings. No annual archives created or updated.'));
        $io->warning(dt('Annual archive creation is disabled in settings. No annual archives created or updated.'));
        return;
      }

      $io->success(dt('Queueing annual archives for @year.', ['@year' => $year]));
      $this->archiveService->queueAnnualArchives($year, 'individual');
      $this->myLogger->info(dt('- Queueing keyword annual archives for @year.', ['@year' => $year]));

      if ($this->archiverSettings->get('archive_by_keyword')) {
        $io->success(dt('- Queueing keyword annual archives for @year.', ['@year' => $year]));
        $this->myLogger->info(dt('- Queueing keyword annual archives for @year.', ['@year' => $year]));
        $this->archiveService->queueAnnualArchives($year, 'aggregate', 'keyword');
      }

      if ($this->archiverSettings->get('archive_by_theme')) {
        $io->success(dt('- Queueing theme annual archives for @year.', ['@year' => $year]));
        $this->myLogger->info(dt('- Queueing theme annual archives for @year.', ['@year' => $year]));
        $this->archiveService->queueAnnualArchives($year, 'aggregate', 'theme');
      }
      $this->myLogger->info(dt('Queueing of @year annuals complete.', ['@year' => $year]));
      $io->success(dt('Queueing of @year annuals complete.', ['@year' => $year]));
      $io->warning(dt('You must now run "drush cron" to process the queue.'));
    }
    else {
      $io->warning(dt('Annual archive creation cancelled.'));
    }
  }

  /**
   * Create or re-fresh the "current" archives for all published datasets.
   */
  #[CLI\Command(name: 'dkan_dataset_archiver:create-current', aliases: ['dkan_dataset_archiver:refresh-current'])]
  #[CLI\Usage(name: 'dkan_dataset_archiver:create-current', description: "Creates or refreshes 'current' archives for all published datasets.")]
  public function createCurrent() {
    /** @var \Drush\Style\DrushStyle $io */
    $io = $this->io();
    $count = $this->archiveService->reQueueCurrentArchiving();
    $this->myLogger->info("DKAN Dataset Archiver queued @count 'current' archives.", ['@count' => $count]);
    $io->success(dt("DKAN Dataset Archiver queued @count 'current' archives.", ['@count' => $count]));
    $io->warning(dt('You must now run "drush cron" to process the queue.'));
  }

}
