<?php

namespace Drupal\Tests\dkan_dataset_archiver\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Tests for the /admin/dkan/archiver side of the dataset archiver module.
 *
 * @group alt_text_validation
 */
class ApiTest extends BrowserTestBase {

  /**
   * Modules to install.
   *
   * @var array
   */
  protected static $modules = [
    'common',
    'dkan_dataset_archiver',
    'field',
    'file',
    'link',
    'metastore_search',
    'metastore',
    'options',
    'user',
    'views',
  ];

  /**
   * The theme to default to during tests.
   *
   * @var string
   */
  protected $defaultTheme = 'stark';

  /**
   * A user permed for administrating alt_text_validation.
   *
   * @var \Drupal\user\UserInterface
   */
  private $permedUser;

  /**
   * A non-permed user.
   *
   * @var \Drupal\user\UserInterface
   */
  private $normalUser;

  /**
   * Perform initial setup tasks that run before every test method.
   */
  public function setUp(): void {
    parent::setUp();
    $this->permedUser = $this->DrupalCreateUser([
      'access dataset archive api',
    ]);
    $this->normalUser = $this->DrupalCreateUser([]);

    // Adjust Enable the various archiving options that have endpoints.
    $this->config('dkan_dataset_archiver.settings')
      ->set('archive_by_keyword', TRUE)
      ->set('archive_by_theme', TRUE)
      ->set('create_current_download', TRUE)
      ->set('create_annual_archives', TRUE)
      ->save();
  }

  /**
   * Tests that the Individual endpoint can be reached with the right perms.
   */
  public function testIndividualEndpointExists(): void {
    $this->drupalLogin($this->permedUser);
    $returned = $this->drupalGet('api/1/archive/individual/');
    $returned = json_decode($returned, TRUE);
    $this->assertSession()->statusCodeEquals(200);
    $this->assertArrayHasKey('data', $returned);
    $this->assertArrayHasKey('meta', $returned);
    // Assert no warnings or errors.
    $this->assertSession()->statusMessageNotExists('warning');
    $this->assertSession()->statusMessageNotExists('error');
  }

  /**
   * Tests that the settings page can't be reached with without perms.
   */
  public function testIndividualEndpointAccessDenied(): void {
    $this->drupalLogin($this->normalUser);
    $this->drupalGet('api/1/archive/individual/');
    $this->assertSession()->statusCodeEquals(403);
  }

  /**
   * Tests that the Aggregate Keyword endpoint can be reached with right perms.
   */
  public function testAggregateKeywordEndpointExists(): void {
    $this->drupalLogin($this->permedUser);
    $returned = $this->drupalGet('api/1/archive/aggregate/keyword/');
    $returned = json_decode($returned, TRUE);
    $this->assertSession()->statusCodeEquals(200);
    $this->assertArrayHasKey('data', $returned);
    $this->assertArrayHasKey('meta', $returned);
    // Assert no warnings or errors.
    $this->assertSession()->statusMessageNotExists('warning');
    $this->assertSession()->statusMessageNotExists('error');
  }

  /**
   * Tests that the Aggregate Keyword endpoint can't be reached without perms.
   */
  public function testAggregateKeywordEndpointAccessDenied(): void {
    $this->drupalLogin($this->normalUser);
    $this->drupalGet('api/1/archive/aggregate/keyword/');
    $this->assertSession()->statusCodeEquals(403);
  }

  /**
   * Tests that the Aggregate Theme endpoint can be reached with right perms.
   */
  public function testAggregateThemeEndpointExists(): void {
    $this->drupalLogin($this->permedUser);
    $returned = $this->drupalGet('api/1/archive/aggregate/theme/');
    $this->assertSession()->statusCodeEquals(200);
    $returned = json_decode($returned, TRUE);
    $this->assertArrayHasKey('data', $returned);
    $this->assertArrayHasKey('meta', $returned);
    // Assert no warnings or errors.
    $this->assertSession()->statusMessageNotExists('warning');
    $this->assertSession()->statusMessageNotExists('error');
  }

  /**
   * Tests that the Aggregate Theme endpoint can't be reached without perms.
   */
  public function testAggregateThemeEndpointAccessDenied(): void {
    $this->drupalLogin($this->normalUser);
    $this->drupalGet('api/1/archive/aggregate/theme/');
    $this->assertSession()->statusCodeEquals(403);
  }

  /**
   * Tests that the Annual endpoint can be reached with the right perms.
   */
  public function testAggregateAnnualEndpointExists(): void {
    $this->drupalLogin($this->permedUser);
    $returned = $this->drupalGet('api/1/archive/aggregate/annual/');
    $this->assertSession()->statusCodeEquals(200);
    $returned = json_decode($returned, TRUE);
    $this->assertArrayHasKey('data', $returned);
    $this->assertArrayHasKey('meta', $returned);
    // Assert no warnings or errors.
    $this->assertSession()->statusMessageNotExists('warning');
    $this->assertSession()->statusMessageNotExists('error');
  }

  /**
   * Tests that the Annual endpoint can't be reached without perms.
   */
  public function testAggregateAnnualEndpointAccessDenied(): void {
    $this->drupalLogin($this->normalUser);
    $this->drupalGet('api/1/archive/aggregate/annual/');
    $this->assertSession()->statusCodeEquals(403);
  }

  /**
   * Tests that the Current All endpoint can be reached with the right perms.
   */
  public function testAggregateCurrentAllEndpointExists(): void {
    $this->drupalLogin($this->permedUser);
    $returned = $this->drupalGet('api/1/archive/aggregate/current/all');
    $this->assertSession()->statusCodeEquals(200);
    $returned = json_decode($returned, TRUE);
    $this->assertArrayHasKey('data', $returned);
    $this->assertArrayHasKey('meta', $returned);
    // Assert no warnings or errors.
    $this->assertSession()->statusMessageNotExists('warning');
    $this->assertSession()->statusMessageNotExists('error');
  }

  /**
   * Tests that the Current All endpoint can't be reached without perms.
   */
  public function testAggregateCurrentAllEndpointAccessDenied(): void {
    $this->drupalLogin($this->normalUser);
    $this->drupalGet('api/1/archive/aggregate/current/all');
    $this->assertSession()->statusCodeEquals(403);
  }

}
