<?php

namespace Drupal\dkan_dataset_archiver\Drush\Commands;

use Drupal\Core\Utility\Token;
use Drupal\dkan_dataset_archiver\Service\ArchiveService;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A Drush command file.
 */
class DkanDatasetArchiverCommands extends DrushCommands {

  /**
   * The archive service.
   *
   * @var \Drupal\dkan_dataset_archiver\Service\ArchiveService
   */
  private $archiveService;

  /**
   * The logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $myLogger;

  /**
   * Constructs a DkanDatasetArchiverCommands object.
   */
  public function __construct(
    private Token $token,
    private ArchiveService $dkanDatasetArchiverArchiveService,
    LoggerInterface $logger,
  ) {
    parent::__construct();
    $this->archiveService = $dkanDatasetArchiverArchiveService;
    $this->myLogger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('token'),
      $container->get('dkan_dataset_archiver.archive_service'),
      $container->get('logger.channel.dkan_dataset_archiver'),
    );
  }

  /**
   * Create the annual archives un-naturally. Use only for testing.
   */
  #[CLI\Command(name: 'dkan_dataset_archiver:create-annual')]
  #[CLI\Usage(name: 'dkan_dataset_archiver:create-annual', description: "ONLY FOR DEVELOPMENT AND TESTING: Generates this year's annual archives.")]
  public function createAnnual() {
    $io = $this->io();

    if ($io->confirm(dt('This should only be run for testing, not on Production. Prematurely generate annual dataset archives?'))) {
      // User confirmed, proceed with deletion logic.
      $io->success(dt('Queueing annual archives.'));
      $this->archiveService->queueAnnualArchives('individual');
      $this->myLogger->info(dt('- Queueing keyword annual archives.'));
      $io->success(dt('- Queueing keyword annual archives.'));
      $this->archiveService->queueAnnualArchives('aggregate', 'keyword');
      $this->myLogger->info(dt('- Queueing theme annual archives.'));
      $io->success(dt('- Queueing theme annual archives.'));
      $this->archiveService->queueAnnualArchives('aggregate', 'theme');
      $this->myLogger->info(dt('Queueing of annuals complete.'));
      $io->success(dt('Queueing of annuals complete.'));
      $io->warning(dt('You must now run "drush cron" to process the queue.'));
    }
    else {
      $io->warning(dt('Annual archive creation cancelled.'));
    }
  }

  /**
   * Create or re-fresh the "current" archives for all published datasets.
   */
  #[CLI\Command(name: 'dkan_dataset_archiver:create-current', aliases: ['dkan_dataset_archiver:refresh-current'])]
  #[CLI\Usage(name: 'dkan_dataset_archiver:create-current', description: "Creates or refreshes 'current' archives for all published datasets.")]
  public function createCurrent() {
    $io = $this->io();
    $count = $this->archiveService->reQueueCurrentArchiving();
    $this->myLogger->info("DKAN Dataset Archiver queued @count 'current' archives.", ['@count' => $count]);
    $io->success(dt("DKAN Dataset Archiver queued @count 'current' archives.", ['@count' => $count]));
    $io->warning(dt('You must now run "drush cron" to process the queue.'));
  }

}
