<?php

namespace Drupal\dkan_dataset_archiver\Hook;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\State\StateInterface;
use Drupal\dkan_dataset_archiver\Service\ArchiveService;
use Drupal\dkan_dataset_archiver\Service\Util;
use League\CommonMark\CommonMarkConverter;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class CoreHooks.
 *
 * Contains hooks that are specific to Drupal core operations.
 */
class CoreHooks implements ContainerInjectionInterface {

  /**
   * The archive service.
   *
   * @var \Drupal\dkan_dataset_archiver\Service\ArchiveService
   */
  protected $archiveService;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The dkan_dataset_archiver logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * Constructs a CoreHooks object.
   *
   * @param \Drupal\dkan_dataset_archiver\Service\ArchiveService $archive_service
   *   The dkan dataset archive service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger channel.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   */
  public function __construct(
    ArchiveService $archive_service,
    ConfigFactoryInterface $config_factory,
    LoggerInterface $logger,
    StateInterface $state,
  ) {
    $this->archiveService = $archive_service;
    $this->configFactory = $config_factory;
    $this->logger = $logger;
    $this->state = $state;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('dkan_dataset_archiver.archive_service'),
      $container->get('config.factory'),
      $container->get('logger.channel.dkan_dataset_archiver'),
      $container->get('state'),
    );
  }

  /**
   * Implements hook_help().
   */
  #[Hook('help')]
  public static function help(string $route_name, RouteMatchInterface $route_match): string | NULL {
    // Check if the route name matches the help page.
    switch ($route_name) {
      case 'help.page.dkan_dataset_archiver':
        // Make a converter instance.
        $converter = new CommonMarkConverter([
          'html_input' => 'strip',
          'allow_unsafe_links' => FALSE,
        ]);
        $text = file_get_contents(__DIR__ . '/../../README.md');
        return $converter->convert($text);
    }
    return NULL;
  }

  /**
   * Implements hook_cron().
   *
   * Queues any aggregations needed due to individual archive creation.
   */
  #[Hook('cron')]
  public function cronAggregate(): void {
    // Check to see if any aggregation tasks need to be run.
    $config = $this->configFactory->get('dkan_dataset_archiver.settings');
    $can_aggregate = (!empty($config->get('archive'))) && ($config->get('archive_by_keyword') || $config->get('archive_by_theme'));
    if ($can_aggregate) {
      // Check to see if the delay has been met.
      $aggregations = $this->state->get('dkan_dataset_archiver.aggregations');
      $last_saved = $aggregations['time'] ?? NULL;
      if (!empty($last_saved)) {
        $delay = $config->get('aggregation_delay') ?? 10;
        $next_run = ($last_saved + ($delay * 60));
        if (time() >= $next_run) {
          // Time to run the aggregation.
          // Remove the time key so we just have terms left.
          unset($aggregations['time']);
          // Loop through keywords and themes if they exist.
          foreach ($aggregations as $type => $data) {
            // Do some juggling to align the data with how it is stored.
            if ($type === 'keyword' || $type === 'theme') {
              $aggregate_of = $type;
              $type = 'aggregate';
            }
            else {
              $aggregate_of = NULL;
            }

            foreach ($data as $privacy => $data) {
              $private = ($privacy === 'private');
              foreach ($data as $term => $archive_ids) {
                // Make sure we don't have duplicates.
                $archive_ids = array_unique($archive_ids);
                $this->archiveService->addToAggregationQueue($type, $term, $archive_ids, $private, $aggregate_of);
                if ($config->get('create_current_download')) {
                  // Lookup all published archives of this type and get ids.
                  $current_ids = $this->archiveService->getDatasetsIdsByThemeOrKeyword($aggregate_of, $term);
                  if (!empty($current_ids['public'])) {
                    $this->archiveService->addToAggregationQueue('current', $term, $current_ids['public'], FALSE, $aggregate_of);
                  }
                  if (!empty($current_ids['private'])) {
                    $this->archiveService->addToAggregationQueue('current', $term, $current_ids['private'], TRUE, $aggregate_of);
                  }
                }
              }
            }
          }
          // The list has been queued, so reset the state to empty.
          $this->state->delete('dkan_dataset_archiver.aggregations');
        }
      }
    }
  }

  /**
   * Implements hook_cron().
   *
   * Queues any annual aggregations needed due it being the first of the year.
   */
  #[Hook('cron')]
  public function cronAnnualAggregate(): void {
    // Check to see if any aggregation tasks need to be run.
    $config = $this->configFactory->get('dkan_dataset_archiver.settings');
    $can_aggregate = (!empty($config->get('archive'))) && ($config->get('create_annual_archives'));
    if ($can_aggregate) {
      // Check to see if the delay has been met.
      $last_annual = $this->state->get('dkan_dataset_archiver.last_annual_archive_created');
      $datetime_object = Util::date();
      $year = $datetime_object->format('Y');
      $is_december_last = ($datetime_object->format('m-d') === '12-31');
      $is_after_11pm = ($datetime_object->format('Hi') >= '2300');
      if ($is_december_last && $is_after_11pm && (empty($last_annual) || $last_annual < $year)) {
        $this->archiveService->queueAnnualArchives('individual');
        $this->archiveService->queueAnnualArchives('aggregate', 'keyword');
        $this->archiveService->queueAnnualArchives('aggregate', 'theme');
        $this->logger->notice('Annual archives created.');
        $this->state->set('dkan_dataset_archiver.last_annual_archive_created', $year);
      }
    }
  }

  /**
   * Add an item to the aggregation queue.
   *
   * @param string $type
   *   The type of aggregation, either 'keyword' or 'theme'.
   * @param string $term
   *   The keyword or theme name for the aggregation.
   * @param array $archive_ids
   *   An array of archive node IDs to include in the aggregation.
   */
  protected function addToAggregationQueue(string $type, string $term, array $archive_ids): void {
    // Add an item to the aggregation queue.
    $this->logger->notice('addToAggregationQueue would create queue for  @type: @term with archives @ids', [
      '@type' => $type,
      '@term' => $term,
      '@ids' => implode(', ', $archive_ids),
    ]);
  }

}
