<?php

namespace Drupal\dkan_dataset_archiver\Controller;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\Query\QueryInterface;
use Drupal\dkan_dataset_archiver\CachedJsonResponseTrait;
use Drupal\dkan_dataset_archiver\Entity\DdaArchiveInterface;
use Drupal\dkan_dataset_archiver\Service\Util;
use League\Flysystem\Adapter\Local;
use League\Flysystem\Filesystem;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;

/**
 * Archive API controller.
 */
class ArchiveApiController implements ContainerInjectionInterface {

  use CachedJsonResponseTrait;

  /**
   * The archiver settings.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $archiverSettings;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The filesystem object.
   *
   * @var \League\Flysystem\Filesystem
   */
  private $filesystem;

  /**
   * The stream wrapper manager.
   *
   * @var \Drupal\Core\StreamWrapper\StreamWrapperManager
   */
  private $streamWrapperManager;

  /**
   * Cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cacheBackend;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Cache\CacheBackendInterface $cacheBackend
   *   Cache backend.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(CacheBackendInterface $cacheBackend, ConfigFactoryInterface $config_factory, EntityTypeManagerInterface $entity_type_manager) {
    $this->archiverSettings = $config_factory->get('dkan_dataset_archiver.settings');
    $this->cacheBackend = $cacheBackend;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('cache.default'),
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * Get structured content of a directory.
   *
   * @param string $directory
   *   The directory to get content from.
   *
   * @return array
   *   The structured content.
   */
  private function getStructuredContentOfDirectory(string $directory): array {
    $content = $this->getFilesystem()->listContents($directory, TRUE);
    return $this->structuredContent($content);
  }

  /**
   * Structure content.
   *
   * @param array $content
   *   Array of files and folders.
   *
   * @return array
   *   Content structured by topic, then year.
   */
  private function structuredContent(array $content): array {
    $flatStructure = array_values(array_filter(array_map(function ($item) {
      return $this->setInfo($item);
    }, array_filter($content, function ($item) {
      return $item['type'] == 'file';
    })), function ($item) {
      $yearSet = isset($item['year']);
      $yearAllowed = $yearSet && $this->yearAllowed($item['year']);
      $topicSet = isset($item['topic']);
      return $yearSet && $yearAllowed && $topicSet;
    }));

    $structure = [];
    foreach ($flatStructure as $file) {
      $topic = $file['topic'];
      unset($file['topic']);
      $year = $file['year'];
      unset($file['year']);

      if (!isset($structure[$topic])) {
        $structure[$topic] = [];
      }

      if (!isset($structure[$topic][$year])) {
        $structure[$topic][$year] = [];
      }

      $structure[$topic][$year][] = $file;
    }

    return $structure;
  }

  /**
   * Restrict the number of years of archive returned.
   */
  private function yearAllowed(string $year): bool {
    $previous_year = (int) Util::date('-1 year')->format('Y');
    $years_to_keep = $this->archiverSettings->get('archive_years_retained');
    $oldest_year = $previous_year - $years_to_keep;
    return ((int) $year) > $oldest_year;
  }

  /**
   * Set item info.
   *
   * @param array $item
   *   Info on one file or folder.
   *
   * @return array
   *   Modified info.
   */
  private function setInfo(array $item): array {
    $info = [];
    $filePath = "public://{$item['path']}";
    $info['url'] = $this->fileCreateUrl($filePath);
    $info['size'] = $item['size'];
    [$month, $year] = $this->getMonthAndYearFromFilename($item['filename']);
    $info['type'] = isset($month) ? 'Monthly' : 'Annual';

    if (isset($month)) {
      $info['month'] = $month;
      $info['day'] = date('d', $item['timestamp']);
    }

    $info['year'] = $year;

    $info['topic'] = $this->getTopicFromDirectoryName($item['dirname']);

    return $info;
  }

  /**
   * Set the filesystem object.
   *
   * @param \League\Flysystem\Filesystem $filesystem
   *   The filesystem object.
   */
  public function setFileSystem(Filesystem $filesystem): void {
    $this->filesystem = $filesystem;
  }

  /**
   * Extract the month MM (if any) and year YYYY from end of filename.
   *
   * @param string $filename
   *   The filename.
   *
   * @return array
   *   Array with month (or NULL) and year (or NULL).
   */
  private function getMonthAndYearFromFilename(string $filename): array {
    if (preg_match('/_(\d\d)_(\d{4})$/', $filename, $matches)) {
      return [$matches[1], $matches[2]];
    }
    if (preg_match('/_(\d{4})$/', $filename, $matches)) {
      return [NULL, $matches[1]];
    }
    return [NULL, NULL];
  }

  /**
   * Extract the topic from the directory name.
   *
   * @param string $directoryName
   *   The directory name.
   *
   * @return string|null
   *   The topic, or NULL if it cannot be determined.
   */
  private function getTopicFromDirectoryName(string $directoryName): ?string {
    $pieces = explode('/', $directoryName);
    return $pieces[1] ?? NULL;
  }

  /**
   * Create a URL for a file URI.
   *
   * @param string $uri
   *   The file URI.
   *
   * @return string
   *   The file URL on the public side of the site.
   */
  private function fileCreateUrl($uri): string {
    if ($wrapper = $this->streamWrapperManager->getViaUri($uri)) {
      $url = $wrapper->getExternalUrl();

      // @todo We need a solution to do this for all environments to convert
      // private edit domain urls to public.
      // Used to be dkan_dataset_archiver_force_url_to_public().
      // @todo Or maybe we don't need this anymore.  Contemplate removing.
      return $url;
    }
    return $uri;
  }

  /**
   * Get (or create) the filesystem object.
   *
   * @return \League\Flysystem\Filesystem
   *   The filesystem object.
   */
  private function getFilesystem(): Filesystem {
    if (!empty($this->filesystem)) {
      $this->filesystem = new Filesystem(new Local(Util::getDrupalPublicFilesDirectory()));
    }
    return $this->filesystem;
  }

  /**
   * Aggregate archives. For endpoint /api/1/archive/aggregate/.
   *
   * @param string $aggregate_of
   *   The type of aggregation, either 'theme' or 'keyword'.
   * @param string $filter
   *   The theme, keyword or other filter aggregate_on, or 'all' or 'none'.
   * @param string $link_type
   *   Set 'absolute' or 'relative' links, any other treated as 'absolute'.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   JsonResponse containing info about the archives.
   */
  public function aggregateTopicArchives(string $aggregate_of, string $filter = 'all', string $link_type = 'absolute'): Response {
    if (!$this->canShowArchivesBasedOnSettings('aggregate', $aggregate_of)) {
      // The aggregation type and settings are not compatible.
      $msg = t('This type of aggregation is not allowed by settings.');
      return new Response($msg, 404);
    }
    $type = 'aggregate';
    $absolute = ($link_type === 'relative') ? FALSE : TRUE;
    // Check for that theme's archives in existing cache.
    $cacheTag = Util::getAggregationTag($type, $aggregate_of, $filter, $absolute);
    if (!empty($cacheTag) && $cachedThemeArchives = $this->getCacheTag($cacheTag)) {
      return $this->getCachedResponse($cachedThemeArchives);
    }
    $ids = $this->aggregationQuery($type, $aggregate_of, $filter);

    $archives = $this->entityTypeManager->getStorage('dda_archive')->loadMultiple($ids);
    $structure = [];
    /** @var \Drupal\dkan_dataset_archiver\Entity\DdaArchiveInterface $archive */
    foreach ($archives as $archive) {
      $item = $this->assemblePayloadItem($archive, $aggregate_of, $type, $absolute);
      $structure[] = $item;
    }

    $payload = [
      'data' => $structure,
      'meta' => $this->getJsonMetaData(count($structure)),
    ];
    // Just in case we end up with no tag, make sure we end up with one.
    $cacheTag = (!empty($cacheTag)) ? $cacheTag : 'archive:never-to-be-read';
    // Aggregate archives don't change often so cache for a day (86400s).
    return $this->setCacheGetResponse($cacheTag, $payload, []);
  }

  /**
   * Aggregate archives. For endpoint /api/1/archive/aggregate/current/.
   *
   * @param string $type
   *   The type of aggregation, either 'current' or 'annual'.
   * @param string $aggregate_of
   *   The type ti filter on, either 'theme' or 'keyword'.
   * @param string $filter
   *   The theme, keyword or other filter aggregate_on, or 'all'.
   * @param string $link_type
   *   Set 'absolute' or 'relative' links, any other treated as 'absolute'.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   JsonResponse containing info about the archives.
   */
  public function aggregateArchives(string $type, string $aggregate_of = 'all', string $filter = 'all', string $link_type = 'absolute'): Response {
    if (!$this->canShowArchivesBasedOnSettings($type, $aggregate_of)) {
      // The aggregation type and settings are not compatible.
      $msg = t('This type of aggregation is not allowed by settings.');
      return new Response($msg, 404);
    }

    $absolute = ($link_type === 'relative') ? FALSE : TRUE;
    // Check for that theme's archives in existing cache.
    $cacheTag = Util::getAggregationTag($type, $aggregate_of, $filter, $absolute);
    if (!empty($cacheTag) && $cachedThemeArchives = $this->getCacheTag($cacheTag)) {
      return $this->getCachedResponse($cachedThemeArchives);
    }
    $ids = $this->aggregationQuery($type, $aggregate_of, $filter);

    $archives = $this->entityTypeManager->getStorage('dda_archive')->loadMultiple($ids);
    $structure = [];
    /** @var \Drupal\dkan_dataset_archiver\Entity\DdaArchiveInterface $archive */
    foreach ($archives as $archive) {
      $item = $this->assemblePayloadItem($archive, $aggregate_of, $type, $absolute);
      $structure[] = $item;
    }

    $payload = [
      'data' => $structure,
      'meta' => $this->getJsonMetaData(count($structure)),
    ];
    // Just in case we end up with no tag, make sure we end up with one.
    $cacheTag = (!empty($cacheTag)) ? $cacheTag : 'archive:never-to-be-read';
    // Aggregate archives don't change often so cache for a day (86400s).
    return $this->setCacheGetResponse($cacheTag, $payload, []);
  }

  /**
   * Get JSON meta data for archive responses.
   *
   * @param int $total_items
   *   Total items in the response.
   * @param int|null $current_page
   *   Current page number.
   *
   * @return array
   *   Meta data array.
   */
  protected function getJsonMetaData(int $total_items, ?int $current_page = 1) {
    $page_size = 500;
    return [
      'total_items' => $total_items,
      // @todo connect this to pagination.
      'current_page' => $current_page,
      'page_size' => $page_size,
      'total_pages' => (empty($total_items)) ? 1 : ceil($total_items / $page_size),
    ];
  }

  /**
   * Individual archives. For endpoint /api/1/archive/individual/.
   *
   * @param string $link_type
   *   Set 'absolute' or 'relative' links, any other treated as 'absolute'.
   * @param string $filter_by
   *   Filter by either "keyword", "theme", or "none".
   * @param string $filter
   *   The actual filter value to look for in the filter_type.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   JsonResponse containing info about the archives.
   */
  public function individualArchives(string $link_type = 'absolute', string $filter_by = 'none', string $filter = 'all'): Response {
    if (!$this->canShowArchivesBasedOnSettings('individual')) {
      // The Archiving is not allowed.
      $msg = t('Archives are not allowed by settings.');
      return new Response($msg, 404);
    }
    $absolute = ($link_type === 'relative') ? FALSE : TRUE;
    // Check for archives in existing cache.
    $cacheTag = Util::getAggregationTag('individual', NULL, "{$filter_by}:{$filter}", $absolute);
    if (!empty($cacheTag) && $cachedThemeArchives = $this->getCacheTag($cacheTag)) {
      return $this->getCachedResponse($cachedThemeArchives);
    }
    $ids = $this->individualQuery($filter_by, $filter);
    $archives = $this->entityTypeManager->getStorage('dda_archive')->loadMultiple($ids);
    $structure = [];
    /** @var \Drupal\dkan_dataset_archiver\Entity\DdaArchiveInterface $archive */
    foreach ($archives as $archive) {
      $item = $this->assemblePayloadItem($archive, NULL, 'individual', $absolute);
      $structure[] = $item;
    }

    $payload = [
      'data' => $structure,
      'meta' => $this->getJsonMetaData(count($structure)),
    ];
    // Just in case we end up with no tag, make sure we end up with one.
    $cacheTag = (!empty($cacheTag)) ? $cacheTag : 'archive:never-to-be-read';
    // Aggregate archives don't change often so cache for a day (86400s).
    return $this->setCacheGetResponse($cacheTag, $payload, []);
  }

  /**
   * Get archive IDs for an aggregation query.
   *
   * @param string $type
   *   The type of aggregation.
   * @param string $aggregate_of
   *   The aggregation_of 'keyword' or 'theme'.
   * @param string $filter
   *   The theme, keyword or other filter aggregate_on, or 'all' or 'none'.
   *
   * @return array
   *   Array of dda_archive entity IDs.
   */
  protected function aggregationQuery(string $type, ?string $aggregate_of, ?string $filter) : array {
    return ($type === 'annual') ? $this->annualAggregationQuery($aggregate_of, $filter) : $this->standardAggregationQuery($type, $aggregate_of, $filter);
  }

  /**
   * Get archive IDs for annual aggregations.
   *
   * @param string|null $aggregate_of
   *   The aggregation_of 'keyword' or 'theme'.
   * @param string $filter
   *   The theme, keyword or other filter aggregate_on, or 'all' or 'none'.
   *
   * @return array
   *   Array of dda_archive entity IDs.
   */
  protected function annualAggregationQuery(?string $aggregate_of, string $filter): array {
    // Truth table.
    // Filter   | Includes annual | Includes annual_subgroups
    // hospital |  no             | yes
    // all      | yes             | yes
    // none     | yes             | no.
    $query = $this->getBaseDdaArchiveQuery();
    $query->condition('archive_type', 'annual', '=');
    $query->sort('dataset_modified', 'DESC');

    if ($filter === 'none' || $aggregate_of === 'none') {
      // None should not include any subgroups so it is pure 'annual'.
      $query->condition('aggregate_of', NULL, 'IS NULL');
    }
    if (in_array($aggregate_of, ['theme', 'keyword'])) {
      // If filtering by theme or keyword, we should include pure annuals.
      $query->condition('aggregate_of', NULL, 'IS NOT NULL');
    }

    if (!in_array($filter, ['all', 'none'])) {
      // It is not a special case, so filter by aggregate_of and aggregate_on.
      $query->condition('aggregate_of', $aggregate_of, '=');
      $query->condition('aggregate_on', $filter);
      // Sub-sort by aggregation.
      $query->sort('aggregate_on', 'ASC');
    }

    $ids = $query->execute();
    return $ids;
  }

  /**
   * Get archive IDs for standard aggregations.
   *
   * @param string $type
   *   The type of aggregation (aggregate, annual, current).
   * @param string|null $aggregate_of
   *   The aggregation_of 'keyword' or 'theme'.
   * @param string $filter
   *   The theme, keyword or other filter aggregate_on, or 'all'.
   *
   * @return array
   *   Array of archive node IDs.
   */
  protected function standardAggregationQuery(string $type, ?string $aggregate_of = NULL, string $filter = 'all') : array {
    $query = $this->getBaseDdaArchiveQuery();
    $archive_types_include = [$type];

    if ($filter !== 'none') {
      // If type keyword or theme we should include the annuals for that type.
      ($type === 'aggregate') ? $archive_types_include[] = 'annual' : NULL;
    }
    $query->condition('archive_type', $archive_types_include, 'IN');
    if ($aggregate_of && $aggregate_of !== 'all') {
      $query->condition('aggregate_of', $aggregate_of);
    }

    if ($filter !== 'all' && $filter !== 'none') {
      // We are not in a special case, so sub-filter.
      $query->condition('aggregate_on', $filter);
    }
    $query->sort('aggregate_on', 'ASC')
      ->sort('dataset_modified', 'DESC');
    $ids = $query->execute();
    return $ids;
  }

  /**
   * Get archive IDs for standard aggregations.
   *
   * @param string $filter_by
   *   The field to filter by ('theme', 'keyword', 'none').
   * @param string $filter
   *   The theme, keyword or other filter aggregate_on, or 'all'.
   *
   * @return array
   *   Array of archive node IDs.
   */
  protected function individualQuery(string $filter_by, string $filter) : array {
    $query = $this->getBaseDdaArchiveQuery();
    $query->condition('archive_type', 'individual');

    if ($filter_by !== 'none' && $filter !== 'all') {
      // We need to filter by something.
      $query->condition("{$filter_by}s", $filter, '=');
    }

    $query->sort('dataset_modified', 'DESC')
      ->sort('name', 'ASC');
    $ids = $query->execute();
    return $ids;
  }

  /**
   * Check if aggregation archiving is enabled for the type of aggregation.
   *
   * @param string $type
   *   Type of aggregation, 'annual', 'theme', keyword, aggregate, current.
   * @param string|null $aggregate_of
   *   The aggregation_of 'keyword' or 'theme'.
   *
   * @return bool
   *   TRUE if theme archiving is disabled.
   */
  protected function canShowArchivesBasedOnSettings(string $type, ?string $aggregate_of = NULL): bool {
    // First case to be TRUE calls it off.
    switch (TRUE) {
      case $this->archiverSettings->get('archive') !== '1':
      case $type === 'keyword' && $this->archiverSettings->get('archive_by_keyword') !== TRUE:
      case $aggregate_of === 'keyword' && $this->archiverSettings->get('archive_by_keyword') !== TRUE:
      case $type === 'theme' && $this->archiverSettings->get('archive_by_theme') !== TRUE:
      case $aggregate_of === 'theme' && $this->archiverSettings->get('archive_by_theme') !== TRUE:
      case $type === 'annual' && $this->archiverSettings->get('create_annual_archives') !== TRUE:
      case $type === 'current' && $this->archiverSettings->get('create_current_download') !== TRUE:
        return FALSE;

      default:
        // It made it this far, so it's OK.
        return TRUE;
    }
  }

  /**
   * Get a base DdaArchive query with common conditions.
   *
   * @return \Drupal\Core\Entity\Query\QueryInterface
   *   The base query.
   */
  protected function getBaseDdaArchiveQuery(): QueryInterface {
    /** @var \Drupal\Core\Entity\Query\QueryInterface $query */
    $query = $this->entityTypeManager->getStorage('dda_archive')->getQuery();
    $query->condition('status', 1);
    $query->accessCheck(TRUE);
    return $query;
  }

  /**
   * Assemble one item for the payload.
   *
   * @param \Drupal\dkan_dataset_archiver\Entity\DdaArchiveInterface $archive
   *   The archive entity.
   * @param string|null $aggregate_of
   *   The aggregation_of 'keyword' or 'theme'.
   * @param string $type
   *   The type of aggregation.
   * @param bool $absolute
   *   Whether to make the URL absolute.
   *
   * @return array
   *   The payload item.
   */
  protected function assemblePayloadItem(DdaArchiveInterface $archive, ?string $aggregate_of, string $type, bool $absolute): array {
    $item = [];
    $aggregate_of = $aggregate_of ?? $archive->get('aggregate_of')->value ?? '';
    $aggregate_on = $archive->get('aggregate_on')->value ?? '';
    // Local archive beats remote url if both exist.
    $local_files = $archive->getResourceFileItems();
    if (!empty($local_files)) {
      $local_url = $local_files[0]?->createFileUrl(!$absolute);
    }

    $remote_url_field_item = $archive->get('remote_url')->first();
    if ($remote_url_field_item) {
      /** @var \Drupal\link\LinkItemInterface $remote_url */
      // @phpstan-ignore-next-line
      $remote_url = $remote_url_field_item->uri;
    }

    $item['name'] = $archive->getName();
    $item['id'] = $archive->id();
    $item['type'] = $archive->getArchiveType();

    if ($aggregate_of) {
      $item[$aggregate_of] = $aggregate_on;
    }

    switch ($type) {
      case 'annual':
        $item['url'] = empty($local_url) ? $remote_url ?? '' : $local_url;

        break;

      case 'individual':
        $item['keywords'] = Util::getListOfMultiValues($archive->get('keywords'));
        $item['themes'] = Util::getListOfMultiValues($archive->get('themes'));
        $resources = Util::getListOfMultiValueUris($archive->get('resource_files'), $absolute);
        $item['resource_files'] = $resources;
        unset($item[$aggregate_of]);

        break;

      case 'aggregate':
      case 'current':
        $item['url'] = empty($local_url) ? $remote_url ?? '' : $local_url;
        break;
    }
    $item['size'] = $archive->get('size')->value;
    $item['date'] = $archive->get('dataset_modified')->value;
    $item['access_level'] = $archive->get('access_level')->value;

    return $item;
  }

}
