# Document Library

A comprehensive document management system with modern UI, hierarchical folders, version control, search, and download tracking.

## Features

### Core Features
- **Document Upload & Management** - Upload and organize documents with ease
- **Hierarchical Folder Structure** - Organize documents in nested folders with slug-based URLs
- **Version Control** - Track document versions with changelog and file history
- **Search & Filter** - Advanced search by title, description, and tags with sorting options
- **Download Tracking** - Monitor document downloads with detailed analytics (user, IP, timestamp)
- **User Permissions** - Granular permission system with 12 distinct permissions
- **Favorites** - Users can mark documents as favorites for quick access
- **Admin Interface** - Dedicated admin page at `/admin/content/documents` with filtering and bulk actions

### Modern UI Features
- **Responsive Design** - Works perfectly on desktop, tablet, and mobile
- **Clean Interface** - Modern, card-based layout with intuitive navigation
- **Drag & Drop Upload** - Enhanced file upload experience
- **Keyboard Shortcuts** - Quick access with Ctrl+K (search) and Ctrl+U (upload)
- **Smooth Animations** - Polished user experience with subtle animations
- **File Type Icons** - Visual indicators for different file types (PDF, Word, Excel, etc.)
- **Real-time Search** - Client-side filtering with debounced input

### Enhanced Features
- **Statistics Dashboard** - Overview of total documents, published/unpublished counts, and downloads
- **Version History** - Complete version tracking with notes and file preservation
- **Bulk Operations** - Select multiple documents for batch publish/unpublish/delete
- **Activity Logs** - Track who downloaded what and when with IP addresses
- **Auto-hide Messages** - Clean UI with auto-dismissing status messages
- **Empty States** - Helpful guidance when no content exists
- **Breadcrumb Navigation** - Easy navigation through folder hierarchy
- **Folder Management** - Create, edit, delete folders with move/delete content options

### Security Features
- **SQL Injection Protection** - All queries use parameterized statements and proper escaping
- **XSS Prevention** - Template auto-escaping and safe DOM manipulation
- **CSRF Protection** - Token validation on sensitive operations
- **Access Control** - Owner-based permissions with "manage all" override
- **File Upload Validation** - Extension and size restrictions
- **Input Sanitization** - All user input properly validated and sanitized

## Installation

1. **Download and Extract**
   ```bash
   cd /path/to/drupal/modules/custom
   # Extract the dl folder here
   ```

2. **Enable the Module**
   ```bash
   drush en dl -y
   drush cr
   ```
   Or via UI: Navigate to Extend > Check "Document Library" > Install

3. **Configure Permissions**
   - Go to: People > Permissions
   - Set permissions for different roles (see Permissions section below)

4. **Configure Settings**
   - Go to: Administration > Document Library > Settings
   - Or: Configuration > Content authoring > Document Library Settings
   - Adjust file size limits, allowed extensions, and features

## File Structure

```
dl/
├── css/
│   └── document-library.css       # Modern UI styles with CSS variables
├── js/
│   └── document-library.js        # JavaScript behaviors and interactions
├── src/
│   ├── Controller/
│   │   └── DocumentLibraryController.php  # Main controller with all routes
│   ├── Form/
│   │   ├── DocumentUploadForm.php
│   │   ├── DocumentEditForm.php
│   │   ├── DocumentDeleteForm.php
│   │   ├── FolderCreateForm.php
│   │   ├── FolderEditForm.php
│   │   ├── FolderDeleteForm.php
│   │   ├── SettingsForm.php
│   │   └── DocumentFieldConfigForm.php
│   └── FolderManager.php          # Service for folder operations
├── templates/
│   ├── dl-library-page.html.twig
│   ├── dl-document-view.html.twig
│   ├── dl-folder-page.html.twig
│   ├── dl-search-page.html.twig
│   ├── dl-favorites-page.html.twig
│   └── dl-admin-documents.html.twig
├── dl.info.yml
├── dl.routing.yml
├── dl.permissions.yml
├── dl.libraries.yml
├── dl.links.menu.yml
├── dl.services.yml
├── dl.module
├── dl.install
└── README.md
```

## Usage

### Uploading Documents

1. Navigate to `/documents`
2. Click "Upload Document" button
3. Fill in the form:
   - Document Title (required)
   - Description (optional)
   - Select Folder (dropdown with hierarchical options)
   - Choose File (required, drag-and-drop supported)
   - Version number (default: 1.0)
   - Tags (comma-separated, optional)
   - Published status (checkbox)
4. Click "Upload Document"

### Managing Documents

**Viewing Documents:**
- Browse the main library at `/documents`
- Navigate folders by clicking folder names
- Use search at `/documents/search`
- View favorites at `/documents/favorites`
- Admin view at `/admin/content/documents`

**Editing Documents:**
- Click on a document to view details
- Click "Edit" button
- Update metadata or upload new version
- Add version notes when uploading new file
- Change folder assignment

**Deleting Documents:**
- Click "Delete" button on document page
- Confirm deletion (this removes all versions and related data)
- Requires 'delete documents' permission or ownership

**Version Control:**
- When uploading a new file during edit, a new version is created
- All previous versions are preserved with separate file storage
- Version history visible on document detail page
- Each version tracks: file, version number, user, timestamp, notes

### Folder Management

**Creating Folders:**
1. Click "New Folder" on library or folder page
2. Enter folder name and description
3. Select parent folder (or root)
4. Folder slug auto-generated from name

**Folder Features:**
- Hierarchical nesting (up to 3 levels supported in routing)
- Unique slugs within parent folder
- Slug-based URLs: `/documents/folder-name/subfolder-name`
- Materialized path for efficient queries
- Document count display
- Move documents between folders via edit form

**Deleting Folders:**
- Option to move contents to parent or delete all
- Cascade updates for child folders
- Owner or 'manage all folders' permission required

### Admin Interface

Navigate to: `/admin/content/documents`

**Features:**
- View all documents (including unpublished)
- Statistics dashboard (total, published, unpublished, downloads)
- Filter by status (All/Published/Unpublished)
- Real-time search (title and owner)
- Sortable columns (title, owner, downloads, size, created)
- Bulk selection with actions:
  - Publish selected
  - Unpublish selected
  - Delete selected
- Folder column shows folder name with clickable link
- Action buttons: View, Edit, Download, Delete

### Permissions

**access document library** - View and browse the document library
**upload documents** - Upload new documents
**edit documents** - Edit own documents
**delete documents** - Delete own documents
**download documents** - Download document files
**administer document library** - Configure module settings (restricted)
**manage all documents** - Edit and delete any document (restricted)
**create folders** - Create new folders
**edit folders** - Edit own folders
**delete folders** - Delete own folders
**manage all folders** - Create, edit, and delete any folder (restricted)

## Configuration

Navigate to: Administration > Document Library > Settings

### General Settings
- **Maximum File Size (MB)** - Upload limit (1-1000 MB, default: 50)
- **Allowed File Extensions** - Space-separated list (default includes common office formats)
- **Items Per Page** - Pagination limit (5-100, default: 20)

### Features
- **Enable Version Control** - Track document versions (default: enabled)
- **Track Downloads** - Log all downloads with user and IP (default: enabled)
- **Enable Favorites** - Allow users to favorite documents (default: enabled)
- **Enable Comments** - Future feature (not implemented)

### User Interface
- **Theme** - Light/Dark/Auto (system preference)
- **Show File Thumbnails** - Display file type icons (default: enabled)

### Field Configuration

Navigate to: Administration > Document Library > Manage Fields

Configure which fields to show/hide in the document library view.

## Database Schema

### Tables

**dl_documents** - Main document metadata
- id (primary key), uuid (unique)
- title, description, tags
- file_id (reference to file entity)
- folder_id (reference to dl_folders, 0 for root)
- version, status (published/unpublished)
- uid (owner), created, changed
- downloads (count)
- Indexes: title, folder_id, uid, status, created

**dl_versions** - Version history
- version_id (primary key)
- document_id, file_id
- version, uid, created, notes
- Indexes: document_id, created

**dl_downloads** - Download tracking
- download_id (primary key)
- document_id, uid, timestamp, ip_address
- Indexes: document_id, uid, timestamp

**dl_favorites** - User favorites
- Primary key: (uid, document_id)
- timestamp
- Indexes: uid, document_id

**dl_folders** - Hierarchical folder structure
- folder_id (primary key)
- parent_id (0 for root level)
- name, slug, description
- path (materialized path: /1/5/12)
- depth, weight, status
- uid, created, changed
- Unique key: (parent_id, slug)
- Indexes: parent_id, path, slug, depth, weight, status, uid

## Keyboard Shortcuts

- `Ctrl+K` or `Cmd+K` - Focus search bar
- `Ctrl+U` or `Cmd+U` - Go to upload page

## API / Programmatic Usage

### Get Document
```php
$document = \Drupal::database()
  ->query('SELECT * FROM {dl_documents} WHERE id = :id', [':id' => $doc_id])
  ->fetchObject();
```

### Create Document Programmatically
```php
$file = File::load($file_id);
$file->setPermanent();
$file->save();

$document_id = \Drupal::database()
  ->insert('dl_documents')
  ->fields([
    'uuid' => \Drupal::service('uuid')->generate(),
    'title' => 'Document Title',
    'description' => 'Description',
    'file_id' => $file_id,
    'folder_id' => $folder_id, // 0 for root
    'version' => '1.0',
    'status' => 1,
    'uid' => $user_id,
    'created' => time(),
    'changed' => time(),
    'tags' => 'tag1, tag2',
  ])
  ->execute();
```

### Using FolderManager Service
```php
$folder_manager = \Drupal::service('dl.folder_manager');

// Create folder
$folder_id = $folder_manager->createFolder('Folder Name', $parent_id, 'Description');

// Get folder
$folder = $folder_manager->getFolder($folder_id);

// Get folder tree
$tree = $folder_manager->buildFolderTree($parent_id, $published_only);

// Get folder options for dropdown
$options = $folder_manager->getFolderOptions($exclude_id, $published_only);

// Delete folder
$folder_manager->deleteFolder($folder_id, $move_contents_to_parent);
```

## Theming

Override templates by copying them to your theme:
```
your_theme/templates/dl/
├── dl-library-page.html.twig
├── dl-document-view.html.twig
├── dl-folder-page.html.twig
├── dl-admin-documents.html.twig
└── ...
```

Override CSS variables:
```css
/* In your theme's CSS */
:root {
  --dl-primary: #10b981;        /* Green primary color */
  --dl-primary-hover: #059669;  /* Darker green on hover */
  --dl-bg: #ffffff;
  --dl-text: #1f2937;
  --dl-text-muted: #6b7280;
  --dl-border: #e5e7eb;
  --dl-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
  --dl-radius: 0.5rem;
}
```

## Supported File Types

Default supported types (configurable):
- PDF (pdf)
- Microsoft Word (doc, docx)
- Microsoft Excel (xls, xlsx)
- Microsoft PowerPoint (ppt, pptx, pps)
- OpenDocument (odt, ods, odp)
- Text files (txt)
- CSV files (csv)
- Images (jpg, jpeg, png, gif)
- ZIP archives (zip)

Icons automatically assigned based on MIME type.

## Security Considerations

### Built-in Security Features

✅ **SQL Injection Protection**
- All database queries use parameterized statements
- LIKE queries properly escaped with `escapeLike()`
- No raw SQL concatenation

✅ **XSS Prevention**
- Twig templates auto-escape all output
- JavaScript uses safe DOM methods (createTextNode)
- No `|raw` filters or innerHTML with user data

✅ **CSRF Protection**
- All forms use Drupal's Form API (built-in CSRF)
- Sensitive AJAX endpoints require CSRF tokens
- Toggle favorite route protected with `_csrf_token: 'TRUE'`

✅ **Access Control**
- Permission checks on all routes
- Owner-based access for edit/delete operations
- 'manage all' permissions for administrators
- Unpublished documents hidden from non-owners

✅ **File Upload Security**
- Extension whitelist validation
- File size limits enforced
- Upload directory restricted to `public://documents/`
- MIME type verification

✅ **Input Validation**
- Form API validation on all inputs
- Maxlength limits on text fields
- Numeric validation with min/max ranges
- Sort parameter whitelist in search

### Best Practices

- Keep allowed file extensions restrictive
- Review permissions regularly
- Monitor download logs for suspicious activity
- Set appropriate file size limits
- Use HTTPS for document downloads
- Regularly update Drupal core and modules

## Troubleshooting

**Documents not appearing?**
- Check document status (published/unpublished)
- Verify user permissions
- Clear Drupal cache: `drush cr`
- Check folder permissions

**Upload fails?**
- Check file size limits in module settings
- Verify file extension is allowed
- Check server PHP upload limits (`upload_max_filesize`, `post_max_size`)
- Ensure `public://documents/` directory exists and is writable
- Check file permissions on server

**Styles not loading?**
- Clear Drupal cache: `drush cr`
- Rebuild theme registry
- Check library is properly attached in templates
- Verify CSS file exists in module

**Folders not working?**
- Clear routing cache: `drush cr`
- Check folder slug uniqueness
- Verify folder status is published
- Check parent folder exists

**Delete button not working?**
- Ensure DocumentDeleteForm.php exists
- Check routing.yml uses correct form class
- Clear cache after code changes
- Verify delete permission

## Performance Tips

- Enable Drupal caching (render, dynamic page cache)
- Add database indexes for custom queries
- Use Views for complex document listings
- Enable aggregation for CSS/JS
- Consider CDN for file downloads
- Use external file storage (S3) for large deployments
- Implement pagination for large folders
- Archive old documents to reduce active dataset

## Migration

### From Categories to Folders

If upgrading from an earlier version with categories:

1. Run update hook: `drush updb`
2. Update hook 9002 removes category_id field
3. Manually migrate category assignments to folders
4. Default folders created automatically

### Data Migration Script

```php
// Migrate documents from old category system
$documents = \Drupal::database()
  ->select('dl_documents', 'd')
  ->fields('d')
  ->execute()
  ->fetchAll();

$category_folder_map = [
  1 => 5, // Map old category ID to new folder ID
  // ...
];

foreach ($documents as $doc) {
  $folder_id = $category_folder_map[$doc->old_category_id] ?? 0;
  \Drupal::database()->update('dl_documents')
    ->fields(['folder_id' => $folder_id])
    ->condition('id', $doc->id)
    ->execute();
}
```

## Update Hooks

**dl_update_9001()** - Add folders table and folder_id field
**dl_update_9002()** - Remove category_id field and vocabulary
**dl_update_9003()** - Add slug field for path-based URLs

Run updates: `drush updb` or visit `/update.php`

## Roadmap / Future Enhancements

- [ ] Document preview (PDF viewer integration)
- [ ] Comments and annotations
- [ ] Email notifications for new uploads
- [ ] Bulk upload with ZIP extraction
- [ ] Advanced file sharing with expiration links
- [ ] Document approval workflow
- [ ] Integration with cloud storage (S3, Google Drive)
- [ ] Full-text search integration (Search API)
- [ ] Document templates
- [ ] OCR for scanned documents
- [ ] Folder permissions (per-folder access control)
- [ ] Document checkout/checkin for editing
- [ ] Automated file conversion
- [ ] REST API endpoints

## Support

For issues, questions, or contributions:
1. Check this documentation
2. Review permissions and settings
3. Clear Drupal cache: `drush cr`
4. Check Drupal logs: Reports > Recent log messages
5. Verify database schema is up to date: `drush updb`

## Changelog

### Version 1.0 (Current)
- Initial stable release
- Hierarchical folder system
- Admin interface with bulk operations
- Security hardening (XSS, SQL injection, CSRF protection)
- Folder name display in admin table
- Proper document delete form
- Search query escaping
- Modern green-themed UI

## License

This module follows Drupal's GPL-2.0+ license.

## Credits

Developed for Drupal 11 with modern web standards and security best practices.

---

**Version:** 1.0
**Drupal:** 11.x
**Status:** Stable
**Last Updated:** December 2025
