/**
 * @file
 * Document Library JavaScript behaviors.
 */

(function ($, Drupal, once) {
  'use strict';

  Drupal.behaviors.documentLibrary = {
    attach: function (context, settings) {

      // Favorite toggle
      once('dl-favorite', '.dl-favorite-btn', context).forEach(function(element) {
        $(element).on('click', function(e) {
          e.preventDefault();
          var $btn = $(this);
          var documentId = $btn.data('document-id');
          var csrfToken = $btn.data('csrf-token');

          $.ajax({
            url: '/documents/' + documentId + '/favorite',
            method: 'POST',
            headers: {
              'X-CSRF-Token': csrfToken
            },
            success: function(response) {
              if (response.favorited) {
                $btn.addClass('is-favorite');
                $btn.attr('title', Drupal.t('Remove from favorites'));
              } else {
                $btn.removeClass('is-favorite');
                $btn.attr('title', Drupal.t('Add to favorites'));
              }
            },
            error: function() {
              alert(Drupal.t('An error occurred. Please try again.'));
            }
          });
        });
      });

      // Search form enhancement
      once('dl-search', '.dl-search-form', context).forEach(function(element) {
        var $form = $(element);
        var $input = $form.find('.dl-search-input');
        var timeout = null;

        // Auto-submit on typing (debounced)
        $input.on('input', function() {
          clearTimeout(timeout);
          timeout = setTimeout(function() {
            if ($input.val().length >= 3 || $input.val().length === 0) {
              $form.submit();
            }
          }, 500);
        });
      });

      // Restore dropped file on upload form
      once('dl-restore-file', '.dl-upload-form', context).forEach(function(element) {
        // Check if there's a pending upload from drag-and-drop
        var pendingUpload = sessionStorage.getItem('dl_pending_upload');
        var pendingData = sessionStorage.getItem('dl_pending_upload_data');

        if (pendingUpload && pendingData) {
          try {
            var fileInfo = JSON.parse(pendingUpload);

            // Convert base64 back to File object
            fetch(pendingData)
              .then(function(res) { return res.blob(); })
              .then(function(blob) {
                // Create a new File object
                var file = new File([blob], fileInfo.name, {
                  type: fileInfo.type,
                  lastModified: fileInfo.lastModified
                });

                // Set the file to the file input
                var $fileInput = $(element).find('input[type="file"]');
                if ($fileInput.length) {
                  // Create DataTransfer object to set files
                  var dataTransfer = new DataTransfer();
                  dataTransfer.items.add(file);
                  $fileInput[0].files = dataTransfer.files;

                  // Trigger change event to show file name
                  $fileInput.trigger('change');

                  // Auto-populate title with filename (without extension)
                  var titleInput = $(element).find('input[name="title"]');
                  if (titleInput.length && !titleInput.val()) {
                    var filename = fileInfo.name.replace(/\.[^/.]+$/, ''); // Remove extension
                    titleInput.val(filename);
                  }

                  // Show success message
                  var $messageDiv = $('<div class="messages messages--status">');
                  var $innerDiv = $('<div role="contentinfo" aria-label="Status message">');
                  $innerDiv.append('<h2 class="visually-hidden">Status message</h2>');
                  $innerDiv.append('<i class="fas fa-check-circle"></i> File "');
                  $innerDiv.append(document.createTextNode(fileInfo.name));
                  $innerDiv.append('" ready for upload');
                  $messageDiv.append($innerDiv);
                  $messageDiv.prependTo(element);

                  setTimeout(function() {
                    $messageDiv.fadeOut(function() { $(this).remove(); });
                  }, 5000);
                }

                // Clear sessionStorage
                sessionStorage.removeItem('dl_pending_upload');
                sessionStorage.removeItem('dl_pending_upload_data');
              })
              .catch(function(error) {
                console.error('Error restoring file:', error);
                sessionStorage.removeItem('dl_pending_upload');
                sessionStorage.removeItem('dl_pending_upload_data');
              });
          } catch (e) {
            console.error('Error parsing pending upload:', e);
            sessionStorage.removeItem('dl_pending_upload');
            sessionStorage.removeItem('dl_pending_upload_data');
          }
        }
      });

      // Drag and drop upload enhancement
      once('dl-dragdrop', '.dl-upload-form .form-managed-file', context).forEach(function(element) {
        var $dropzone = $(element);

        $dropzone.on('dragover', function(e) {
          e.preventDefault();
          e.stopPropagation();
          $(this).addClass('dl-dragover');
        });

        $dropzone.on('dragleave', function(e) {
          e.preventDefault();
          e.stopPropagation();
          $(this).removeClass('dl-dragover');
        });

        $dropzone.on('drop', function(e) {
          e.preventDefault();
          e.stopPropagation();
          $(this).removeClass('dl-dragover');

          var files = e.originalEvent.dataTransfer.files;
          if (files.length > 0) {
            var $input = $(this).find('input[type="file"]');
            if ($input.length && $input[0].files !== undefined) {
              // Create DataTransfer object to set files
              var dataTransfer = new DataTransfer();
              for (var i = 0; i < files.length; i++) {
                dataTransfer.items.add(files[i]);
              }
              $input[0].files = dataTransfer.files;
              $input.trigger('change');
            }
          }
        });
      });

      // Confirmation dialogs
      once('dl-confirm', '.dl-btn-danger', context).forEach(function(element) {
        $(element).on('click', function(e) {
          if (!confirm(Drupal.t('Are you sure you want to proceed?'))) {
            e.preventDefault();
            return false;
          }
        });
      });

      // Admin page: Filter buttons
      once('dl-admin-filter', '.dl-filter-btn', context).forEach(function(element) {
        $(element).on('click', function() {
          var $btn = $(this);
          var filter = $btn.data('filter');

          // Update active state
          $('.dl-filter-btn').removeClass('active');
          $btn.addClass('active');

          // Filter rows
          var $rows = $('.dl-admin-table tbody tr');

          if (filter === 'all') {
            $rows.show();
          } else {
            $rows.each(function() {
              var $row = $(this);
              var status = $row.data('status');

              if (status === filter) {
                $row.show();
              } else {
                $row.hide();
              }
            });
          }
        });
      });

      // Admin page: Search functionality
      once('dl-admin-search', '#dl-admin-search', context).forEach(function(element) {
        var timeout = null;
        $(element).on('input', function() {
          clearTimeout(timeout);
          var searchTerm = $(this).val().toLowerCase();

          timeout = setTimeout(function() {
            var $rows = $('.dl-admin-table tbody tr');

            if (searchTerm === '') {
              $rows.show();
            } else {
              $rows.each(function() {
                var $row = $(this);
                var title = $row.find('.dl-title-link').text().toLowerCase();
                var owner = $row.find('.col-owner').text().toLowerCase();

                if (title.indexOf(searchTerm) > -1 || owner.indexOf(searchTerm) > -1) {
                  $row.show();
                } else {
                  $row.hide();
                }
              });
            }
          }, 300);
        });
      });

      // Admin page: Select all checkbox
      once('dl-select-all', '#dl-select-all', context).forEach(function(element) {
        $(element).on('change', function() {
          var isChecked = $(this).prop('checked');
          $('.dl-row-checkbox:visible').prop('checked', isChecked);
          updateBulkActions();
        });
      });

      // Admin page: Row checkboxes
      once('dl-row-checkbox', '.dl-row-checkbox', context).forEach(function(element) {
        $(element).on('change', function() {
          updateBulkActions();
        });
      });

      // Admin page: Table sorting
      once('dl-admin-sort', '.dl-admin-table th.sortable', context).forEach(function(element) {
        $(element).on('click', function() {
          var $th = $(this);
          var $table = $th.closest('table');
          var column = $th.data('sort');
          var $icon = $th.find('i');
          var order = 'asc';

          // Determine sort order
          if ($th.hasClass('sort-asc')) {
            order = 'desc';
            $th.removeClass('sort-asc').addClass('sort-desc');
            $icon.removeClass('fa-sort fa-sort-up').addClass('fa-sort-down');
          } else {
            $th.removeClass('sort-desc').addClass('sort-asc');
            $icon.removeClass('fa-sort fa-sort-down').addClass('fa-sort-up');
          }

          // Remove sort classes from other columns
          $table.find('th.sortable').not($th).removeClass('sort-asc sort-desc')
            .find('i').removeClass('fa-sort-up fa-sort-down').addClass('fa-sort');

          // Sort rows
          var $tbody = $table.find('tbody');
          var rows = $tbody.find('tr:visible').get();

          rows.sort(function(a, b) {
            var aVal, bVal;

            switch(column) {
              case 'title':
                aVal = $(a).find('.dl-title-link').text().toLowerCase();
                bVal = $(b).find('.dl-title-link').text().toLowerCase();
                break;
              case 'owner':
                aVal = $(a).find('.col-owner').text().toLowerCase();
                bVal = $(b).find('.col-owner').text().toLowerCase();
                break;
              case 'downloads':
                aVal = parseInt($(a).find('.dl-downloads-count').text()) || 0;
                bVal = parseInt($(b).find('.dl-downloads-count').text()) || 0;
                break;
              case 'size':
                aVal = parseSizeToBytes($(a).find('.col-size').text());
                bVal = parseSizeToBytes($(b).find('.col-size').text());
                break;
              case 'created':
                aVal = $(a).find('.col-date').text();
                bVal = $(b).find('.col-date').text();
                break;
              default:
                aVal = $(a).find('td').eq($th.index()).text();
                bVal = $(b).find('td').eq($th.index()).text();
            }

            if (order === 'asc') {
              return aVal > bVal ? 1 : aVal < bVal ? -1 : 0;
            } else {
              return aVal < bVal ? 1 : aVal > bVal ? -1 : 0;
            }
          });

          $.each(rows, function(index, row) {
            $tbody.append(row);
          });
        });
      });

      // Animate cards on scroll
      if ('IntersectionObserver' in window) {
        var observer = new IntersectionObserver(function(entries) {
          entries.forEach(function(entry) {
            if (entry.isIntersecting) {
              entry.target.style.opacity = '1';
              entry.target.style.transform = 'translateY(0)';
            }
          });
        }, {
          threshold: 0.1
        });

        once('dl-animate', '.dl-document-card', context).forEach(function(element) {
          element.style.opacity = '0';
          element.style.transform = 'translateY(20px)';
          element.style.transition = 'opacity 0.4s, transform 0.4s';
          observer.observe(element);
        });
      }

      // Keyboard shortcuts
      once('dl-shortcuts', 'body', context).forEach(function(element) {
        $(document).on('keydown', function(e) {
          // Ctrl/Cmd + K: Focus search
          if ((e.ctrlKey || e.metaKey) && e.key === 'k') {
            e.preventDefault();
            $('.dl-search-input').focus();
          }

          // Ctrl/Cmd + U: Go to upload
          if ((e.ctrlKey || e.metaKey) && e.key === 'u') {
            e.preventDefault();
            var uploadLink = $('a[href*="/documents/upload"]');
            if (uploadLink.length) {
              window.location.href = uploadLink.attr('href');
            }
          }
        });
      });

      // File type icons
      once('dl-file-icon', '.dl-document-card', context).forEach(function(element) {
        var $card = $(element);
        var mimeType = $card.data('mime-type');
        var $icon = $card.find('.dl-document-icon');

        if ($icon.length && mimeType) {
          var iconClass = getFileIcon(mimeType);
          $icon.html('<i class="' + iconClass + '"></i>');
        }
      });

      // Tooltips
      once('dl-tooltip', '[title]', context).forEach(function(element) {
        $(element).attr('data-toggle', 'tooltip');
      });

      // Auto-hide messages
      once('dl-messages', '.messages', context).forEach(function(element) {
        $(element).delay(5000).fadeOut('slow');
      });

      // Print functionality
      once('dl-print', '.dl-btn-print', context).forEach(function(element) {
        $(element).on('click', function(e) {
          e.preventDefault();
          window.print();
        });
      });

      // Copy to clipboard
      once('dl-copy', '.dl-btn-copy', context).forEach(function(element) {
        $(element).on('click', function(e) {
          e.preventDefault();
          var text = $(this).data('copy-text');

          if (navigator.clipboard) {
            navigator.clipboard.writeText(text).then(function() {
              alert(Drupal.t('Copied to clipboard!'));
            });
          }
        });
      });

      // Folder tree expand/collapse
      once('dl-folder-toggle', '.dl-folder-toggle', context).forEach(function(element) {
        $(element).on('click', function(e) {
          e.preventDefault();
          e.stopPropagation();

          var $toggle = $(this);
          var $folderItem = $toggle.closest('.dl-folder-item');

          // Toggle expanded class
          $folderItem.toggleClass('expanded');

          // Save state to localStorage
          var folderId = $folderItem.find('a').attr('href').split('/').pop();
          var expandedFolders = JSON.parse(localStorage.getItem('dl-expanded-folders') || '[]');

          if ($folderItem.hasClass('expanded')) {
            if (!expandedFolders.includes(folderId)) {
              expandedFolders.push(folderId);
            }
          } else {
            expandedFolders = expandedFolders.filter(function(id) { return id !== folderId; });
          }

          localStorage.setItem('dl-expanded-folders', JSON.stringify(expandedFolders));
        });
      });

      // Restore expanded state from localStorage
      once('dl-folder-restore', '.dl-folder-tree', context).forEach(function(element) {
        var expandedFolders = JSON.parse(localStorage.getItem('dl-expanded-folders') || '[]');

        expandedFolders.forEach(function(folderId) {
          $('.dl-folder-item', element).each(function() {
            var $item = $(this);
            var href = $item.find('a').attr('href');
            if (href && href.split('/').pop() === folderId) {
              $item.addClass('expanded');
            }
          });
        });
      });

      // Auto-expand current folder's parents
      once('dl-folder-auto-expand', '.dl-folder-tree', context).forEach(function(element) {
        var $activeItem = $('.dl-folder-item.active', element);

        if ($activeItem.length) {
          // Expand all parent folders
          $activeItem.parents('.dl-folder-item').addClass('expanded');
        }
      });

      // Filedepot-style tree navigation
      once('dl-tree-toggle-click', '.dl-tree-toggle', context).forEach(function(element) {
        $(element).on('click', function(e) {
          e.preventDefault();
          e.stopPropagation();

          var $treeItem = $(this).closest('.dl-tree-item');
          $treeItem.toggleClass('expanded');

          // Save state
          var folderId = $treeItem.data('folder-id');
          var expandedFolders = JSON.parse(localStorage.getItem('dl-tree-expanded') || '[]');

          if ($treeItem.hasClass('expanded')) {
            if (!expandedFolders.includes(folderId)) {
              expandedFolders.push(folderId);
            }
          } else {
            expandedFolders = expandedFolders.filter(function(id) { return id !== folderId; });
          }

          localStorage.setItem('dl-tree-expanded', JSON.stringify(expandedFolders));
        });
      });

      // Restore tree expansion state
      once('dl-tree-restore-state', '.dl-tree-item', context).forEach(function(element) {
        var expandedFolders = JSON.parse(localStorage.getItem('dl-tree-expanded') || '[]');
        var folderId = $(element).data('folder-id');

        if (expandedFolders.includes(folderId)) {
          $(element).addClass('expanded');
        }
      });

      // Auto-expand active item's parents in tree
      once('dl-tree-auto-expand', '.dl-tree-item.active', context).forEach(function(element) {
        $(element).parents('.dl-tree-item').addClass('expanded');
      });

      // Drag and drop file upload
      once('dl-drag-drop-upload', '.dl-file-list-container', context).forEach(function(container) {
        var $container = $(container);
        var $dropZone = $('#dl-drop-zone');
        var dragCounter = 0;

        $container.on('dragenter', function(e) {
          e.preventDefault();
          e.stopPropagation();
          dragCounter++;
          $dropZone.addClass('active');
        });

        $container.on('dragleave', function(e) {
          e.preventDefault();
          e.stopPropagation();
          dragCounter--;
          if (dragCounter === 0) {
            $dropZone.removeClass('active');
          }
        });

        $container.on('dragover', function(e) {
          e.preventDefault();
          e.stopPropagation();
        });

        $container.on('drop', function(e) {
          e.preventDefault();
          e.stopPropagation();
          dragCounter = 0;
          $dropZone.removeClass('active');

          var files = e.originalEvent.dataTransfer.files;
          if (files.length > 0) {
            handleFileUpload(files, $dropZone.data('folder-id'));
          }
        });
      });

      // File input change handler
      once('dl-file-input-change', '#dl-upload-zone', context).forEach(function(element) {
        $(element).on('change', function() {
          var files = this.files;
          var folderId = $(this).data('folder-id');
          if (files.length > 0) {
            handleFileUpload(files, folderId);
          }
        });
      });

      // Table sorting
      once('dl-table-sort', '.dl-file-table th.sortable', context).forEach(function(element) {
        $(element).on('click', function() {
          var $th = $(this);
          var $table = $th.closest('table');
          var column = $th.data('sort');
          var $icon = $th.find('i');
          var order = 'asc';

          // Determine sort order
          if ($th.hasClass('sort-asc')) {
            order = 'desc';
            $th.removeClass('sort-asc').addClass('sort-desc');
            $icon.removeClass('fa-sort fa-sort-up').addClass('fa-sort-down');
          } else {
            $th.removeClass('sort-desc').addClass('sort-asc');
            $icon.removeClass('fa-sort fa-sort-down').addClass('fa-sort-up');
          }

          // Remove sort classes from other columns
          $table.find('th.sortable').not($th).removeClass('sort-asc sort-desc')
            .find('i').removeClass('fa-sort-up fa-sort-down').addClass('fa-sort');

          // Sort rows
          var $tbody = $table.find('tbody');
          var rows = $tbody.find('tr').not('.dl-empty-row').get();

          rows.sort(function(a, b) {
            var aVal, bVal;

            switch(column) {
              case 'filename':
                aVal = $(a).find('.col-filename a').text().toLowerCase();
                bVal = $(b).find('.col-filename a').text().toLowerCase();
                break;
              case 'date':
                aVal = $(a).find('.col-date').text();
                bVal = $(b).find('.col-date').text();
                // Convert to timestamp for comparison
                aVal = new Date(aVal).getTime() || 0;
                bVal = new Date(bVal).getTime() || 0;
                break;
              case 'size':
                aVal = parseSizeToBytes($(a).find('.col-size').text());
                bVal = parseSizeToBytes($(b).find('.col-size').text());
                break;
              case 'owner':
                aVal = $(a).find('.col-owner').text().toLowerCase();
                bVal = $(b).find('.col-owner').text().toLowerCase();
                break;
              default:
                return 0;
            }

            // Folders always come first
            var aIsFolder = $(a).hasClass('dl-folder-row');
            var bIsFolder = $(b).hasClass('dl-folder-row');

            if (aIsFolder && !bIsFolder) return -1;
            if (!aIsFolder && bIsFolder) return 1;

            // Then sort by value
            if (aVal < bVal) return order === 'asc' ? -1 : 1;
            if (aVal > bVal) return order === 'asc' ? 1 : -1;
            return 0;
          });

          $.each(rows, function(index, row) {
            $tbody.append(row);
          });
        });
      });

      // Select all checkbox
      once('dl-select-all', '#select-all', context).forEach(function(element) {
        $(element).on('change', function() {
          var checked = $(this).prop('checked');
          $('.dl-file-table tbody .dl-checkbox').prop('checked', checked);
          $('.dl-file-table tbody tr').toggleClass('selected', checked);
        });
      });

      // Individual row checkboxes
      once('dl-row-checkbox', '.dl-file-table tbody .dl-checkbox', context).forEach(function(element) {
        $(element).on('change', function() {
          $(this).closest('tr').toggleClass('selected', $(this).prop('checked'));

          // Update select-all checkbox
          var totalCheckboxes = $('.dl-file-table tbody .dl-checkbox').length;
          var checkedCheckboxes = $('.dl-file-table tbody .dl-checkbox:checked').length;
          $('#select-all').prop('checked', totalCheckboxes === checkedCheckboxes && totalCheckboxes > 0);
        });
      });

    }
  };

  /**
   * Update bulk actions visibility and count
   */
  function updateBulkActions() {
    var checkedCount = $('.dl-row-checkbox:checked').length;
    var $bulkActions = $('.dl-bulk-actions');

    if (checkedCount > 0) {
      $bulkActions.show();
      $('.dl-bulk-count').text(checkedCount + ' ' + Drupal.t('items selected'));
    } else {
      $bulkActions.hide();
    }
  }

  /**
   * Handle file upload
   */
  function handleFileUpload(files, folderId) {
    // Show upload progress notification
    var fileCount = files.length;

    // Create a simple notification
    var $notification = $('<div class="dl-upload-notification">')
      .html('<i class="fas fa-spinner fa-spin"></i> Uploading ' + fileCount + ' file(s)...')
      .appendTo('body');

    if (fileCount === 1) {
      // Store file data temporarily and redirect to upload form
      var file = files[0];

      // Store file info in sessionStorage
      var fileData = {
        name: file.name,
        size: file.size,
        type: file.type,
        lastModified: file.lastModified,
        folderId: folderId || 0
      };
      sessionStorage.setItem('dl_pending_upload', JSON.stringify(fileData));

      // Use FileReader to store the file as base64 in sessionStorage
      var reader = new FileReader();
      reader.onload = function(e) {
        sessionStorage.setItem('dl_pending_upload_data', e.target.result);
        // Redirect to upload form with folder in query string
        window.location.href = '/documents/upload?folder=' + (folderId || 0);
      };
      reader.onerror = function() {
        $notification.html('<i class="fas fa-exclamation-circle"></i> Error reading file')
          .addClass('dl-notification-error');
        setTimeout(function() {
          $notification.fadeOut(function() { $(this).remove(); });
        }, 3000);
      };
      reader.readAsDataURL(file);
    } else {
      // Multiple files - show message
      $notification.html('<i class="fas fa-info-circle"></i> Please use the Upload button for multiple files')
        .addClass('dl-notification-info');
      setTimeout(function() {
        $notification.fadeOut(function() { $(this).remove(); });
      }, 3000);
    }
  }

  /**
   * Parse size string to bytes for sorting
   */
  function parseSizeToBytes(sizeStr) {
    if (!sizeStr || sizeStr === '-') return 0;

    var units = {'B': 1, 'KB': 1024, 'MB': 1048576, 'GB': 1073741824};
    var parts = sizeStr.trim().match(/^([\d.]+)\s*([A-Z]+)$/i);

    if (!parts) return 0;

    var value = parseFloat(parts[1]);
    var unit = parts[2].toUpperCase();

    return value * (units[unit] || 1);
  }

  /**
   * Get file icon class based on MIME type.
   */
  function getFileIcon(mimeType) {
    var icons = {
      'application/pdf': 'fas fa-file-pdf',
      'application/msword': 'fas fa-file-word',
      'application/vnd.openxmlformats-officedocument.wordprocessingml.document': 'fas fa-file-word',
      'application/vnd.ms-excel': 'fas fa-file-excel',
      'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet': 'fas fa-file-excel',
      'application/vnd.ms-powerpoint': 'fas fa-file-powerpoint',
      'application/vnd.openxmlformats-officedocument.presentationml.presentation': 'fas fa-file-powerpoint',
      'text/plain': 'fas fa-file-alt',
      'text/csv': 'fas fa-file-csv',
      'application/zip': 'fas fa-file-archive',
      'image/jpeg': 'fas fa-file-image',
      'image/png': 'fas fa-file-image',
      'image/gif': 'fas fa-file-image'
    };

    return icons[mimeType] || 'fas fa-file';
  }

})(jQuery, Drupal, once);
