<?php

namespace Drupal\dl\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\file\Entity\File;
use Drupal\Core\Database\Connection;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Url;
use Drupal\dl\FolderManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for editing documents.
 */
class DocumentEditForm extends FormBase
{

  protected $database;
  protected $config;
  protected $folderManager;

  public function __construct(Connection $database, ConfigFactoryInterface $config_factory, FolderManager $folder_manager)
  {
    $this->database = $database;
    $this->config = $config_factory->get('dl.settings');
    $this->folderManager = $folder_manager;
  }

  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('database'),
      $container->get('config.factory'),
      $container->get('dl.folder_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId()
  {
    return 'dl_document_edit_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $document = NULL)
  {
    $doc = $this->database->query('SELECT * FROM {dl_documents} WHERE id = :id', [':id' => $document])->fetchObject();

    if (!$doc) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException();
    }

    // Check permissions
    if ($doc->uid != $this->currentUser()->id() && !$this->currentUser()->hasPermission('manage all documents')) {
      throw new \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException();
    }

    $form['#attributes']['class'][] = 'dl-edit-form';
    $form['#attached']['library'][] = 'dl/document-library';
    $form['#prefix'] = '<div class="dl-form-page">';
    $form['#suffix'] = '</div>';

    $form['intro'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['dl-form-hero']],
      'content' => [
        '#markup' => '<div class="dl-form-hero__content">
          <span class="dl-form-hero__icon"><i class="fas fa-file-pen"></i></span>
          <div>
            <p class="dl-form-hero__eyebrow">' . $this->t('Document Library') . '</p>
            <h1 class="dl-form-hero__title">' . $this->t('Edit document') . '</h1>
            <p class="dl-form-hero__lede">' . $this->t('Update metadata, move the file, or upload a new version while keeping history intact.') . '</p>
          </div>
        </div>',
      ],
      'actions' => [
        '#type' => 'container',
        '#attributes' => ['class' => ['dl-form-hero__actions']],
        'back' => [
          '#type' => 'link',
          '#title' => $this->t('Back to Document'),
          '#url' => Url::fromRoute('dl.document.view', ['document' => $document]),
          '#attributes' => ['class' => ['dl-btn', 'dl-btn-secondary']],
        ],
      ],
    ];

    $form['card'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['dl-form-card']],
    ];

    $form['card']['document_id'] = [
      '#type' => 'value',
      '#value' => $document,
    ];

    $form['card']['title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Document Title'),
      '#required' => TRUE,
      '#maxlength' => 255,
      '#default_value' => $doc->title,
    ];

    $form['card']['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description'),
      '#rows' => 4,
      '#default_value' => $doc->description,
    ];

    $form['card']['folder'] = [
      '#type' => 'select',
      '#title' => $this->t('Folder'),
      '#options' => $this->folderManager->getFolderOptions(),
      '#default_value' => $doc->folder_id,
      '#description' => $this->t('Select the folder where this document should be stored.'),
    ];

    // Get settings
    $max_file_size = $this->config->get('max_file_size') ?: 50;
    $allowed_extensions = $this->config->get('allowed_extensions') ?: 'pdf doc docx xls xlsx ppt pptx txt csv zip';

    $form['card']['file'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('Replace File (optional)'),
      '#upload_location' => 'public://documents/',
      '#upload_validators' => [
        'FileSizeLimit' => [
          'fileLimit' => (int) $max_file_size * 1024 * 1024,
        ],
      ],
      '#file_extensions' => trim($allowed_extensions),
      '#description' => $this->t('Leave empty to keep the current file. Upload a new file to create a new version.'),
    ];

    $form['card']['version_notes'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Version Notes'),
      '#rows' => 3,
      '#description' => $this->t('Describe changes in this version (only if uploading a new file).'),
      '#states' => [
        'visible' => [
          ':input[name="files[file]"]' => ['filled' => TRUE],
        ],
      ],
    ];

    $form['card']['version'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Version'),
      '#default_value' => $doc->version,
      '#size' => 10,
      '#required' => TRUE,
    ];

    $form['card']['tags'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Tags'),
      '#description' => $this->t('Comma-separated tags.'),
      '#maxlength' => 255,
      '#default_value' => $doc->tags,
    ];

    $form['card']['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Published'),
      '#default_value' => $doc->status,
    ];

    $form['card']['actions'] = [
      '#type' => 'actions',
    ];

    $form['card']['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Update Document'),
      '#button_type' => 'primary',
      '#attributes' => ['class' => ['dl-btn', 'dl-btn-primary']],
    ];

    $form['card']['actions']['cancel'] = [
      '#type' => 'link',
      '#title' => $this->t('Cancel'),
      '#url' => \Drupal\Core\Url::fromRoute('dl.document.view', ['document' => $document]),
      '#attributes' => ['class' => ['dl-btn', 'dl-btn-secondary']],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state)
  {
    $document_id = $form_state->getValue('document_id');
    $doc = $this->database->query('SELECT * FROM {dl_documents} WHERE id = :id', [':id' => $document_id])->fetchObject();

    $file_id = $doc->file_id;
    $new_version = FALSE;

    // Check if new file uploaded
    if ($form_state->getValue('file') && !empty($form_state->getValue('file')[0])) {
      $new_file_id = $form_state->getValue('file')[0];
      $file = File::load($new_file_id);

      if ($file) {
        $file->setPermanent();
        $file->save();
        $file_id = $new_file_id;
        $new_version = TRUE;

        // Create version entry
        $this->database->insert('dl_versions')
          ->fields([
            'document_id' => $document_id,
            'file_id' => $file_id,
            'version' => $form_state->getValue('version'),
            'uid' => $this->currentUser()->id(),
            'created' => time(),
            'notes' => $form_state->getValue('version_notes') ?: 'Updated version',
          ])
          ->execute();
      }
    }

    // Update document
    $this->database->update('dl_documents')
      ->fields([
        'title' => $form_state->getValue('title'),
        'description' => $form_state->getValue('description'),
        'file_id' => $file_id,
        'folder_id' => $form_state->getValue('folder') ?: 0,
        'version' => $form_state->getValue('version'),
        'status' => $form_state->getValue('status') ? 1 : 0,
        'changed' => time(),
        'tags' => $form_state->getValue('tags'),
      ])
      ->condition('id', $document_id)
      ->execute();

    $message = $new_version
      ? $this->t('Document "@title" has been updated with a new version.', ['@title' => $form_state->getValue('title')])
      : $this->t('Document "@title" has been updated.', ['@title' => $form_state->getValue('title')]);

    $this->messenger()->addStatus($message);
    $form_state->setRedirect('dl.document.view', ['document' => $document_id]);
  }
}
