<?php

namespace Drupal\dl\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\dl\FolderManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for creating folders.
 */
class FolderCreateForm extends FormBase {

  /**
   * The folder manager service.
   *
   * @var \Drupal\dl\FolderManager
   */
  protected $folderManager;

  /**
   * Constructs a FolderCreateForm object.
   *
   * @param \Drupal\dl\FolderManager $folder_manager
   *   The folder manager service.
   */
  public function __construct(FolderManager $folder_manager) {
    $this->folderManager = $folder_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('dl.folder_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dl_folder_create_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $parent = NULL) {
    if ($parent === NULL) {
      $parent = (int) \Drupal::request()->query->get('parent', 0);
    }

    $form['#attributes']['class'][] = 'dl-folder-form';
    $form['#attached']['library'][] = 'dl/document-library';
    $form['#prefix'] = '<div class="dl-form-page">';
    $form['#suffix'] = '</div>';

    $form['intro'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['dl-form-hero']],
      'content' => [
        '#markup' => '<div class="dl-form-hero__content">
          <span class="dl-form-hero__icon"><i class="fas fa-folder-plus"></i></span>
          <div>
            <p class="dl-form-hero__eyebrow">' . $this->t('Document Library') . '</p>
            <h1 class="dl-form-hero__title">' . $this->t('Create a new folder') . '</h1>
            <p class="dl-form-hero__lede">' . $this->t('Group related documents together and keep your library organized with a clear folder structure.') . '</p>
          </div>
        </div>',
      ],
      'actions' => [
        '#type' => 'container',
        '#attributes' => ['class' => ['dl-form-hero__actions']],
        'back' => [
          '#type' => 'link',
          '#title' => $this->t('Back to Library'),
          '#url' => Url::fromRoute('dl.library'),
          '#attributes' => ['class' => ['dl-btn', 'dl-btn-secondary']],
        ],
      ],
    ];

    $form['card'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['dl-form-card']],
    ];

    $form['card']['parent_id'] = [
      '#type' => 'value',
      '#value' => $parent ?: 0,
    ];

    $form['card']['name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Folder Name'),
      '#required' => TRUE,
      '#maxlength' => 255,
    ];

    $form['card']['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description'),
      '#rows' => 3,
    ];

    $form['card']['parent_select'] = [
      '#type' => 'select',
      '#title' => $this->t('Parent Folder'),
      '#options' => $this->folderManager->getFolderOptions(),
      '#default_value' => $parent ?: 0,
      '#description' => $this->t('Select the parent folder (leave as Root for top-level folder).'),
    ];

    $form['card']['weight'] = [
      '#type' => 'number',
      '#title' => $this->t('Sort Order'),
      '#default_value' => 0,
      '#description' => $this->t('Lower numbers appear first.'),
    ];

    $form['card']['actions'] = [
      '#type' => 'actions',
    ];

    $form['card']['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Create Folder'),
      '#button_type' => 'primary',
      '#attributes' => ['class' => ['dl-btn', 'dl-btn-primary']],
    ];

    $form['card']['actions']['cancel'] = [
      '#type' => 'link',
      '#title' => $this->t('Cancel'),
      '#url' => Url::fromRoute('dl.library'),
      '#attributes' => ['class' => ['dl-btn', 'dl-btn-secondary']],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $name = $form_state->getValue('name');
    $parent_id = $form_state->getValue('parent_select');
    $description = $form_state->getValue('description');
    $weight = (int) $form_state->getValue('weight');

    $folder_id = $this->folderManager->createFolder($name, $parent_id, $description, $weight);

    $this->messenger()->addStatus($this->t('Folder "@name" has been created successfully.', [
      '@name' => $name,
    ]));

    // Redirect to folder view
    $form_state->setRedirect('dl.folder', ['folder' => $folder_id]);
  }

}
