<?php

namespace Drupal\dl\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\Core\Database\Connection;
use Drupal\file\Entity\File;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for deleting documents.
 */
class DocumentDeleteForm extends ConfirmFormBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The document ID.
   *
   * @var int
   */
  protected $documentId;

  /**
   * The document object.
   *
   * @var object
   */
  protected $document;

  /**
   * Constructs a DocumentDeleteForm object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(Connection $database) {
    $this->database = $database;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dl_document_delete_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $document = NULL) {
    $this->documentId = $document;
    $this->document = $this->database->query(
      'SELECT * FROM {dl_documents} WHERE id = :id',
      [':id' => $document]
    )->fetchObject();

    if (!$this->document) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException();
    }

    // Check permissions
    if ($this->document->uid != $this->currentUser()->id() &&
        !$this->currentUser()->hasPermission('manage all documents')) {
      throw new \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException();
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to delete the document "@title"?', [
      '@title' => $this->document->title,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->t('This action cannot be undone. All versions of this document will be deleted.');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('dl.document.view', ['document' => $this->documentId]);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Delete associated file
    $file = File::load($this->document->file_id);
    if ($file) {
      $file->delete();
    }

    // Delete all version files
    $versions = $this->database->query(
      'SELECT file_id FROM {dl_versions} WHERE document_id = :id',
      [':id' => $this->documentId]
    )->fetchAll();

    foreach ($versions as $version) {
      $version_file = File::load($version->file_id);
      if ($version_file && $version_file->id() != $this->document->file_id) {
        $version_file->delete();
      }
    }

    // Delete version records
    $this->database->delete('dl_versions')
      ->condition('document_id', $this->documentId)
      ->execute();

    // Delete favorites
    $this->database->delete('dl_favorites')
      ->condition('document_id', $this->documentId)
      ->execute();

    // Delete download records
    $this->database->delete('dl_downloads')
      ->condition('document_id', $this->documentId)
      ->execute();

    // Delete document
    $this->database->delete('dl_documents')
      ->condition('id', $this->documentId)
      ->execute();

    $this->messenger()->addStatus($this->t('Document "@title" has been deleted.', [
      '@title' => $this->document->title,
    ]));

    $form_state->setRedirect('dl.library');
  }

}
