<?php

namespace Drupal\dl\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure document fields display and order.
 */
class DocumentFieldConfigForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['dl.field_config'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dl_document_field_config_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('dl.field_config');

    $form['#attached']['library'][] = 'dl/document-library';
    $form['#attached']['library'][] = 'core/drupal.tabledrag';

    $form['description'] = [
      '#markup' => '<p>' . $this->t('Configure the document fields, their order, widgets, and visibility. Drag fields to reorder them.') . '</p>',
    ];

    // Get current field configuration
    $fields = $config->get('fields') ?: $this->getDefaultFieldConfig();

    $form['fields'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Field'),
        $this->t('Widget'),
        $this->t('Weight'),
        $this->t('Enabled'),
      ],
      '#empty' => $this->t('No fields available.'),
      '#tabledrag' => [
        [
          'action' => 'order',
          'relationship' => 'sibling',
          'group' => 'field-weight',
        ],
      ],
    ];

    // Sort fields by weight
    uasort($fields, function($a, $b) {
      return $a['weight'] <=> $b['weight'];
    });

    foreach ($fields as $field_name => $field_config) {
      $form['fields'][$field_name]['#attributes']['class'][] = 'draggable';
      $form['fields'][$field_name]['#weight'] = $field_config['weight'];

      // Field label
      $form['fields'][$field_name]['label'] = [
        '#markup' => '<strong>' . $field_config['label'] . '</strong><br><small>' . $field_name . '</small>',
      ];

      // Widget selector
      $form['fields'][$field_name]['widget'] = [
        '#type' => 'select',
        '#options' => $this->getWidgetOptions($field_name),
        '#default_value' => $field_config['widget'],
        '#title' => $this->t('Widget for @field', ['@field' => $field_config['label']]),
        '#title_display' => 'invisible',
      ];

      // Weight
      $form['fields'][$field_name]['weight'] = [
        '#type' => 'weight',
        '#title' => $this->t('Weight for @field', ['@field' => $field_config['label']]),
        '#title_display' => 'invisible',
        '#default_value' => $field_config['weight'],
        '#attributes' => ['class' => ['field-weight']],
        '#delta' => 50,
      ];

      // Enabled checkbox
      $form['fields'][$field_name]['enabled'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Enable @field', ['@field' => $field_config['label']]),
        '#title_display' => 'invisible',
        '#default_value' => $field_config['enabled'],
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('dl.field_config');
    $fields = [];

    foreach ($form_state->getValue('fields') as $field_name => $field_data) {
      $fields[$field_name] = [
        'label' => $this->getDefaultFieldConfig()[$field_name]['label'],
        'type' => $this->getDefaultFieldConfig()[$field_name]['type'],
        'widget' => $field_data['widget'],
        'weight' => $field_data['weight'],
        'enabled' => $field_data['enabled'],
      ];
    }

    $config->set('fields', $fields)->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Get default field configuration.
   */
  protected function getDefaultFieldConfig() {
    return [
      'title' => [
        'label' => $this->t('Document Title'),
        'type' => 'string',
        'widget' => 'textfield',
        'weight' => 0,
        'enabled' => TRUE,
      ],
      'description' => [
        'label' => $this->t('Description'),
        'type' => 'text',
        'widget' => 'textarea',
        'weight' => 1,
        'enabled' => TRUE,
      ],
      'folder' => [
        'label' => $this->t('Folder'),
        'type' => 'reference',
        'widget' => 'select',
        'weight' => 2,
        'enabled' => TRUE,
      ],
      'file' => [
        'label' => $this->t('File'),
        'type' => 'file',
        'widget' => 'managed_file',
        'weight' => 3,
        'enabled' => TRUE,
      ],
      'version' => [
        'label' => $this->t('Version'),
        'type' => 'string',
        'widget' => 'textfield',
        'weight' => 5,
        'enabled' => TRUE,
      ],
      'tags' => [
        'label' => $this->t('Tags'),
        'type' => 'string',
        'widget' => 'textfield',
        'weight' => 6,
        'enabled' => TRUE,
      ],
      'status' => [
        'label' => $this->t('Published'),
        'type' => 'boolean',
        'widget' => 'checkbox',
        'weight' => 7,
        'enabled' => TRUE,
      ],
    ];
  }

  /**
   * Get widget options for a field type.
   */
  protected function getWidgetOptions($field_name) {
    $default_config = $this->getDefaultFieldConfig();
    $field_type = $default_config[$field_name]['type'];

    $options = [];

    switch ($field_type) {
      case 'string':
        $options = [
          'textfield' => $this->t('Text field'),
          'textfield_autocomplete' => $this->t('Autocomplete text field'),
        ];
        break;

      case 'text':
        $options = [
          'textarea' => $this->t('Text area'),
          'text_textarea_with_summary' => $this->t('Text area with summary'),
        ];
        break;

      case 'reference':
        $options = [
          'select' => $this->t('Select list'),
          'entity_autocomplete' => $this->t('Autocomplete'),
          'checkboxes' => $this->t('Check boxes/radio buttons'),
        ];
        break;

      case 'file':
        $options = [
          'managed_file' => $this->t('File upload'),
          'file_generic' => $this->t('Generic file'),
        ];
        break;

      case 'boolean':
        $options = [
          'checkbox' => $this->t('Single on/off checkbox'),
        ];
        break;

      default:
        $options = [
          'default' => $this->t('Default'),
        ];
    }

    return $options;
  }

}
