<?php

namespace Drupal\dl\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\file\Entity\File;
use Drupal\Core\Database\Connection;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\dl\FolderManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for uploading documents.
 */
class DocumentUploadForm extends FormBase
{

  protected $database;
  protected $config;
  protected $folderManager;

  public function __construct(Connection $database, ConfigFactoryInterface $config_factory, FolderManager $folder_manager)
  {
    $this->database = $database;
    $this->config = $config_factory->get('dl.settings');
    $this->folderManager = $folder_manager;
  }

  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('database'),
      $container->get('config.factory'),
      $container->get('dl.folder_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId()
  {
    return 'dl_document_upload_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $folder = NULL)
  {
    $form['#attributes']['class'][] = 'dl-upload-form';
    $form['#attached']['library'][] = 'dl/document-library';

    $form['folder_id'] = [
      '#type' => 'value',
      '#value' => $folder ?: 0,
    ];

    $form['title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Document Title'),
      '#required' => TRUE,
      '#maxlength' => 255,
    ];

    $form['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description'),
      '#rows' => 4,
    ];

    $form['folder'] = [
      '#type' => 'select',
      '#title' => $this->t('Folder'),
      '#options' => $this->folderManager->getFolderOptions(),
      '#default_value' => $folder ?: 0,
      '#description' => $this->t('Select the folder where this document should be stored.'),
    ];

    // Get settings
    $max_file_size = $this->config->get('max_file_size') ?: 50;
    $allowed_extensions = $this->config->get('allowed_extensions') ?: 'jpg jpeg gif png txt doc docx xls xlsx pdf ppt pptx pps odt ods odp csv zip';

    $form['file'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('File'),
      '#required' => TRUE,
      '#upload_location' => 'public://documents/',
      '#upload_validators' => [
        'file_validate_extensions' => [trim($allowed_extensions)],
        'file_validate_size' => [(int) $max_file_size * 1024 * 1024],
      ],
    ];

    $form['version'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Version'),
      '#default_value' => '1.0',
      '#size' => 10,
    ];

    $form['tags'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Tags'),
      '#description' => $this->t('Comma-separated tags for better organization and search.'),
      '#maxlength' => 255,
    ];

    $form['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Published'),
      '#default_value' => 1,
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Upload Document'),
      '#button_type' => 'primary',
    ];

    $form['actions']['cancel'] = [
      '#type' => 'link',
      '#title' => $this->t('Cancel'),
      '#url' => \Drupal\Core\Url::fromRoute('dl.library'),
      '#attributes' => ['class' => ['button']],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state)
  {
    $file_id = $form_state->getValue('file')[0];
    $file = File::load($file_id);

    if ($file) {
      $file->setPermanent();
      $file->save();

      $uuid = \Drupal::service('uuid')->generate();

      // Insert document
      $document_id = $this->database->insert('dl_documents')
        ->fields([
          'uuid' => $uuid,
          'title' => $form_state->getValue('title'),
          'description' => $form_state->getValue('description'),
          'file_id' => $file_id,
          'folder_id' => $form_state->getValue('folder') ?: 0,
          'version' => $form_state->getValue('version'),
          'status' => $form_state->getValue('status') ? 1 : 0,
          'uid' => $this->currentUser()->id(),
          'created' => time(),
          'changed' => time(),
          'downloads' => 0,
          'tags' => $form_state->getValue('tags'),
        ])
        ->execute();

      // Create initial version entry
      $this->database->insert('dl_versions')
        ->fields([
          'document_id' => $document_id,
          'file_id' => $file_id,
          'version' => $form_state->getValue('version'),
          'uid' => $this->currentUser()->id(),
          'created' => time(),
          'notes' => 'Initial upload',
        ])
        ->execute();

      $this->messenger()->addStatus($this->t('Document "@title" has been uploaded successfully.', [
        '@title' => $form_state->getValue('title'),
      ]));

      $form_state->setRedirect('dl.document.view', ['document' => $document_id]);
    }
  }
}
