<?php

namespace Drupal\dl\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\dl\FolderManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for creating folders.
 */
class FolderCreateForm extends FormBase {

  /**
   * The folder manager service.
   *
   * @var \Drupal\dl\FolderManager
   */
  protected $folderManager;

  /**
   * Constructs a FolderCreateForm object.
   *
   * @param \Drupal\dl\FolderManager $folder_manager
   *   The folder manager service.
   */
  public function __construct(FolderManager $folder_manager) {
    $this->folderManager = $folder_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('dl.folder_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dl_folder_create_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $parent = NULL) {
    $form['#attributes']['class'][] = 'dl-folder-form';

    $form['parent_id'] = [
      '#type' => 'value',
      '#value' => $parent ?: 0,
    ];

    $form['name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Folder Name'),
      '#required' => TRUE,
      '#maxlength' => 255,
    ];

    $form['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description'),
      '#rows' => 3,
    ];

    $form['parent_select'] = [
      '#type' => 'select',
      '#title' => $this->t('Parent Folder'),
      '#options' => $this->folderManager->getFolderOptions(),
      '#default_value' => $parent ?: 0,
      '#description' => $this->t('Select the parent folder (leave as Root for top-level folder).'),
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Create Folder'),
      '#button_type' => 'primary',
    ];

    $form['actions']['cancel'] = [
      '#type' => 'link',
      '#title' => $this->t('Cancel'),
      '#url' => \Drupal\Core\Url::fromRoute('dl.library'),
      '#attributes' => ['class' => ['button']],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $name = $form_state->getValue('name');
    $parent_id = $form_state->getValue('parent_select');
    $description = $form_state->getValue('description');

    $folder_id = $this->folderManager->createFolder($name, $parent_id, $description);

    $this->messenger()->addStatus($this->t('Folder "@name" has been created successfully.', [
      '@name' => $name,
    ]));

    // Redirect to folder view
    $form_state->setRedirect('dl.folder', ['folder' => $folder_id]);
  }

}
