# Folder Admin - Taxonomy Term Style Update

## Date: 2025-12-09

## Overview
Updated the folder admin interface to match Drupal's default taxonomy term overview behavior and the documents admin page look and feel.

## Key Changes

### 1. Form Structure (FolderAdminForm.php)

**Changed to match taxonomy term form style:**

- ✅ Removed extra wrapper containers
- ✅ Simplified table structure (4 columns: Name, Status, Operations, Weight)
- ✅ Used `#type => 'operations'` for action links (native Drupal)
- ✅ Used `#theme => 'indentation'` for hierarchy visualization
- ✅ Proper tabledrag integration with visible parent field
- ✅ Weight field updates on drag and drop

**Table Columns:**
1. **Name**: Folder name with indentation and expand/collapse button
2. **Status**: Published/Unpublished badge
3. **Operations**: Edit | Delete links (Drupal dropbutton style)
4. **Weight**: Sort order field (updates on drag)

### 2. Expand/Collapse Behavior

**Like taxonomy terms:**
- `+` button to expand
- `−` button to collapse
- Shows/hides child folders
- Maintains hierarchy structure
- All folders expanded by default

### 3. Drag and Drop with Weight Updates

**Key feature - Weight field auto-updates:**

```javascript
// When folder is dragged and dropped:
1. Tabledrag moves the row visually
2. JavaScript detects the change
3. Groups folders by parent ID
4. Updates weight values (0, 1, 2, 3...) within each parent group
5. Weight field value updated automatically
6. Form submission saves the new weight to database
```

**Result:** The `weight` column in `dl_folders` table is properly updated when you drag and drop folders.

### 4. Styling

**Matches documents admin page:**
- Same color scheme (green primary, clean borders)
- Consistent spacing and typography
- Same statistics card design
- Same button styles
- Responsive design maintained

### 5. Template Structure

**Clean and consistent:**
- Header with actions
- Statistics dashboard (4 cards)
- Table wrapper with form
- Matches `/admin/content/documents` exactly

## Files Modified

### 1. [FolderAdminForm.php](web/modules/custom/dl/src/Form/FolderAdminForm.php)
- Complete rewrite to match taxonomy term form structure
- Simplified from 7 columns to 4 columns
- Added proper indentation theming
- Changed operations to Drupal's native operations render element
- Weight field now visible and functional

### 2. [folder-admin.css](web/modules/custom/dl/css/folder-admin.css)
- Complete rewrite to match document admin styling
- Uses CSS variables from document-library.css
- Clean, modern table design
- Proper tabledrag handle styling
- Responsive design

### 3. [folder-admin.js](web/modules/custom/dl/js/folder-admin.js)
- Rewritten for taxonomy-style behavior
- Expand/collapse with `+` and `−` buttons
- **Auto-updates weight fields on drag and drop**
- Groups folders by parent for proper weight calculation
- Cleaner, more efficient code

### 4. [dl-admin-folders.html.twig](web/modules/custom/dl/templates/dl-admin-folders.html.twig)
- Updated icon (fa-times-circle → fa-eye-slash for unpublished)
- Better documentation
- Consistent with documents admin template

## How Weight Updates Work

### Database Flow:

1. **Initial Load:**
   - Folders loaded from `dl_folders` table ordered by `weight` ASC
   - Form displays folders in weight order

2. **Drag and Drop:**
   - User drags folder to new position
   - JavaScript `updateWeights()` function runs
   - Calculates new weight based on visible order
   - Updates weight select field value

3. **Save:**
   - Form submits with updated weight values
   - `submitForm()` extracts weight from form values
   - Calls `FolderManager::updateFolder()` with new weight
   - Database `dl_folders.weight` column updated

4. **Next Load:**
   - Folders display in new order based on updated weights

### Code Example:

```php
// In FolderAdminForm::submitForm()
foreach ($folders as $folder_id => $row) {
  $weight = isset($row['weight']) ? (int) $row['weight'] : 0;

  $this->folderManager->updateFolder($folder_id, [
    'parent_id' => $parent,
    'weight' => $weight,  // This updates dl_folders.weight column
  ]);
}
```

## Features

### ✅ Taxonomy-Style Behavior
- Hierarchical display with indentation
- Expand/collapse buttons (`+` / `−`)
- Native Drupal tabledrag
- Operations dropbutton

### ✅ Weight Field Management
- **Automatically updates on drag and drop**
- Grouped by parent (siblings have sequential weights)
- Saves to database on form submission
- Controls sort order in folder tree

### ✅ Matching Documents Admin
- Same header structure
- Same statistics cards
- Same button styles
- Same color scheme
- Consistent user experience

### ✅ Responsive Design
- Works on desktop, tablet, mobile
- Touch-friendly for mobile drag and drop
- Responsive statistics grid

## Testing Checklist

- [ ] Navigate to `/admin/content/documents/folders`
- [ ] Verify statistics display correctly
- [ ] Click `+` button - folder children show
- [ ] Click `−` button - folder children hide
- [ ] Drag folder up/down - weight field updates
- [ ] Drag folder to change parent - parent field updates
- [ ] Click "Save" - changes persist
- [ ] Reload page - folders in new order
- [ ] Check `dl_folders` table - weight column updated
- [ ] Click "Edit" link - opens edit form
- [ ] Click "Delete" link - opens delete form
- [ ] Test on mobile - responsive layout works

## Database Verification

To verify weight updates are working:

```sql
-- Check folder weights
SELECT folder_id, name, parent_id, weight, status
FROM dl_folders
ORDER BY parent_id, weight;

-- After dragging folders, weights should change
-- Example:
-- folder_id | name        | parent_id | weight | status
-- 1         | Folder A    | 0         | 0      | 1
-- 2         | Folder B    | 0         | 1      | 1
-- 3         | Subfolder 1 | 1         | 0      | 1
-- 4         | Subfolder 2 | 1         | 1      | 1
```

## Comparison

### Before (Old Style):
- 7 columns with hidden fields
- Custom expand/collapse with chevron icons
- Complex JavaScript
- Different styling from documents page
- Weight hidden in separate column

### After (Taxonomy Style):
- 4 clean columns
- Standard `+`/`−` buttons like taxonomy
- Simple, efficient JavaScript
- Matches documents admin exactly
- **Weight field auto-updates on drag**

## Access URL

```
/admin/content/documents/folders
```

Required permission: `administer document library`

## Cache Clear

Already cleared! Run again if needed:

```bash
ddev drush cr
```

---

**Status**: ✅ Complete
**Style**: Taxonomy term overview
**Weight Updates**: ✅ Automatic
**Look & Feel**: Documents admin page
**Cache**: ✅ Cleared
**Ready**: ✅ Ready for testing
