<?php

namespace Drupal\dl\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\dl\FolderManager;
use Drupal\file\Entity\File;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Controller for Document Library.
 */
class DocumentLibraryController extends ControllerBase
{

  protected $database;
  protected $fileUrlGenerator;
  protected $requestStack;
  protected $config;
  protected $folderManager;
  protected $adminFolderService;

  public function __construct(Connection $database, FileUrlGeneratorInterface $file_url_generator, RequestStack $request_stack, ConfigFactoryInterface $config_factory, FolderManager $folder_manager, \Drupal\dl\AdminFolderService $admin_folder_service)
  {
    $this->database = $database;
    $this->fileUrlGenerator = $file_url_generator;
    $this->requestStack = $request_stack;
    $this->config = $config_factory->get('dl.settings');
    $this->folderManager = $folder_manager;
    $this->adminFolderService = $admin_folder_service;
  }

  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('database'),
      $container->get('file_url_generator'),
      $container->get('request_stack'),
      $container->get('config.factory'),
      $container->get('dl.folder_manager'),
      $container->get('dl.admin_folder_service')
    );
  }

  /**
   * Main library page.
   */
  public function libraryPage()
  {
    $build = [];

    // Add CSS and JS
    $build['#attached']['library'][] = 'dl/document-library';

    // Get statistics
    $total_docs = $this->database->query('SELECT COUNT(*) FROM {dl_documents} WHERE status = 1')->fetchField();
    $total_downloads = $this->database->query('SELECT SUM(downloads) FROM {dl_documents}')->fetchField();
    $user_docs = $this->database->query('SELECT COUNT(*) FROM {dl_documents} WHERE uid = :uid AND status = 1', [':uid' => $this->currentUser()->id()])->fetchField();

    // Get recent documents
    $items_per_page = $this->config->get('items_per_page') ?: 20;
    $query = $this->database->select('dl_documents', 'd')
      ->fields('d')
      ->condition('status', 1)
      ->orderBy('created', 'DESC')
      ->range(0, $items_per_page);
    $documents = $query->execute()->fetchAll();

    // Get folder tree
    $folder_tree = $this->folderManager->buildFolderTree(0, TRUE);

    $build['#theme'] = 'dl_library_page';
    $build['#statistics'] = [
      'total' => $total_docs,
      'downloads' => $total_downloads ?: 0,
      'user_docs' => $user_docs,
    ];
    $build['#documents'] = $this->formatDocuments($documents);
    $build['#folder_tree'] = $folder_tree;
    $build['#can_upload'] = $this->currentUser()->hasPermission('upload documents');
    $build['#can_create_folder'] = $this->currentUser()->hasPermission('create folders');
    $build['#enable_favorites'] = $this->config->get('enable_favorites') ?? TRUE;

    return $build;
  }

  /**
   * View single document.
   */
  public function viewDocument($document)
  {
    $doc = $this->database->query('SELECT * FROM {dl_documents} WHERE id = :id', [':id' => $document])->fetchObject();

    if (!$doc || !$doc->status) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException();
    }

    // Get file
    $file = File::load($doc->file_id);
    if (!$file) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException();
    }

    // Get version history with user info
    $version_records = $this->database->query('SELECT * FROM {dl_versions} WHERE document_id = :id ORDER BY created DESC', [':id' => $document])->fetchAll();
    $versions = [];
    foreach ($version_records as $version) {
      $version_user = \Drupal\user\Entity\User::load($version->uid);
      $versions[] = [
        'version' => $version->version,
        'created' => $version->created,
        'notes' => $version->notes,
        'user_name' => $version_user ? $version_user->getDisplayName() : 'Unknown',
      ];
    }

    // Get owner
    $owner = \Drupal\user\Entity\User::load($doc->uid);

    // Check if favorited
    $is_favorite = $this->database->query('SELECT COUNT(*) FROM {dl_favorites} WHERE uid = :uid AND document_id = :did', [
      ':uid' => $this->currentUser()->id(),
      ':did' => $document,
    ])->fetchField();

    // Get folder and breadcrumbs
    $folder = NULL;
    $breadcrumbs = [];
    if ($doc->folder_id > 0) {
      $folder = $this->folderManager->getFolder($doc->folder_id);
      $breadcrumbs = $this->folderManager->getFolderBreadcrumbs($doc->folder_id);

      // Add slug_path to each breadcrumb
      foreach ($breadcrumbs as &$crumb) {
        $crumb->slug_path = $this->folderManager->getFolderSlugPath($crumb->folder_id);
      }

      // Add slug_path to folder
      if ($folder) {
        $folder->slug_path = $this->folderManager->getFolderSlugPath($doc->folder_id);
      }
    }

    $build = [];
    $build['#attached']['library'][] = 'dl/document-library';
    $build['#theme'] = 'dl_document_view';
    $build['#document'] = $doc;
    $build['#file'] = $file;
    $build['#file_url'] = $this->fileUrlGenerator->generateAbsoluteString($file->getFileUri());
    $build['#file_size'] = $this->formatBytes($file->getSize());
    $build['#owner'] = $owner;
    $build['#versions'] = $versions;
    $build['#is_favorite'] = $is_favorite;
    $build['#folder'] = $folder;
    $build['#breadcrumbs'] = $breadcrumbs;
    $build['#can_edit'] = $this->currentUser()->hasPermission('edit documents') &&
      ($doc->uid == $this->currentUser()->id() || $this->currentUser()->hasPermission('manage all documents'));
    $build['#can_download'] = $this->currentUser()->hasPermission('download documents');
    $build['#enable_favorites'] = $this->config->get('enable_favorites') ?? TRUE;
    $build['#enable_versioning'] = $this->config->get('enable_versioning') ?? TRUE;
    $build['#enable_downloads_tracking'] = $this->config->get('enable_downloads_tracking') ?? TRUE;
    $build['#csrf_token'] = \Drupal::csrfToken()->get('document-favorite');
    // Ensure cacheability metadata includes session for CSRF token
    $build['#cache']['contexts'][] = 'session';
    return $build;
  }

  /**
   * Download document.
   */
  public function downloadDocument($document)
  {
    $doc = $this->database->query('SELECT * FROM {dl_documents} WHERE id = :id', [':id' => $document])->fetchObject();

    if (!$doc || !$doc->status) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException();
    }

    $file = File::load($doc->file_id);
    if (!$file) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException();
    }

    // Track download
    $this->database->insert('dl_downloads')
      ->fields([
        'document_id' => $document,
        'uid' => $this->currentUser()->id(),
        'timestamp' => time(),
        'ip_address' => $this->requestStack->getCurrentRequest()->getClientIp(),
      ])
      ->execute();

    // Increment download count
    $this->database->update('dl_documents')
      ->fields(['downloads' => $doc->downloads + 1])
      ->condition('id', $document)
      ->execute();

    // Return file
    $uri = $file->getFileUri();
    $path = \Drupal::service('file_system')->realpath($uri);

    $response = new BinaryFileResponse($path);
    $response->setContentDisposition('attachment', $file->getFilename());

    return $response;
  }

  /**
   * Search page.
   */
  public function searchPage()
  {
    $request = $this->requestStack->getCurrentRequest();
    $keyword = $request->query->get('q', '');
    $sort = $request->query->get('sort', 'date');

    $documents = [];

    if ($keyword) {
      // Escape special LIKE characters
      $keyword_escaped = $this->database->escapeLike($keyword);

      $query = $this->database->select('dl_documents', 'd')
        ->fields('d')
        ->condition('status', 1);

      if ($keyword) {
        $or = $query->orConditionGroup()
          ->condition('title', '%' . $keyword_escaped . '%', 'LIKE')
          ->condition('description', '%' . $keyword_escaped . '%', 'LIKE')
          ->condition('tags', '%' . $keyword_escaped . '%', 'LIKE');
        $query->condition($or);
      }

      // Validate sort parameter
      $allowed_sorts = ['title', 'downloads', 'date'];
      if (!in_array($sort, $allowed_sorts)) {
        $sort = 'date';
      }

      switch ($sort) {
        case 'title':
          $query->orderBy('title', 'ASC');
          break;
        case 'downloads':
          $query->orderBy('downloads', 'DESC');
          break;
        default:
          $query->orderBy('created', 'DESC');
      }

      $documents = $query->execute()->fetchAll();
    }

    $build = [];
    $build['#attached']['library'][] = 'dl/document-library';
    $build['#theme'] = 'dl_search_page';
    $build['#keyword'] = $keyword;
    $build['#sort'] = $sort;
    $build['#documents'] = $this->formatDocuments($documents);

    return $build;
  }

  /**
   * Favorites page.
   */
  public function favoritesPage()
  {
    $query = $this->database->select('dl_favorites', 'f')
      ->fields('f')
      ->condition('uid', $this->currentUser()->id())
      ->orderBy('timestamp', 'DESC');

    $favorites = $query->execute()->fetchAll();

    $documents = [];
    foreach ($favorites as $fav) {
      $doc = $this->database->query('SELECT * FROM {dl_documents} WHERE id = :id AND status = 1', [':id' => $fav->document_id])->fetchObject();
      if ($doc) {
        $documents[] = $doc;
      }
    }

    $build = [];
    $build['#attached']['library'][] = 'dl/document-library';
    $build['#theme'] = 'dl_favorites_page';
    $build['#documents'] = $this->formatDocuments($documents);
    $build['#csrf_token'] = \Drupal::csrfToken()->get('document-favorite');

    return $build;
  }

  /**
   * Toggle favorite.
   */
  public function toggleFavorite($document)
  {
    $request = $this->requestStack->getCurrentRequest();
    $token = $request->headers->get('X-CSRF-Token');

    if (!\Drupal::csrfToken()->validate($token, 'document-favorite')) {
      \Drupal::logger('dl')->warning('CSRF validation failed for doc @id', ['@id' => $document]);
      return new JsonResponse(['error' => 'Invalid CSRF token'], 403);
    }

    $uid = $this->currentUser()->id();

    $exists = $this->database->query('SELECT COUNT(*) FROM {dl_favorites} WHERE uid = :uid AND document_id = :did', [
      ':uid' => $uid,
      ':did' => $document,
    ])->fetchField();

    if ($exists) {
      $this->database->delete('dl_favorites')
        ->condition('uid', $uid)
        ->condition('document_id', $document)
        ->execute();
      $favorited = FALSE;
    } else {
      $this->database->insert('dl_favorites')
        ->fields([
          'uid' => $uid,
          'document_id' => $document,
          'timestamp' => time(),
        ])
        ->execute();
      $favorited = TRUE;
    }

    return new JsonResponse(['favorited' => $favorited]);
  }

  /**
   * Document title callback.
   */
  public function documentTitle($document)
  {
    $doc = $this->database->query('SELECT title FROM {dl_documents} WHERE id = :id', [':id' => $document])->fetchObject();
    return $doc ? $doc->title : 'Document';
  }

  /**
   * Folder page - view documents in a specific folder.
   */
  public function folderPage($folder = 0)
  {
    $folder_obj = NULL;
    $breadcrumbs = [];

    // Get folder info if not root
    if ($folder > 0) {
      $folder_obj = $this->folderManager->getFolder($folder);
      if (!$folder_obj || !$folder_obj->status) {
        throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException();
      }
      $breadcrumbs = $this->folderManager->getFolderBreadcrumbs($folder);

      // Add slug_path to each breadcrumb
      foreach ($breadcrumbs as $crumb) {
        $crumb->slug_path = $this->folderManager->getFolderSlugPath($crumb->folder_id);
      }
    }

    // Get documents in this folder
    $query = $this->database->select('dl_documents', 'd')
      ->fields('d')
      ->condition('folder_id', $folder)
      ->condition('status', 1)
      ->orderBy('created', 'DESC');
    $documents = $query->execute()->fetchAll();

    // Get subfolders
    $subfolders = $this->folderManager->getFolderTree($folder, TRUE);

    // Get folder tree for sidebar
    $folder_tree = $this->folderManager->buildFolderTree(0, TRUE);

    // Add slug_path to folder object if it exists
    if ($folder_obj) {
      $folder_obj->slug_path = $this->folderManager->getFolderSlugPath($folder);
    }

    // Add slug_path to subfolders
    $subfolders_with_paths = [];
    foreach ($subfolders as $subfolder) {
      $subfolder_data = clone $subfolder;
      $subfolder_data->slug_path = $this->folderManager->getFolderSlugPath($subfolder->folder_id);
      $subfolders_with_paths[] = $subfolder_data;
    }
    $subfolders = $subfolders_with_paths;

    $build = [];
    $build['#attached']['library'][] = 'dl/document-library';
    $build['#theme'] = 'dl_folder_page';
    $build['#folder'] = $folder_obj;
    $build['#breadcrumbs'] = $breadcrumbs;
    $build['#documents'] = $this->formatDocuments($documents);
    $build['#subfolders'] = $subfolders;
    $build['#folder_tree'] = $folder_tree;
    $build['#can_upload'] = $this->currentUser()->hasPermission('upload documents');
    $build['#can_create_folder'] = $this->currentUser()->hasPermission('create folders');
    $build['#can_edit_folder'] = $folder_obj && (
      ($folder_obj->uid == $this->currentUser()->id() && $this->currentUser()->hasPermission('edit folders')) ||
      $this->currentUser()->hasPermission('manage all folders')
    );

    return $build;
  }

  /**
   * Folder title callback.
   */
  public function folderTitle($folder = 0)
  {
    if ($folder == 0) {
      return $this->t('Root Folder');
    }
    $folder_obj = $this->folderManager->getFolder($folder);
    return $folder_obj ? $folder_obj->name : $this->t('Folder');
  }

  /**
   * Folder page using slug-based path.
   */
  public function folderPageByPath($path)
  {
    $folder_obj = $this->folderManager->getFolderByPath($path);

    if (!$folder_obj) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException();
    }

    return $this->folderPage($folder_obj->folder_id);
  }

  /**
   * Folder title callback using slug-based path.
   */
  public function folderTitleByPath($path)
  {
    $folder_obj = $this->folderManager->getFolderByPath($path);
    return $folder_obj ? $folder_obj->name : $this->t('Folder');
  }

  /**
   * Folder page using path segments.
   */
  public function folderPageBySegments($segment1 = NULL, $segment2 = NULL, $segment3 = NULL)
  {
    // Build path from segments
    $segments = array_filter([$segment1, $segment2, $segment3]);
    $path = implode('/', $segments);

    return $this->folderPageByPath($path);
  }

  /**
   * Folder title callback using path segments.
   */
  public function folderTitleBySegments($segment1 = NULL, $segment2 = NULL, $segment3 = NULL)
  {
    // Build path from segments
    $segments = array_filter([$segment1, $segment2, $segment3]);
    $path = implode('/', $segments);

    return $this->folderTitleByPath($path);
  }

  /**
   * Format documents with additional data.
   */
  protected function formatDocuments($documents)
  {
    $formatted = [];

    foreach ($documents as $doc) {
      $file = File::load($doc->file_id);
      $owner = \Drupal\user\Entity\User::load($doc->uid);

      // Get folder slug path and name if document is in a folder
      $folder_slug_path = '';
      $folder_name = '';
      if ($doc->folder_id > 0) {
        $folder_slug_path = $this->folderManager->getFolderSlugPath($doc->folder_id);
        $folder = $this->folderManager->getFolder($doc->folder_id);
        if ($folder) {
          $folder_name = $folder->name;
        }
      }

      $formatted[] = [
        'id' => $doc->id,
        'title' => $doc->title,
        'description' => $doc->description,
        'file' => $file,
        'file_url' => $file ? $this->fileUrlGenerator->generateAbsoluteString($file->getFileUri()) : '',
        'file_size' => $file ? $this->formatBytes($file->getSize()) : '',
        'file_type' => $file ? $file->getMimeType() : '',
        'version' => $doc->version,
        'downloads' => $doc->downloads,
        'owner' => $owner,
        'created' => $doc->created,
        'changed' => $doc->changed,
        'folder_id' => $doc->folder_id,
        'folder_slug_path' => $folder_slug_path,
        'folder_name' => $folder_name,
        'status' => $doc->status,
      ];
    }

    return $formatted;
  }

  /**
   * Admin documents listing page.
   */
  public function adminDocumentsPage()
  {
    $build = [];

    // Add CSS and JS
    $build['#attached']['library'][] = 'dl/document-library';

    // Get all documents (including unpublished)
    $query = $this->database->select('dl_documents', 'd')
      ->fields('d')
      ->orderBy('created', 'DESC');
    $documents = $query->execute()->fetchAll();

    // Get folder tree (including unpublished)
    $folder_tree = $this->folderManager->buildFolderTree(0, FALSE);

    // Get statistics
    $total_docs = count($documents);
    $published_docs = $this->database->query('SELECT COUNT(*) FROM {dl_documents} WHERE status = 1')->fetchField();
    $total_downloads = $this->database->query('SELECT SUM(downloads) FROM {dl_documents}')->fetchField();

    $build['#theme'] = 'dl_admin_documents';
    $build['#documents'] = $this->formatDocuments($documents);
    $build['#folder_tree'] = $folder_tree;
    $build['#statistics'] = [
      'total' => $total_docs,
      'published' => $published_docs,
      'unpublished' => $total_docs - $published_docs,
      'downloads' => $total_downloads ?: 0,
    ];
    $build['#can_upload'] = $this->currentUser()->hasPermission('upload documents');
    $build['#can_create_folder'] = $this->currentUser()->hasPermission('create folders');

    return $build;
  }

  /**
   * Admin folders listing page.
   */
  public function adminFoldersPage() {
    // Get statistics from service
    $statistics = $this->adminFolderService->getFolderStatistics();

    // Get the folder admin form
    $form = $this->formBuilder()->getForm(\Drupal\dl\Form\FolderAdminForm::class);

    $build = [];
    $build['#attached']['library'][] = 'dl/document-library';
    $build['#theme'] = 'dl_admin_folders';
    $build['#statistics'] = $statistics;
    $build['#folders_form'] = $form;

    return $build;
  }

  /**
   * Format bytes to human readable format.
   */
  protected function formatBytes($bytes, $precision = 2)
  {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);
    $bytes /= (1 << (10 * $pow));
    return round($bytes, $precision) . ' ' . $units[$pow];
  }
}
