<?php

namespace Drupal\dl\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\dl\FolderManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for deleting folders.
 */
class FolderDeleteForm extends ConfirmFormBase {

  /**
   * The folder manager service.
   *
   * @var \Drupal\dl\FolderManager
   */
  protected $folderManager;

  /**
   * The folder ID.
   *
   * @var int
   */
  protected $folderId;

  /**
   * The folder object.
   *
   * @var object
   */
  protected $folder;

  /**
   * Constructs a FolderDeleteForm object.
   *
   * @param \Drupal\dl\FolderManager $folder_manager
   *   The folder manager service.
   */
  public function __construct(FolderManager $folder_manager) {
    $this->folderManager = $folder_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('dl.folder_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dl_folder_delete_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $folder = NULL) {
    $this->folderId = $folder;
    $this->folder = $this->folderManager->getFolder($folder);

    if (!$this->folder) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException();
    }

    // Check permissions
    if ($this->folder->uid != $this->currentUser()->id() &&
        !$this->currentUser()->hasPermission('manage all folders')) {
      throw new \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException();
    }

    $form = parent::buildForm($form, $form_state);

    $doc_count = $this->folderManager->getFolderDocumentCount($folder, FALSE);
    $children = $this->folderManager->getFolderTree($folder, FALSE);

    if ($doc_count > 0 || !empty($children)) {
      $form['move_contents'] = [
        '#type' => 'radios',
        '#title' => $this->t('What to do with folder contents?'),
        '#options' => [
          1 => $this->t('Move to parent folder (@count documents, @subfolders subfolders)', [
            '@count' => $doc_count,
            '@subfolders' => count($children),
          ]),
          0 => $this->t('Delete all contents (Warning: This cannot be undone!)'),
        ],
        '#default_value' => 1,
        '#required' => TRUE,
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to delete the folder "@name"?', [
      '@name' => $this->folder->name,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('dl.folder', ['folder' => $this->folderId]);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $move_contents = $form_state->getValue('move_contents', TRUE);

    $this->folderManager->deleteFolder($this->folderId, $move_contents);

    $this->messenger()->addStatus($this->t('Folder "@name" has been deleted.', [
      '@name' => $this->folder->name,
    ]));

    $form_state->setRedirect('dl.library');
  }

}
