<?php

namespace Drupal\dl\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\dl\FolderManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for editing folders.
 */
class FolderEditForm extends FormBase {

  /**
   * The folder manager service.
   *
   * @var \Drupal\dl\FolderManager
   */
  protected $folderManager;

  /**
   * Constructs a FolderEditForm object.
   *
   * @param \Drupal\dl\FolderManager $folder_manager
   *   The folder manager service.
   */
  public function __construct(FolderManager $folder_manager) {
    $this->folderManager = $folder_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('dl.folder_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dl_folder_edit_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $folder = NULL) {
    $folder_obj = $this->folderManager->getFolder($folder);

    if (!$folder_obj) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException();
    }

    // Check permissions
    if ($folder_obj->uid != $this->currentUser()->id() &&
        !$this->currentUser()->hasPermission('manage all folders')) {
      throw new \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException();
    }

    $form['#attributes']['class'][] = 'dl-folder-form';
    $form['#attached']['library'][] = 'dl/document-library';
    $form['#prefix'] = '<div class="dl-form-page">';
    $form['#suffix'] = '</div>';

    $form['intro'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['dl-form-hero']],
      'content' => [
        '#markup' => '<div class="dl-form-hero__content">
          <span class="dl-form-hero__icon"><i class="fas fa-folder-cog"></i></span>
          <div>
            <p class="dl-form-hero__eyebrow">' . $this->t('Document Library') . '</p>
            <h1 class="dl-form-hero__title">' . $this->t('Edit folder') . '</h1>
            <p class="dl-form-hero__lede">' . $this->t('Update the folder details, parent, and sort order to keep your hierarchy organized.') . '</p>
          </div>
        </div>',
      ],
      'actions' => [
        '#type' => 'container',
        '#attributes' => ['class' => ['dl-form-hero__actions']],
        'back' => [
          '#type' => 'link',
          '#title' => $this->t('Back to Folder'),
          '#url' => Url::fromRoute('dl.folder', ['folder' => $folder]),
          '#attributes' => ['class' => ['dl-btn', 'dl-btn-secondary']],
        ],
      ],
    ];

    $form['card'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['dl-form-card']],
    ];

    $form['card']['folder_id'] = [
      '#type' => 'value',
      '#value' => $folder,
    ];

    $form['card']['name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Folder Name'),
      '#required' => TRUE,
      '#maxlength' => 255,
      '#default_value' => $folder_obj->name,
    ];

    $form['card']['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description'),
      '#rows' => 3,
      '#default_value' => $folder_obj->description,
    ];

    $form['card']['parent_select'] = [
      '#type' => 'select',
      '#title' => $this->t('Parent Folder'),
      '#options' => $this->folderManager->getFolderOptions($folder, FALSE),
      '#default_value' => $folder_obj->parent_id,
      '#description' => $this->t('Select the parent folder. Cannot move folder into itself or its children.'),
    ];

    $form['card']['weight'] = [
      '#type' => 'number',
      '#title' => $this->t('Sort Order'),
      '#default_value' => $folder_obj->weight,
      '#description' => $this->t('Lower numbers appear first.'),
    ];

    $form['card']['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Published'),
      '#default_value' => $folder_obj->status,
    ];

    $form['card']['actions'] = [
      '#type' => 'actions',
    ];

    $form['card']['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Update Folder'),
      '#button_type' => 'primary',
      '#attributes' => ['class' => ['dl-btn', 'dl-btn-primary']],
    ];

    $form['card']['actions']['cancel'] = [
      '#type' => 'link',
      '#title' => $this->t('Cancel'),
      '#url' => \Drupal\Core\Url::fromRoute('dl.folder', ['folder' => $folder]),
      '#attributes' => ['class' => ['dl-btn', 'dl-btn-secondary']],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $folder_id = $form_state->getValue('folder_id');
    $name = $form_state->getValue('name');

    $this->folderManager->updateFolder($folder_id, [
      'name' => $name,
      'description' => $form_state->getValue('description'),
      'parent_id' => $form_state->getValue('parent_select'),
      'weight' => $form_state->getValue('weight'),
      'status' => $form_state->getValue('status') ? 1 : 0,
    ]);

    $this->messenger()->addStatus($this->t('Folder "@name" has been updated successfully.', [
      '@name' => $name,
    ]));

    // Redirect to the updated folder using slug path
    $slug_path = $this->folderManager->getFolderSlugPath($folder_id);
    $url = \Drupal\Core\Url::fromUri('internal:/documents/' . $slug_path);
    $form_state->setRedirectUrl($url);
  }

}
