<?php

namespace Drupal\dl\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\user\EntityOwnerTrait;

/**
 * Defines the Document entity.
 *
 * @ContentEntityType(
 *   id = "document",
 *   label = @Translation("Document"),
 *   label_collection = @Translation("Documents"),
 *   label_singular = @Translation("document"),
 *   label_plural = @Translation("documents"),
 *   label_count = @PluralTranslation(
 *     singular = "@count document",
 *     plural = "@count documents",
 *   ),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\dl\DocumentListBuilder",
 *     "views_data" = "Drupal\views\EntityViewsData",
 *     "form" = {
 *       "default" = "Drupal\dl\Form\DocumentForm",
 *       "add" = "Drupal\dl\Form\DocumentForm",
 *       "edit" = "Drupal\dl\Form\DocumentForm",
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm",
 *     },
 *     "access" = "Drupal\dl\DocumentAccessControlHandler",
 *     "route_provider" = {
 *       "html" = "Drupal\dl\Routing\DocumentHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "document",
 *   translatable = FALSE,
 *   revisionable = FALSE,
 *   admin_permission = "administer document library",
 *   entity_keys = {
 *     "id" = "id",
 *     "uuid" = "uuid",
 *     "label" = "title",
 *     "owner" = "uid",
 *     "published" = "status",
 *   },
 *   links = {
 *     "canonical" = "/documents/{document}",
 *     "add-form" = "/documents/add",
 *     "edit-form" = "/documents/{document}/edit",
 *     "delete-form" = "/documents/{document}/delete",
 *     "collection" = "/admin/content/documents",
 *   },
 *   field_ui_base_route = "entity.document.settings",
 * )
 */
class Document extends ContentEntityBase implements DocumentInterface {

  use EntityChangedTrait;
  use EntityOwnerTrait;

  /**
   * {@inheritdoc}
   */
  public function getTitle() {
    return $this->get('title')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setTitle($title) {
    $this->set('title', $title);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime() {
    return $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setCreatedTime($timestamp) {
    $this->set('created', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function isPublished() {
    return (bool) $this->get('status')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setPublished($published = NULL) {
    if ($published !== NULL) {
      @trigger_error('The $published parameter is deprecated since Drupal 9.3.0 and will be removed in Drupal 10.0.0. Use setPublished() or setUnpublished() instead.', E_USER_DEPRECATED);
      $this->set('status', $published ? 1 : 0);
      return $this;
    }
    $this->set('status', 1);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setUnpublished() {
    $this->set('status', 0);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getDownloadCount() {
    return (int) $this->get('downloads')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function incrementDownloadCount() {
    $this->set('downloads', $this->getDownloadCount() + 1);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage) {
    parent::preSave($storage);

    // Set created time if new entity.
    if ($this->isNew()) {
      $this->setCreatedTime(\Drupal::time()->getRequestTime());
    }

    // Set changed time.
    $this->setChangedTime(\Drupal::time()->getRequestTime());

    // Ensure owner is set.
    if (!$this->getOwnerId()) {
      $this->setOwnerId(\Drupal::currentUser()->id());
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function postDelete(EntityStorageInterface $storage, array $entities) {
    parent::postDelete($storage, $entities);

    foreach ($entities as $entity) {
      /** @var \Drupal\dl\Entity\Document $entity */
      // Delete the physical file when document is deleted
      if ($entity->hasField('field_file') && !$entity->get('field_file')->isEmpty()) {
        $file = $entity->get('field_file')->entity;
        if ($file) {
          // Delete the file entity (which will also delete the physical file)
          $file->delete();
        }
      }

      // Clean up related data from custom tables
      $database = \Drupal::database();
      $document_id = $entity->id();

      // Delete version history
      $database->delete('dl_versions')
        ->condition('document_id', $document_id)
        ->execute();

      // Delete download history
      $database->delete('dl_downloads')
        ->condition('document_id', $document_id)
        ->execute();

      // Delete favorites
      $database->delete('dl_favorites')
        ->condition('document_id', $document_id)
        ->execute();
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    // Add owner trait fields.
    $fields += static::ownerBaseFieldDefinitions($entity_type);

    $fields['title'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Document Title'))
      ->setDescription(t('The title of the document.'))
      ->setRequired(TRUE)
      ->setSetting('max_length', 255)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'string',
        'weight' => -5,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    // Note: description, field_file, field_version, and field_tags are now
    // configurable fields defined in config/install/ YAML files.
    // They will appear in the "Manage fields" UI and can be edited there.

    $fields['folder_id'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Folder'))
      ->setDescription(t('The folder ID this document belongs to (references dl_folders table). 0 = root folder.'))
      ->setDefaultValue(0)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'number_integer',
        'weight' => 10,
      ])
      ->setDisplayOptions('form', [
        'type' => 'number',
        'weight' => 10,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['downloads'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Download Count'))
      ->setDescription(t('The number of times this document has been downloaded.'))
      ->setDefaultValue(0)
      ->setReadOnly(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'number_integer',
        'weight' => 25,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Published'))
      ->setDescription(t('Whether the document is published.'))
      ->setDefaultValue(TRUE)
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'weight' => 30,
        'settings' => [
          'display_label' => TRUE,
        ],
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time the document was created.'))
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'timestamp',
        'weight' => 35,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time the document was last edited.'))
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'timestamp',
        'weight' => 40,
      ])
      ->setDisplayConfigurable('view', TRUE);

    return $fields;
  }

}
