<?php

namespace Drupal\dl\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\dl\FolderManager;
use Drupal\dl\AdminFolderService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Admin form to manage folder hierarchy and ordering.
 */
class FolderAdminForm extends FormBase {

  /**
   * The folder manager service.
   *
   * @var \Drupal\dl\FolderManager
   */
  protected $folderManager;

  /**
   * Database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The admin folder service.
   *
   * @var \Drupal\dl\AdminFolderService
   */
  protected $adminFolderService;

  /**
   * Constructs the form.
   */
  public function __construct(FolderManager $folder_manager, \Drupal\Core\Database\Connection $database, AdminFolderService $admin_folder_service) {
    $this->folderManager = $folder_manager;
    $this->database = $database;
    $this->adminFolderService = $admin_folder_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('dl.folder_manager'),
      $container->get('database'),
      $container->get('dl.admin_folder_service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dl_folder_admin_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['#attached']['library'][] = 'dl/document-library';
    $form['#attached']['library'][] = 'dl/folder-admin';
    $form['#attached']['library'][] = 'core/drupal.tabledrag';
    $form['#tree'] = TRUE;

    $rows = $this->adminFolderService->getAllFoldersWithMetadata();

    $form['folders'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Name'),
        $this->t('Owner'),
        $this->t('Documents'),
        $this->t('Status'),
        $this->t('Operations'),
        $this->t('Weight'),
      ],
      '#empty' => $this->t('No folders available.'),
      '#tabledrag' => [
        [
          'action' => 'match',
          'relationship' => 'parent',
          'group' => 'folder-parent',
          'subgroup' => 'folder-parent',
          'source' => 'folder-id',
          'hidden' => FALSE,
        ],
        [
          'action' => 'order',
          'relationship' => 'sibling',
          'group' => 'folder-weight',
        ],
      ],
    ];

    foreach ($rows as $row) {
      $folder_id = $row['folder_id'];
      $depth = $row['depth'];

      $form['folders'][$folder_id]['#attributes']['class'][] = 'draggable';
      $form['folders'][$folder_id]['#weight'] = $row['weight'];

      // Indentation for hierarchy
      $indentation = [];
      for ($i = 1; $i < $depth; $i++) {
        $indentation[] = [
          '#theme' => 'indentation',
          '#size' => 1,
        ];
      }

      // Name column with indentation (no expand/collapse button)
      $name_cell = [];
      if (!empty($indentation)) {
        $name_cell['indentation'] = $indentation;
      }
      $name_cell['name'] = [
        '#markup' => '<span class="folder-name">' . htmlspecialchars($row['name']) . '</span>',
      ];
      $name_cell['#prefix'] = '<div class="folder-name-cell">';
      $name_cell['#suffix'] = '</div>';

      $form['folders'][$folder_id]['name'] = $name_cell;

      // Owner column
      $form['folders'][$folder_id]['owner'] = [
        '#markup' => htmlspecialchars($row['owner_name']),
      ];

      // Documents count column
      $doc_count = $row['document_count'];
      $form['folders'][$folder_id]['documents'] = [
        '#markup' => '<span class="dl-doc-count-value">' . $doc_count . '</span>',
      ];

      // Status column
      if ($row['status']) {
        $status_badge = '<span class="dl-badge dl-badge-success">' . $this->t('Published') . '</span>';
      }
      else {
        $status_badge = '<span class="dl-badge dl-badge-warning">' . $this->t('Unpublished') . '</span>';
      }

      $form['folders'][$folder_id]['status'] = [
        '#markup' => $status_badge,
      ];

      // Operations column with icon-based actions (matching documents admin)
      $edit_url = Url::fromRoute('dl.folder.edit', ['folder' => $folder_id])->toString();
      $delete_url = Url::fromRoute('dl.folder.delete', ['folder' => $folder_id])->toString();

      $actions_html = '<div class="dl-action-buttons">';
      $actions_html .= '<a href="' . $edit_url . '" class="dl-action-btn" title="' . $this->t('Edit') . '">&#9998;</a>';
      $actions_html .= '<a href="' . $delete_url . '" class="dl-action-btn dl-action-delete" title="' . $this->t('Delete') . '">&#128465;</a>';
      $actions_html .= '</div>';

      $form['folders'][$folder_id]['operations'] = [
        '#markup' => $actions_html,
      ];

      // Weight field (visible for tabledrag)
      $form['folders'][$folder_id]['weight'] = [
        '#type' => 'weight',
        '#title' => $this->t('Weight for @title', ['@title' => $row['name']]),
        '#title_display' => 'invisible',
        '#default_value' => $row['weight'],
        '#attributes' => ['class' => ['folder-weight']],
        '#delta' => 50,
      ];

      // Hidden parent ID field
      $form['folders'][$folder_id]['parent'] = [
        '#type' => 'hidden',
        '#value' => $row['parent_id'],
        '#attributes' => ['class' => ['folder-parent']],
      ];

      // Hidden folder ID field
      $form['folders'][$folder_id]['folder_id'] = [
        '#type' => 'hidden',
        '#value' => $folder_id,
        '#attributes' => ['class' => ['folder-id']],
      ];

      // Hidden depth field
      $form['folders'][$folder_id]['depth'] = [
        '#type' => 'hidden',
        '#value' => $depth,
        '#attributes' => ['class' => ['folder-depth']],
      ];
    }

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
      '#button_type' => 'primary',
      '#attributes' => ['class' => ['dl-btn', 'dl-btn-primary']],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $folders = $form_state->getValue('folders') ?? [];

    if (empty($folders)) {
      $this->messenger()->addWarning($this->t('No folder data to save.'));
      return;
    }

    $updated_count = 0;
    foreach ($folders as $folder_id => $row) {
      if (!is_numeric($folder_id)) {
        continue;
      }

      // Get parent and weight from the row
      $parent = isset($row['parent']) ? (int) $row['parent'] : 0;
      $weight = isset($row['weight']) ? (int) $row['weight'] : 0;

      // Update folder with new parent and weight (sort order)
      $this->folderManager->updateFolder($folder_id, [
        'parent_id' => $parent,
        'weight' => $weight,
      ]);

      $updated_count++;
    }

    if ($updated_count > 0) {
      $this->messenger()->addStatus($this->t('The configuration options have been saved.'));
    }
  }

}
