<?php

namespace Drupal\dl\Routing;

use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\Routing\AdminHtmlRouteProvider;
use Symfony\Component\Routing\Route;

/**
 * Provides routes for Document entities.
 */
class DocumentHtmlRouteProvider extends AdminHtmlRouteProvider {

  /**
   * {@inheritdoc}
   */
  public function getRoutes(EntityTypeInterface $entity_type) {
    $collection = parent::getRoutes($entity_type);

    $entity_type_id = $entity_type->id();

    // Override canonical route to use custom controller with version history, favorites, etc.
    if ($canonical_route = $this->getCanonicalRoute($entity_type)) {
      $collection->add("entity.{$entity_type_id}.canonical", $canonical_route);
    }

    // Add settings form route - this is CRITICAL for Field UI!
    if ($settings_form_route = $this->getSettingsFormRoute($entity_type)) {
      $collection->add("entity.{$entity_type_id}.settings", $settings_form_route);
    }

    return $collection;
  }

  /**
   * Gets the canonical route with custom controller.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   *
   * @return \Symfony\Component\Routing\Route|null
   *   The generated route, if available.
   */
  protected function getCanonicalRoute(EntityTypeInterface $entity_type) {
    $entity_type_id = $entity_type->id();
    $route = new Route("/documents/{{$entity_type_id}}");
    $route
      ->setDefaults([
        '_controller' => '\Drupal\dl\Controller\DocumentLibraryController::viewDocument',
        '_title_callback' => '\Drupal\dl\Controller\DocumentLibraryController::documentTitle',
      ])
      ->setRequirement('_permission', 'access document library')
      ->setRequirement($entity_type_id, '\d+')
      ->setOption('parameters', [
        $entity_type_id => ['type' => 'entity:' . $entity_type_id],
      ]);

    return $route;
  }

  /**
   * Gets the settings form route.
   *
   * This route is REQUIRED for field_ui_base_route to work.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   *
   * @return \Symfony\Component\Routing\Route|null
   *   The generated route, if available.
   */
  protected function getSettingsFormRoute(EntityTypeInterface $entity_type) {
    $entity_type_id = $entity_type->id();

    $route = new Route("/admin/structure/{$entity_type_id}/settings");
    $route
      ->setDefaults([
        '_form' => 'Drupal\dl\Form\DocumentSettingsForm',
        '_title' => 'Document settings',
      ])
      ->setRequirement('_permission', $entity_type->getAdminPermission())
      ->setOption('_admin_route', TRUE);

    return $route;
  }

  /**
   * {@inheritdoc}
   */
  protected function getCollectionRoute(EntityTypeInterface $entity_type) {
    $route = parent::getCollectionRoute($entity_type);

    if ($route) {
      $route->setDefault('_title', 'Documents');
    }

    return $route;
  }

  /**
   * {@inheritdoc}
   */
  protected function getAddFormRoute(EntityTypeInterface $entity_type) {
    $route = parent::getAddFormRoute($entity_type);

    if ($route) {
      // Allow adding from different contexts.
      $route->setOption('parameters', [
        'folder' => [
          'type' => 'entity:taxonomy_term',
        ],
      ]);
    }

    return $route;
  }

}
