/**
 * @file
 * Folder admin behaviors - taxonomy term style.
 */

(function ($, Drupal) {
  'use strict';

  /**
   * Behavior for folder expand/collapse (like taxonomy terms).
   */
  Drupal.behaviors.folderAdminTree = {
    attach: function (context, settings) {
      var $toggles = $('.folder-toggle', context).once('folder-toggle');

      if ($toggles.length === 0) {
        return;
      }

      // Attach click handler to toggle buttons
      $toggles.on('click', function (e) {
        e.preventDefault();
        e.stopPropagation();

        var $button = $(this);
        var folderId = $button.data('folder-id');
        var $row = $button.closest('tr');
        var isExpanded = $button.hasClass('expanded');

        if (isExpanded) {
          // Collapse
          collapseFolder($row, folderId);
          $button.removeClass('expanded');
          $button.find('.folder-toggle-icon').text('+');
          $button.attr('title', Drupal.t('Show child folders'));
        } else {
          // Expand
          expandFolder($row, folderId);
          $button.addClass('expanded');
          $button.find('.folder-toggle-icon').text('−');
          $button.attr('title', Drupal.t('Hide child folders'));
        }
      });

      /**
       * Expand a folder to show its direct children.
       */
      function expandFolder($row, folderId) {
        var $nextRow = $row.next('tr');
        var rowDepth = parseInt($row.find('.folder-depth').val(), 10);

        while ($nextRow.length > 0) {
          var nextDepth = parseInt($nextRow.find('.folder-depth').val(), 10);

          // Stop if we've reached a sibling or parent
          if (nextDepth <= rowDepth) {
            break;
          }

          // Show only direct children (depth + 1)
          if (nextDepth === rowDepth + 1) {
            $nextRow.removeClass('folder-hidden').show();
          }

          $nextRow = $nextRow.next('tr');
        }
      }

      /**
       * Collapse a folder to hide all its descendants.
       */
      function collapseFolder($row, folderId) {
        var $nextRow = $row.next('tr');
        var rowDepth = parseInt($row.find('.folder-depth').val(), 10);

        while ($nextRow.length > 0) {
          var nextDepth = parseInt($nextRow.find('.folder-depth').val(), 10);

          // Stop if we've reached a sibling or parent
          if (nextDepth <= rowDepth) {
            break;
          }

          // Hide all descendants
          $nextRow.addClass('folder-hidden').hide();

          // Also collapse any expanded children
          var $toggle = $nextRow.find('.folder-toggle');
          if ($toggle.hasClass('expanded')) {
            $toggle.removeClass('expanded');
            $toggle.find('.folder-toggle-icon').text('+');
            $toggle.attr('title', Drupal.t('Show child folders'));
          }

          $nextRow = $nextRow.next('tr');
        }
      }

      // Initialize all folders as expanded by default
      $toggles.each(function() {
        var $button = $(this);
        var $row = $button.closest('tr');
        var folderId = $button.data('folder-id');

        expandFolder($row, folderId);
        $button.addClass('expanded');
        $button.find('.folder-toggle-icon').text('−');
        $button.attr('title', Drupal.t('Hide child folders'));
      });
    }
  };

  /**
   * Behavior to handle tabledrag weight changes.
   */
  Drupal.behaviors.folderAdminTableDrag = {
    attach: function (context, settings) {
      var $table = $('#edit-folders', context).once('folder-tabledrag');

      if ($table.length === 0) {
        return;
      }

      // Wait for tabledrag to initialize
      setTimeout(function() {
        if (Drupal.tableDrag && Drupal.tableDrag['edit-folders']) {
          var tableDrag = Drupal.tableDrag['edit-folders'];
          var originalOnDrop = tableDrag.onDrop;

          // Override onDrop to update weight fields
          tableDrag.onDrop = function() {
            // Call original onDrop
            if (originalOnDrop) {
              originalOnDrop.apply(this, arguments);
            }

            // Update all weight fields based on new order
            updateWeights($table);
          };

          // Also handle row swap
          var originalRowSwap = tableDrag.row.prototype.swap;
          tableDrag.row.prototype.swap = function(position, row) {
            originalRowSwap.apply(this, arguments);
            updateWeights($table);
          };
        }
      }, 100);

      /**
       * Update weight values after drag and drop.
       */
      function updateWeights($table) {
        var $rows = $table.find('tbody tr.draggable:visible');

        // Group rows by parent
        var parentGroups = {};

        $rows.each(function() {
          var $row = $(this);
          var parentId = $row.find('.folder-parent').val() || '0';

          if (!parentGroups[parentId]) {
            parentGroups[parentId] = [];
          }
          parentGroups[parentId].push($row);
        });

        // Update weights within each parent group
        $.each(parentGroups, function(parentId, rows) {
          $.each(rows, function(index, $row) {
            var $weightField = $row.find('.folder-weight');
            if ($weightField.length) {
              $weightField.val(index);
              // Trigger change event
              $weightField.trigger('change');
            }
          });
        });
      }
    }
  };

  /**
   * Behavior for save button feedback.
   */
  Drupal.behaviors.folderAdminActions = {
    attach: function (context, settings) {
      // Enhanced save button feedback
      $('form#dl-folder-admin-form .dl-btn-primary', context).once('save-feedback').on('click', function() {
        var $btn = $(this);
        $btn.prop('disabled', true);
        $btn.addClass('is-loading');

        // Re-enable after form submission (failsafe)
        setTimeout(function() {
          $btn.prop('disabled', false);
          $btn.removeClass('is-loading');
        }, 5000);
      });
    }
  };

})(jQuery, Drupal);
