<?php

namespace Drupal\dl;

use Drupal\Core\Database\Connection;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\user\Entity\User;

/**
 * Service for handling admin folder operations.
 */
class AdminFolderService {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The folder manager service.
   *
   * @var \Drupal\dl\FolderManager
   */
  protected $folderManager;

  /**
   * Constructs an AdminFolderService object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   * @param \Drupal\dl\FolderManager $folder_manager
   *   The folder manager service.
   */
  public function __construct(Connection $database, AccountProxyInterface $current_user, FolderManager $folder_manager) {
    $this->database = $database;
    $this->currentUser = $current_user;
    $this->folderManager = $folder_manager;
  }

  /**
   * Get folder statistics for admin dashboard.
   *
   * @return array
   *   Array containing folder statistics.
   */
  public function getFolderStatistics() {
    $total_folders = $this->database->select('dl_folders', 'f')
      ->countQuery()
      ->execute()
      ->fetchField();

    $published_folders = $this->database->select('dl_folders', 'f')
      ->condition('status', 1)
      ->countQuery()
      ->execute()
      ->fetchField();

    $root_folders = $this->database->select('dl_folders', 'f')
      ->condition('parent_id', 0)
      ->countQuery()
      ->execute()
      ->fetchField();

    return [
      'total' => (int) $total_folders,
      'published' => (int) $published_folders,
      'unpublished' => (int) ($total_folders - $published_folders),
      'roots' => (int) $root_folders,
    ];
  }

  /**
   * Get all folders with metadata for admin display.
   *
   * @return array
   *   Array of folders with additional metadata.
   */
  public function getAllFoldersWithMetadata() {
    $tree = $this->folderManager->buildFolderTree(0, FALSE);
    $folders = [];
    $this->flattenTreeWithMetadata($tree, $folders);
    return $folders;
  }

  /**
   * Flatten folder tree with metadata for admin table.
   *
   * @param array $tree
   *   Nested folder array.
   * @param array $folders
   *   Accumulated folders (by reference).
   * @param int $depth
   *   Current depth.
   */
  protected function flattenTreeWithMetadata(array $tree, array &$folders, $depth = 1) {
    foreach ($tree as $folder) {
      $owner = User::load($folder['uid'] ?? 0);

      $folders[] = [
        'folder_id' => $folder['folder_id'],
        'name' => $folder['name'],
        'slug' => $folder['slug'] ?? '',
        'parent_id' => $folder['parent_id'],
        'weight' => $folder['weight'],
        'depth' => $depth,
        'status' => $folder['status'] ?? 1,
        'document_count' => $folder['document_count'] ?? 0,
        'owner_name' => $owner ? $owner->getDisplayName() : 'Unknown',
        'created' => $folder['created'] ?? 0,
        'changed' => $folder['changed'] ?? 0,
        'has_children' => !empty($folder['children']),
      ];

      if (!empty($folder['children'])) {
        $this->flattenTreeWithMetadata($folder['children'], $folders, $depth + 1);
      }
    }
  }

  /**
   * Update folder hierarchy from drag-and-drop.
   *
   * @param array $folder_data
   *   Array of folder data with new parent and weight values.
   *
   * @return bool
   *   TRUE on success.
   */
  public function updateFolderHierarchy(array $folder_data) {
    foreach ($folder_data as $folder_id => $data) {
      $parent = isset($data['parent']) ? (int) $data['parent'] : 0;
      $weight = isset($data['weight']) ? (int) $data['weight'] : 0;

      $this->folderManager->updateFolder($folder_id, [
        'parent_id' => $parent,
        'weight' => $weight,
      ]);
    }

    return TRUE;
  }

  /**
   * Get folder info for admin display.
   *
   * @param int $folder_id
   *   The folder ID.
   *
   * @return array|null
   *   Folder info array or NULL if not found.
   */
  public function getFolderInfo($folder_id) {
    $folder = $this->folderManager->getFolder($folder_id);

    if (!$folder) {
      return NULL;
    }

    $owner = User::load($folder->uid);
    $document_count = $this->folderManager->getFolderDocumentCount($folder_id, FALSE);

    return [
      'folder_id' => $folder->folder_id,
      'name' => $folder->name,
      'slug' => $folder->slug,
      'description' => $folder->description ?? '',
      'parent_id' => $folder->parent_id,
      'weight' => $folder->weight,
      'depth' => $folder->depth,
      'status' => $folder->status,
      'document_count' => $document_count,
      'owner_name' => $owner ? $owner->getDisplayName() : 'Unknown',
      'created' => $folder->created,
      'changed' => $folder->changed,
    ];
  }

}
