<?php

namespace Drupal\dl\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\dl\FolderManager;
use Drupal\file\Entity\File;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Controller for Document Library.
 */
class DocumentLibraryController extends ControllerBase
{

  protected $database;
  protected $fileUrlGenerator;
  protected $requestStack;
  protected $config;
  protected $folderManager;
  protected $adminFolderService;

  public function __construct(Connection $database, FileUrlGeneratorInterface $file_url_generator, RequestStack $request_stack, ConfigFactoryInterface $config_factory, FolderManager $folder_manager, \Drupal\dl\AdminFolderService $admin_folder_service)
  {
    $this->database = $database;
    $this->fileUrlGenerator = $file_url_generator;
    $this->requestStack = $request_stack;
    $this->config = $config_factory->get('dl.settings');
    $this->folderManager = $folder_manager;
    $this->adminFolderService = $admin_folder_service;
  }

  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('database'),
      $container->get('file_url_generator'),
      $container->get('request_stack'),
      $container->get('config.factory'),
      $container->get('dl.folder_manager'),
      $container->get('dl.admin_folder_service')
    );
  }

  /**
   * Main library page.
   */
  public function libraryPage()
  {
    $build = [];

    // Add CSS and JS
    $build['#attached']['library'][] = 'dl/document-library';

    // Get statistics
    $total_docs = $this->database->query('SELECT COUNT(*) FROM {document} WHERE status = 1')->fetchField();
    $total_downloads = $this->database->query('SELECT SUM(downloads) FROM {document}')->fetchField();
    $user_docs = $this->database->query('SELECT COUNT(*) FROM {document} WHERE uid = :uid AND status = 1', [':uid' => $this->currentUser()->id()])->fetchField();

    // Get recent documents using Entity API
    $items_per_page = $this->config->get('items_per_page') ?: 20;
    $entity_storage = \Drupal::entityTypeManager()->getStorage('document');
    $query = $entity_storage->getQuery()
      ->accessCheck(TRUE)
      ->condition('status', 1)
      ->sort('created', 'DESC')
      ->range(0, $items_per_page);
    $entity_ids = $query->execute();
    $documents = $entity_storage->loadMultiple($entity_ids);

    // Get folder tree
    $folder_tree = $this->folderManager->buildFolderTree(0, TRUE);

    $build['#theme'] = 'dl_library_page';
    $build['#statistics'] = [
      'total' => $total_docs,
      'downloads' => $total_downloads ?: 0,
      'user_docs' => $user_docs,
    ];
    $build['#documents'] = $this->formatDocuments($documents);
    $build['#folder_tree'] = $folder_tree;
    $build['#can_upload'] = $this->currentUser()->hasPermission('upload documents');
    $build['#can_create_folder'] = $this->currentUser()->hasPermission('create folders');
    $build['#enable_favorites'] = $this->config->get('enable_favorites') ?? TRUE;

    return $build;
  }

  /**
   * View single document.
   */
  public function viewDocument($document)
  {
    // The $document parameter is already converted to a Document entity by the route parameter converter
    $doc = $document;

    if (!$doc || !$doc->get('status')->value) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException();
    }

    // Get file from field
    $file = $doc->get('field_file')->entity;
    if (!$file) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException();
    }

    // Get version history with user info
    $version_records = $this->database->query('SELECT * FROM {dl_versions} WHERE document_id = :id ORDER BY created DESC', [':id' => $doc->id()])->fetchAll();
    $versions = [];
    foreach ($version_records as $version) {
      $version_user = \Drupal\user\Entity\User::load($version->uid);
      $versions[] = [
        'version' => $version->version,
        'created' => $version->created,
        'notes' => $version->notes,
        'user_name' => $version_user ? $version_user->getDisplayName() : 'Unknown',
      ];
    }

    // Get owner
    $owner = $doc->get('uid')->entity;

    // Check if favorited
    $is_favorite = $this->database->query('SELECT COUNT(*) FROM {dl_favorites} WHERE uid = :uid AND document_id = :did', [
      ':uid' => $this->currentUser()->id(),
      ':did' => $doc->id(),
    ])->fetchField();

    // Get folder and breadcrumbs
    $folder = NULL;
    $breadcrumbs = [];
    $folder_id = $doc->get('folder_id')->value;
    if ($folder_id > 0) {
      $folder = $this->folderManager->getFolder($folder_id);
      $breadcrumbs = $this->folderManager->getFolderBreadcrumbs($folder_id);

      // Add slug_path to each breadcrumb
      foreach ($breadcrumbs as &$crumb) {
        $crumb->slug_path = $this->folderManager->getFolderSlugPath($crumb->folder_id);
      }

      // Add slug_path to folder
      if ($folder) {
        $folder->slug_path = $this->folderManager->getFolderSlugPath($folder_id);
      }
    }

    // Get file size from the filesize field
    // Note: For File entities, the size is stored in the 'filesize' field
    $file_size = $file->getSize();

    // If getSize() returns null/0, try getting from filesize field directly
    if (!$file_size && $file->hasField('filesize')) {
      $file_size = $file->get('filesize')->value;
    }

    $file_size_formatted = $file_size ? $this->formatBytes($file_size) : '0 B';

    $build = [];
    $build['#attached']['library'][] = 'dl/document-library';
    $build['#theme'] = 'dl_document_view';
    $build['#document'] = $doc;
    $build['#file'] = $file;
    $build['#file_url'] = $this->fileUrlGenerator->generateAbsoluteString($file->getFileUri());
    $build['#file_size'] = $file_size_formatted;
    $build['#owner'] = $owner;
    $build['#versions'] = $versions;
    $build['#is_favorite'] = $is_favorite;
    $build['#folder'] = $folder;
    $build['#breadcrumbs'] = $breadcrumbs;
    $build['#can_edit'] = $this->currentUser()->hasPermission('edit documents') &&
      ($doc->get('uid')->target_id == $this->currentUser()->id() || $this->currentUser()->hasPermission('manage all documents'));
    $build['#can_download'] = $this->currentUser()->hasPermission('download documents');
    $build['#enable_favorites'] = $this->config->get('enable_favorites') ?? TRUE;
    $build['#enable_versioning'] = $this->config->get('enable_versioning') ?? TRUE;
    $build['#enable_downloads_tracking'] = $this->config->get('enable_downloads_tracking') ?? TRUE;
    $build['#csrf_token'] = \Drupal::csrfToken()->get('document-favorite');
    // Ensure cacheability metadata includes session for CSRF token
    $build['#cache']['contexts'][] = 'session';
    return $build;
  }

  /**
   * Download document.
   */
  public function downloadDocument($document)
  {
    // The download route passes the numeric ID, not an entity (no parameter conversion)
    $doc = \Drupal::entityTypeManager()->getStorage('document')->load($document);

    if (!$doc || !$doc->get('status')->value) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException();
    }

    $file = $doc->get('field_file')->entity;
    if (!$file) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException();
    }

    // Track download
    $this->database->insert('dl_downloads')
      ->fields([
        'document_id' => $doc->id(),
        'uid' => $this->currentUser()->id(),
        'timestamp' => time(),
        'ip_address' => $this->requestStack->getCurrentRequest()->getClientIp(),
      ])
      ->execute();

    // Increment download count
    $current_downloads = $doc->get('downloads')->value;
    $doc->set('downloads', $current_downloads + 1);
    $doc->save();

    // Return file
    $uri = $file->getFileUri();
    $path = \Drupal::service('file_system')->realpath($uri);

    // If realpath fails (e.g., for remote streams), use the URI directly
    if (!$path || !file_exists($path)) {
      $path = $uri;
    }

    $response = new BinaryFileResponse($path);
    $response->setContentDisposition('attachment', $file->getFilename());

    return $response;
  }

  /**
   * Search page.
   */
  public function searchPage()
  {
    $request = $this->requestStack->getCurrentRequest();
    $keyword = $request->query->get('q', '');
    $sort = $request->query->get('sort', 'date');

    $documents = [];

    if ($keyword) {
      // Escape special LIKE characters
      $keyword_escaped = $this->database->escapeLike($keyword);

      $query = $this->database->select('document', 'd')
        ->fields('d')
        ->condition('status', 1);

      if ($keyword) {
        $or = $query->orConditionGroup()
          ->condition('title', '%' . $keyword_escaped . '%', 'LIKE')
          ->condition('description', '%' . $keyword_escaped . '%', 'LIKE')
          ->condition('tags', '%' . $keyword_escaped . '%', 'LIKE');
        $query->condition($or);
      }

      // Validate sort parameter
      $allowed_sorts = ['title', 'downloads', 'date'];
      if (!in_array($sort, $allowed_sorts)) {
        $sort = 'date';
      }

      switch ($sort) {
        case 'title':
          $query->orderBy('title', 'ASC');
          break;
        case 'downloads':
          $query->orderBy('downloads', 'DESC');
          break;
        default:
          $query->orderBy('created', 'DESC');
      }

      $documents = $query->execute()->fetchAll();
    }

    $build = [];
    $build['#attached']['library'][] = 'dl/document-library';
    $build['#theme'] = 'dl_search_page';
    $build['#keyword'] = $keyword;
    $build['#sort'] = $sort;
    $build['#documents'] = $this->formatDocuments($documents);

    return $build;
  }

  /**
   * Favorites page.
   */
  public function favoritesPage()
  {
    $query = $this->database->select('dl_favorites', 'f')
      ->fields('f')
      ->condition('uid', $this->currentUser()->id())
      ->orderBy('timestamp', 'DESC');

    $favorites = $query->execute()->fetchAll();

    $documents = [];
    foreach ($favorites as $fav) {
      $doc = $this->database->query('SELECT * FROM {document} WHERE id = :id AND status = 1', [':id' => $fav->document_id])->fetchObject();
      if ($doc) {
        $documents[] = $doc;
      }
    }

    $build = [];
    $build['#attached']['library'][] = 'dl/document-library';
    $build['#theme'] = 'dl_favorites_page';
    $build['#documents'] = $this->formatDocuments($documents);
    $build['#csrf_token'] = \Drupal::csrfToken()->get('document-favorite');

    return $build;
  }

  /**
   * Toggle favorite.
   */
  public function toggleFavorite($document)
  {
    $request = $this->requestStack->getCurrentRequest();
    $token = $request->headers->get('X-CSRF-Token');

    if (!\Drupal::csrfToken()->validate($token, 'document-favorite')) {
      \Drupal::logger('dl')->warning('CSRF validation failed for doc @id', ['@id' => $document]);
      return new JsonResponse(['error' => 'Invalid CSRF token'], 403);
    }

    $uid = $this->currentUser()->id();

    $exists = $this->database->query('SELECT COUNT(*) FROM {dl_favorites} WHERE uid = :uid AND document_id = :did', [
      ':uid' => $uid,
      ':did' => $document,
    ])->fetchField();

    if ($exists) {
      $this->database->delete('dl_favorites')
        ->condition('uid', $uid)
        ->condition('document_id', $document)
        ->execute();
      $favorited = FALSE;
    } else {
      $this->database->insert('dl_favorites')
        ->fields([
          'uid' => $uid,
          'document_id' => $document,
          'timestamp' => time(),
        ])
        ->execute();
      $favorited = TRUE;
    }

    return new JsonResponse(['favorited' => $favorited]);
  }

  /**
   * Document title callback.
   */
  public function documentTitle($document)
  {
    // The $document parameter is already converted to a Document entity by the route parameter converter
    return $document ? $document->label() : 'Document';
  }

  /**
   * Folder page - view documents in a specific folder.
   */
  public function folderPage($folder = 0)
  {
    $folder_obj = NULL;
    $breadcrumbs = [];

    // Get folder info if not root
    if ($folder > 0) {
      $folder_obj = $this->folderManager->getFolder($folder);
      if (!$folder_obj || !$folder_obj->status) {
        throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException();
      }
      $breadcrumbs = $this->folderManager->getFolderBreadcrumbs($folder);

      // Add slug_path to each breadcrumb
      foreach ($breadcrumbs as $crumb) {
        $crumb->slug_path = $this->folderManager->getFolderSlugPath($crumb->folder_id);
      }
    }

    // Get documents in this folder
    $query = $this->database->select('document', 'd')
      ->fields('d')
      ->condition('folder_id', $folder)
      ->condition('status', 1)
      ->orderBy('created', 'DESC');
    $documents = $query->execute()->fetchAll();

    // Get subfolders
    $subfolders = $this->folderManager->getFolderTree($folder, TRUE);

    // Get folder tree for sidebar
    $folder_tree = $this->folderManager->buildFolderTree(0, TRUE);

    // Add slug_path to folder object if it exists
    if ($folder_obj) {
      $folder_obj->slug_path = $this->folderManager->getFolderSlugPath($folder);
    }

    // Add slug_path to subfolders
    $subfolders_with_paths = [];
    foreach ($subfolders as $subfolder) {
      $subfolder_data = clone $subfolder;
      $subfolder_data->slug_path = $this->folderManager->getFolderSlugPath($subfolder->folder_id);
      $subfolders_with_paths[] = $subfolder_data;
    }
    $subfolders = $subfolders_with_paths;

    $build = [];
    $build['#attached']['library'][] = 'dl/document-library';
    $build['#theme'] = 'dl_folder_page';
    $build['#folder'] = $folder_obj;
    $build['#breadcrumbs'] = $breadcrumbs;
    $build['#documents'] = $this->formatDocuments($documents);
    $build['#subfolders'] = $subfolders;
    $build['#folder_tree'] = $folder_tree;
    $build['#can_upload'] = $this->currentUser()->hasPermission('upload documents');
    $build['#can_create_folder'] = $this->currentUser()->hasPermission('create folders');
    $build['#can_edit_folder'] = $folder_obj && (
      ($folder_obj->uid == $this->currentUser()->id() && $this->currentUser()->hasPermission('edit folders')) ||
      $this->currentUser()->hasPermission('manage all folders')
    );

    return $build;
  }

  /**
   * Folder title callback.
   */
  public function folderTitle($folder = 0)
  {
    if ($folder == 0) {
      return $this->t('Root Folder');
    }
    $folder_obj = $this->folderManager->getFolder($folder);
    return $folder_obj ? $folder_obj->name : $this->t('Folder');
  }

  /**
   * Folder page using slug-based path.
   */
  public function folderPageByPath($path)
  {
    $folder_obj = $this->folderManager->getFolderByPath($path);

    if (!$folder_obj) {
      throw new \Symfony\Component\HttpKernel\Exception\NotFoundHttpException();
    }

    return $this->folderPage($folder_obj->folder_id);
  }

  /**
   * Folder title callback using slug-based path.
   */
  public function folderTitleByPath($path)
  {
    $folder_obj = $this->folderManager->getFolderByPath($path);
    return $folder_obj ? $folder_obj->name : $this->t('Folder');
  }

  /**
   * Folder page using path segments.
   */
  public function folderPageBySegments($segment1 = NULL, $segment2 = NULL, $segment3 = NULL)
  {
    // Build path from segments
    $segments = array_filter([$segment1, $segment2, $segment3]);
    $path = implode('/', $segments);

    return $this->folderPageByPath($path);
  }

  /**
   * Folder title callback using path segments.
   */
  public function folderTitleBySegments($segment1 = NULL, $segment2 = NULL, $segment3 = NULL)
  {
    // Build path from segments
    $segments = array_filter([$segment1, $segment2, $segment3]);
    $path = implode('/', $segments);

    return $this->folderTitleByPath($path);
  }

  /**
   * Format documents with additional data.
   */
  protected function formatDocuments($documents)
  {
    $formatted = [];

    foreach ($documents as $doc) {
      // Handle both entity objects and database result objects for backwards compatibility
      if (is_object($doc) && method_exists($doc, 'get')) {
        // Entity object
        $file = !$doc->get('field_file')->isEmpty() ? $doc->get('field_file')->entity : NULL;
        $owner = $doc->get('uid')->entity;
        $folder_id = $doc->get('folder_id')->value;

        $formatted[] = [
          'id' => $doc->id(),
          'title' => $doc->get('title')->value,
          'description' => $doc->get('field_description')->value ?? '',
          'file' => $file,
          'file_url' => $file ? $this->fileUrlGenerator->generateAbsoluteString($file->getFileUri()) : '',
          'file_size' => $file ? $this->formatBytes($file->getSize()) : '',
          'file_type' => $file ? $file->getMimeType() : '',
          'version' => $doc->get('field_version')->value ?? '',
          'downloads' => $doc->get('downloads')->value,
          'owner' => $owner,
          'created' => $doc->get('created')->value,
          'changed' => $doc->get('changed')->value,
          'folder_id' => $folder_id,
          'folder_slug_path' => $folder_id > 0 ? $this->folderManager->getFolderSlugPath($folder_id) : '',
          'folder_name' => $folder_id > 0 ? ($this->folderManager->getFolder($folder_id)->name ?? '') : '',
          'status' => $doc->get('status')->value,
        ];
      } else {
        // Database result object (legacy)
        $file_id = $doc->file_id ?? NULL;
        $file = $file_id ? File::load($file_id) : NULL;
        $owner = \Drupal\user\Entity\User::load($doc->uid);
        $folder_id = $doc->folder_id ?? 0;

        $formatted[] = [
          'id' => $doc->id,
          'title' => $doc->title,
          'description' => $doc->description ?? '',
          'file' => $file,
          'file_url' => $file ? $this->fileUrlGenerator->generateAbsoluteString($file->getFileUri()) : '',
          'file_size' => $file ? $this->formatBytes($file->getSize()) : '',
          'file_type' => $file ? $file->getMimeType() : '',
          'version' => $doc->version ?? '',
          'downloads' => $doc->downloads,
          'owner' => $owner,
          'created' => $doc->created,
          'changed' => $doc->changed,
          'folder_id' => $folder_id,
          'folder_slug_path' => $folder_id > 0 ? $this->folderManager->getFolderSlugPath($folder_id) : '',
          'folder_name' => $folder_id > 0 ? ($this->folderManager->getFolder($folder_id)->name ?? '') : '',
          'status' => $doc->status,
        ];
      }
    }

    return $formatted;
  }

  /**
   * Admin documents listing page.
   */
  public function adminDocumentsPage()
  {
    $build = [];

    // Add CSS and JS
    $build['#attached']['library'][] = 'dl/document-library';

    // Get all documents (including unpublished)
    $query = $this->database->select('document', 'd')
      ->fields('d')
      ->orderBy('created', 'DESC');
    $documents = $query->execute()->fetchAll();

    // Get folder tree (including unpublished)
    $folder_tree = $this->folderManager->buildFolderTree(0, FALSE);

    // Get statistics
    $total_docs = count($documents);
    $published_docs = $this->database->query('SELECT COUNT(*) FROM {document} WHERE status = 1')->fetchField();
    $total_downloads = $this->database->query('SELECT SUM(downloads) FROM {document}')->fetchField();

    $build['#theme'] = 'dl_admin_documents';
    $build['#documents'] = $this->formatDocuments($documents);
    $build['#folder_tree'] = $folder_tree;
    $build['#statistics'] = [
      'total' => $total_docs,
      'published' => $published_docs,
      'unpublished' => $total_docs - $published_docs,
      'downloads' => $total_downloads ?: 0,
    ];
    $build['#can_upload'] = $this->currentUser()->hasPermission('upload documents');
    $build['#can_create_folder'] = $this->currentUser()->hasPermission('create folders');

    return $build;
  }

  /**
   * Admin folders listing page.
   */
  public function adminFoldersPage() {
    // Get statistics from service
    $statistics = $this->adminFolderService->getFolderStatistics();

    // Get the folder admin form
    $form = $this->formBuilder()->getForm(\Drupal\dl\Form\FolderAdminForm::class);

    $build = [];
    $build['#attached']['library'][] = 'dl/document-library';
    $build['#theme'] = 'dl_admin_folders';
    $build['#statistics'] = $statistics;
    $build['#folders_form'] = $form;

    return $build;
  }

  /**
   * Format bytes to human readable format.
   */
  protected function formatBytes($bytes, $precision = 2)
  {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);
    $bytes /= (1 << (10 * $pow));
    return round($bytes, $precision) . ' ' . $units[$pow];
  }

  /**
   * Handle bulk operations on documents.
   */
  public function bulkOperations()
  {
    $request = $this->requestStack->getCurrentRequest();

    // Validate CSRF token
    $token = $request->headers->get('X-CSRF-Token');
    if (!$token || !\Drupal::csrfToken()->validate($token)) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => $this->t('Invalid CSRF token.'),
      ], 403);
    }

    $operation = $request->request->get('operation');
    $document_ids = $request->request->get('document_ids', []);

    // Validate input
    if (empty($operation) || empty($document_ids) || !is_array($document_ids)) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => $this->t('Invalid request.'),
      ], 400);
    }

    // Load entity storage
    $entity_storage = \Drupal::entityTypeManager()->getStorage('document');
    $count = 0;

    try {
      switch ($operation) {
        case 'delete':
          // Delete each document
          foreach ($document_ids as $id) {
            $document = $entity_storage->load($id);
            if ($document) {
              // Check permission
              if ($document->get('uid')->target_id == $this->currentUser()->id() || $this->currentUser()->hasPermission('manage all documents')) {
                $document->delete();
                $count++;
              }
            }
          }
          $message = $this->formatPlural($count, '1 document deleted.', '@count documents deleted.');
          break;

        case 'publish':
          // Publish each document
          foreach ($document_ids as $id) {
            $document = $entity_storage->load($id);
            if ($document) {
              // Check permission
              if ($document->get('uid')->target_id == $this->currentUser()->id() || $this->currentUser()->hasPermission('manage all documents')) {
                $document->set('status', 1);
                $document->save();
                $count++;
              }
            }
          }
          $message = $this->formatPlural($count, '1 document published.', '@count documents published.');
          break;

        case 'unpublish':
          // Unpublish each document
          foreach ($document_ids as $id) {
            $document = $entity_storage->load($id);
            if ($document) {
              // Check permission
              if ($document->get('uid')->target_id == $this->currentUser()->id() || $this->currentUser()->hasPermission('manage all documents')) {
                $document->set('status', 0);
                $document->save();
                $count++;
              }
            }
          }
          $message = $this->formatPlural($count, '1 document unpublished.', '@count documents unpublished.');
          break;

        default:
          return new JsonResponse([
            'success' => FALSE,
            'message' => $this->t('Unknown operation.'),
          ], 400);
      }

      return new JsonResponse([
        'success' => TRUE,
        'message' => $message,
        'count' => $count,
      ]);
    } catch (\Exception $e) {
      \Drupal::logger('dl')->error('Bulk operation error: @error', ['@error' => $e->getMessage()]);
      return new JsonResponse([
        'success' => FALSE,
        'message' => $this->t('An error occurred. Please try again.'),
      ], 500);
    }
  }
}
