<?php

namespace Drupal\dl;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Entity\EntityAccessControlHandler;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Session\AccountInterface;

/**
 * Access controller for the Document entity.
 */
class DocumentAccessControlHandler extends EntityAccessControlHandler {

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(EntityInterface $entity, $operation, AccountInterface $account) {
    /** @var \Drupal\dl\Entity\DocumentInterface $entity */

    switch ($operation) {
      case 'view':
        // Check if published or user has admin permission.
        if (!$entity->isPublished()) {
          return AccessResult::allowedIfHasPermission($account, 'administer document library');
        }
        return AccessResult::allowedIfHasPermission($account, 'access document library');

      case 'download':
        // Custom permission for downloading.
        if (!$entity->isPublished()) {
          return AccessResult::allowedIfHasPermission($account, 'administer document library');
        }
        return AccessResult::allowedIfHasPermission($account, 'download documents');

      case 'update':
        // Check edit permission or ownership.
        if ($account->hasPermission('administer document library')) {
          return AccessResult::allowed()->cachePerPermissions();
        }
        if ($account->hasPermission('edit own documents') && $entity->getOwnerId() == $account->id()) {
          return AccessResult::allowed()
            ->cachePerPermissions()
            ->cachePerUser()
            ->addCacheableDependency($entity);
        }
        if ($account->hasPermission('edit documents')) {
          return AccessResult::allowed()->cachePerPermissions();
        }
        return AccessResult::neutral()->cachePerPermissions();

      case 'delete':
        // Check delete permission or ownership.
        if ($account->hasPermission('administer document library')) {
          return AccessResult::allowed()->cachePerPermissions();
        }
        if ($account->hasPermission('delete own documents') && $entity->getOwnerId() == $account->id()) {
          return AccessResult::allowed()
            ->cachePerPermissions()
            ->cachePerUser()
            ->addCacheableDependency($entity);
        }
        if ($account->hasPermission('delete documents')) {
          return AccessResult::allowed()->cachePerPermissions();
        }
        return AccessResult::neutral()->cachePerPermissions();
    }

    // Unknown operation, no opinion.
    return AccessResult::neutral()->cachePerPermissions();
  }

  /**
   * {@inheritdoc}
   */
  protected function checkCreateAccess(AccountInterface $account, array $context, $entity_bundle = NULL) {
    return AccessResult::allowedIfHasPermission($account, 'upload documents')
      ->orIf(AccessResult::allowedIfHasPermission($account, 'administer document library'));
  }

}
