<?php

namespace Drupal\dl;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Link;
use Drupal\Core\Url;

/**
 * Defines a class to build a listing of Document entities.
 */
class DocumentListBuilder extends EntityListBuilder {

  /**
   * {@inheritdoc}
   */
  public function buildHeader() {
    $header['id'] = $this->t('ID');
    $header['title'] = $this->t('Title');
    $header['folder'] = $this->t('Folder');
    $header['owner'] = $this->t('Owner');
    $header['downloads'] = $this->t('Downloads');
    $header['status'] = $this->t('Status');
    $header['created'] = $this->t('Created');
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity) {
    /** @var \Drupal\dl\Entity\DocumentInterface $entity */
    $row['id'] = $entity->id();
    $row['title'] = Link::fromTextAndUrl(
      $entity->getTitle(),
      $entity->toUrl()
    );

    // Folder - load from dl_folders table.
    $folder_id = $entity->get('folder_id')->value;
    if ($folder_id && $folder_id > 0) {
      $folder = \Drupal::database()->select('dl_folders', 'f')
        ->fields('f', ['name'])
        ->condition('folder_id', $folder_id)
        ->execute()
        ->fetchField();
      $row['folder'] = $folder ?: $this->t('- Unknown -');
    }
    else {
      $row['folder'] = $this->t('- Root -');
    }

    // Owner.
    $owner = $entity->getOwner();
    $row['owner'] = $owner ? $owner->getDisplayName() : $this->t('Anonymous');

    // Downloads.
    $row['downloads'] = $entity->getDownloadCount();

    // Status.
    $row['status'] = $entity->isPublished() ? $this->t('Published') : $this->t('Unpublished');

    // Created.
    $row['created'] = \Drupal::service('date.formatter')->format($entity->getCreatedTime(), 'short');

    return $row + parent::buildRow($entity);
  }

  /**
   * {@inheritdoc}
   */
  protected function getDefaultOperations(EntityInterface $entity) {
    $operations = parent::getDefaultOperations($entity);

    // Add download operation.
    if ($entity->access('view')) {
      $operations['download'] = [
        'title' => $this->t('Download'),
        'weight' => 15,
        'url' => Url::fromRoute('dl.document.download', ['document' => $entity->id()]),
      ];
    }

    return $operations;
  }

}
