<?php

namespace Drupal\dl;

use Drupal\Core\Database\Connection;
use Drupal\Core\Session\AccountProxyInterface;

/**
 * Service for managing document library folders.
 */
class FolderManager {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * Constructs a FolderManager object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   */
  public function __construct(Connection $database, AccountProxyInterface $current_user) {
    $this->database = $database;
    $this->currentUser = $current_user;
  }

  /**
   * Create a new folder.
   *
   * @param string $name
   *   The folder name.
   * @param int $parent_id
   *   The parent folder ID (0 for root).
   * @param string $description
   *   The folder description.
   * @param int $weight
   *   Sort weight for ordering.
   *
   * @return int
   *   The new folder ID.
   */
  public function createFolder($name, $parent_id = 0, $description = '', $weight = 0) {
    // Get parent folder info
    $parent_depth = 0;
    $parent_path = '';
    if ($parent_id > 0) {
      $parent = $this->getFolder($parent_id);
      if ($parent) {
        $parent_depth = $parent->depth;
        $parent_path = $parent->path;
      }
    }

    // Generate slug
    $slug = $this->generateSlug($name, $parent_id);

    // Insert folder
    $folder_id = $this->database->insert('dl_folders')
      ->fields([
        'parent_id' => $parent_id,
        'name' => $name,
        'slug' => $slug,
        'description' => $description,
        'path' => '',
        'depth' => $parent_depth + 1,
        'weight' => (int) $weight,
        'status' => 1,
        'uid' => $this->currentUser->id(),
        'created' => time(),
        'changed' => time(),
      ])
      ->execute();

    // Update path with folder_id
    $path = $parent_path . '/' . $folder_id;
    $this->database->update('dl_folders')
      ->fields(['path' => $path])
      ->condition('folder_id', $folder_id)
      ->execute();

    return $folder_id;
  }

  /**
   * Get a folder by ID.
   *
   * @param int $folder_id
   *   The folder ID.
   *
   * @return object|false
   *   The folder object or FALSE if not found.
   */
  public function getFolder($folder_id) {
    return $this->database->query(
      'SELECT * FROM {dl_folders} WHERE folder_id = :id',
      [':id' => $folder_id]
    )->fetchObject();
  }

  /**
   * Get folder tree starting from a parent.
   *
   * @param int $parent_id
   *   The parent folder ID (0 for root).
   * @param bool $published_only
   *   Whether to only return published folders.
   *
   * @return array
   *   Array of folder objects.
   */
  public function getFolderTree($parent_id = 0, $published_only = TRUE) {
    $query = $this->database->select('dl_folders', 'f')
      ->fields('f')
      ->condition('parent_id', $parent_id)
      ->orderBy('weight', 'ASC')
      ->orderBy('name', 'ASC');

    if ($published_only) {
      $query->condition('status', 1);
    }

    return $query->execute()->fetchAll();
  }

  /**
   * Get all folders as a hierarchical tree.
   *
   * @param int $parent_id
   *   The parent folder ID.
   * @param bool $published_only
   *   Whether to only return published folders.
   *
   * @return array
   *   Hierarchical array of folders.
   */
  public function buildFolderTree($parent_id = 0, $published_only = TRUE) {
    $folders = $this->getFolderTree($parent_id, $published_only);
    $tree = [];

    foreach ($folders as $folder) {
      $folder_array = (array) $folder;

      // Load owner name
      if ($folder->uid) {
        $owner = \Drupal\user\Entity\User::load($folder->uid);
        $folder_array['owner_name'] = $owner ? $owner->getDisplayName() : 'Unknown';
      } else {
        $folder_array['owner_name'] = 'System';
      }

      // Add slug_path
      $folder_array['slug_path'] = $this->getFolderSlugPath($folder->folder_id);

      $folder_array['children'] = $this->buildFolderTree($folder->folder_id, $published_only);
      $folder_array['document_count'] = $this->getFolderDocumentCount($folder->folder_id);
      $tree[] = $folder_array;
    }

    return $tree;
  }

  /**
   * Get document count for a folder.
   *
   * @param int $folder_id
   *   The folder ID.
   * @param bool $published_only
   *   Whether to only count published documents.
   *
   * @return int
   *   Number of documents.
   */
  public function getFolderDocumentCount($folder_id, $published_only = TRUE) {
    $query = $this->database->select('document', 'd')
      ->condition('folder_id', $folder_id);

    if ($published_only) {
      $query->condition('status', 1);
    }

    return $query->countQuery()->execute()->fetchField();
  }

  /**
   * Update a folder.
   *
   * @param int $folder_id
   *   The folder ID.
   * @param array $fields
   *   Fields to update.
   *
   * @return bool
   *   TRUE on success.
   */
  public function updateFolder($folder_id, array $fields) {
    $fields['changed'] = time();

    // If name changed, regenerate slug
    if (isset($fields['name'])) {
      $folder = $this->getFolder($folder_id);
      $parent_id = isset($fields['parent_id']) ? $fields['parent_id'] : $folder->parent_id;
      $fields['slug'] = $this->generateSlug($fields['name'], $parent_id, $folder_id);
    }

    $this->database->update('dl_folders')
      ->fields($fields)
      ->condition('folder_id', $folder_id)
      ->execute();

    // If parent changed, update path and depth
    if (isset($fields['parent_id'])) {
      $this->updateFolderPath($folder_id);
    }

    return TRUE;
  }

  /**
   * Update folder path after parent change.
   *
   * @param int $folder_id
   *   The folder ID.
   */
  protected function updateFolderPath($folder_id) {
    $folder = $this->getFolder($folder_id);
    if (!$folder) {
      return;
    }

    // Get new parent path
    $parent_path = '';
    $parent_depth = 0;
    if ($folder->parent_id > 0) {
      $parent = $this->getFolder($folder->parent_id);
      if ($parent) {
        $parent_path = $parent->path;
        $parent_depth = $parent->depth;
      }
    }

    // Update folder path and depth
    $new_path = $parent_path . '/' . $folder_id;
    $new_depth = $parent_depth + 1;

    $this->database->update('dl_folders')
      ->fields([
        'path' => $new_path,
        'depth' => $new_depth,
      ])
      ->condition('folder_id', $folder_id)
      ->execute();

    // Update children paths recursively
    $children = $this->getFolderTree($folder_id, FALSE);
    foreach ($children as $child) {
      $this->updateFolderPath($child->folder_id);
    }
  }

  /**
   * Delete a folder.
   *
   * @param int $folder_id
   *   The folder ID.
   * @param bool $move_contents_to_parent
   *   Whether to move contents to parent or delete them.
   *
   * @return bool
   *   TRUE on success.
   */
  public function deleteFolder($folder_id, $move_contents_to_parent = TRUE) {
    $folder = $this->getFolder($folder_id);
    if (!$folder) {
      return FALSE;
    }

    if ($move_contents_to_parent) {
      // Move documents to parent folder
      $this->database->update('document')
        ->fields(['folder_id' => $folder->parent_id])
        ->condition('folder_id', $folder_id)
        ->execute();

      // Move child folders to parent
      $this->database->update('dl_folders')
        ->fields(['parent_id' => $folder->parent_id])
        ->condition('parent_id', $folder_id)
        ->execute();

      // Update paths for moved children
      $children = $this->getFolderTree($folder->parent_id, FALSE);
      foreach ($children as $child) {
        $this->updateFolderPath($child->folder_id);
      }
    }
    else {
      // Delete all child folders recursively
      $children = $this->getFolderTree($folder_id, FALSE);
      foreach ($children as $child) {
        $this->deleteFolder($child->folder_id, FALSE);
      }

      // Delete documents in folder using Entity API
      $entity_storage = \Drupal::entityTypeManager()->getStorage('document');
      $query = $entity_storage->getQuery()
        ->accessCheck(FALSE)
        ->condition('folder_id', $folder_id);
      $entity_ids = $query->execute();
      if (!empty($entity_ids)) {
        $entities = $entity_storage->loadMultiple($entity_ids);
        $entity_storage->delete($entities);
      }
    }

    // Delete the folder
    $this->database->delete('dl_folders')
      ->condition('folder_id', $folder_id)
      ->execute();

    return TRUE;
  }

  /**
   * Get breadcrumb trail for a folder.
   *
   * @param int $folder_id
   *   The folder ID.
   *
   * @return array
   *   Array of folder objects from root to current.
   */
  public function getFolderBreadcrumbs($folder_id) {
    if ($folder_id == 0) {
      return [];
    }

    $folder = $this->getFolder($folder_id);
    if (!$folder) {
      return [];
    }

    // Parse path to get all parent IDs
    $path_parts = array_filter(explode('/', $folder->path));
    $breadcrumbs = [];

    foreach ($path_parts as $folder_id) {
      $f = $this->getFolder($folder_id);
      if ($f) {
        $breadcrumbs[] = $f;
      }
    }

    return $breadcrumbs;
  }

  /**
   * Get folders as a flat list for dropdowns.
   *
   * @param int $exclude_id
   *   Folder ID to exclude (and its children).
   * @param bool $published_only
   *   Whether to only return published folders.
   *
   * @return array
   *   Array of folder_id => indented name.
   */
  public function getFolderOptions($exclude_id = NULL, $published_only = TRUE) {
    $query = $this->database->select('dl_folders', 'f')
      ->fields('f')
      ->orderBy('path', 'ASC');

    if ($published_only) {
      $query->condition('status', 1);
    }

    $folders = $query->execute()->fetchAll();
    $options = [0 => '-- Root --'];

    foreach ($folders as $folder) {
      // Skip excluded folder and its children
      if ($exclude_id && (
        $folder->folder_id == $exclude_id ||
        strpos($folder->path, '/' . $exclude_id . '/') !== FALSE
      )) {
        continue;
      }

      $indent = str_repeat('--', $folder->depth);
      $options[$folder->folder_id] = $indent . ' ' . $folder->name;
    }

    return $options;
  }

  /**
   * Generate a URL-friendly slug from folder name.
   *
   * @param string $name
   *   The folder name.
   * @param int $parent_id
   *   The parent folder ID.
   * @param int $exclude_folder_id
   *   Folder ID to exclude when checking uniqueness (for updates).
   *
   * @return string
   *   The generated slug.
   */
  public function generateSlug($name, $parent_id = 0, $exclude_folder_id = NULL) {
    // Convert to lowercase and replace spaces/special chars with hyphens
    $slug = strtolower($name);
    $slug = preg_replace('/[^a-z0-9]+/', '-', $slug);
    $slug = trim($slug, '-');

    // Ensure uniqueness within parent
    $base_slug = $slug;
    $counter = 1;

    while ($this->slugExists($slug, $parent_id, $exclude_folder_id)) {
      $slug = $base_slug . '-' . $counter;
      $counter++;
    }

    return $slug;
  }

  /**
   * Check if a slug exists within a parent folder.
   *
   * @param string $slug
   *   The slug to check.
   * @param int $parent_id
   *   The parent folder ID.
   * @param int $exclude_folder_id
   *   Folder ID to exclude from check (for updates).
   *
   * @return bool
   *   TRUE if slug exists.
   */
  protected function slugExists($slug, $parent_id = 0, $exclude_folder_id = NULL) {
    $query = $this->database->select('dl_folders', 'f')
      ->condition('slug', $slug)
      ->condition('parent_id', $parent_id);

    if ($exclude_folder_id) {
      $query->condition('folder_id', $exclude_folder_id, '!=');
    }

    return $query->countQuery()->execute()->fetchField() > 0;
  }

  /**
   * Get folder by slug path (e.g., "folder1/subfolder2").
   *
   * @param string $path
   *   The slug path (e.g., "folder1/subfolder2").
   *
   * @return object|false
   *   The folder object or FALSE if not found.
   */
  public function getFolderByPath($path) {
    if (empty($path)) {
      return FALSE;
    }

    // Split path into slug segments
    $segments = explode('/', trim($path, '/'));
    $parent_id = 0;
    $folder = FALSE;

    // Walk through each segment to find the final folder
    foreach ($segments as $slug) {
      $folder = $this->database->query(
        'SELECT * FROM {dl_folders} WHERE slug = :slug AND parent_id = :parent_id AND status = 1',
        [':slug' => $slug, ':parent_id' => $parent_id]
      )->fetchObject();

      if (!$folder) {
        return FALSE;
      }

      $parent_id = $folder->folder_id;
    }

    return $folder;
  }

  /**
   * Get the full slug path for a folder (e.g., "folder1/subfolder2").
   *
   * @param int $folder_id
   *   The folder ID.
   *
   * @return string
   *   The full slug path.
   */
  public function getFolderSlugPath($folder_id) {
    if ($folder_id == 0) {
      return '';
    }

    $breadcrumbs = $this->getFolderBreadcrumbs($folder_id);
    $slugs = [];

    foreach ($breadcrumbs as $folder) {
      $slugs[] = $folder->slug;
    }

    return implode('/', $slugs);
  }

}
