// viewer-utils.js
import THREE from "./init.js";
import Toastify from "toastify-js";
import { core, setCore } from './core.js';
import TWEEN from "three/examples/jsm/libs/tween.module.js";

export const initClippingPlanes = () => {
  const clippingPlanes = [
    new THREE.Plane(new THREE.Vector3(-1, 0, 0), 0),
    new THREE.Plane(new THREE.Vector3(0, -1, 0), 0),
    new THREE.Plane(new THREE.Vector3(0, 0, -1), 0),
  ];
  setCore('clippingPlanes', clippingPlanes);
  return clippingPlanes;
};

export var toastifyOptions = {
  duration: 6500,
  gravity: "bottom",
  close: true,
  callback() {
    Toastify.reposition();
  },
};

export const showToast = (message) => {
    var myToast = Toastify(toastifyOptions);
    myToast.options.text = message;
    myToast.showToast();
};

function fetchObjectFromConfig(_name) {
  //console.log("Fetching config for", _name, core.objectsConfig);
  return core.objectsConfig?.models?.find(model => model.name === _name);
}

export const setupObject = (_object, _light, _controls, _helperObjects) => {
  const model = fetchObjectFromConfig(_object.children[0].name); //TODO: check for multiple objects
  if (typeof core.objectsConfig !== "undefined" && model) {
    if (typeof core.objectsConfig.models == undefined || core.objectsConfig.models?.length == 0) {
      if (typeof model.position !== undefined) _object.position.set(model.position.x, model.position.y, model.position.z);

      if (typeof model.scale !== undefined) _object.scale.set(model.scale.x, model.scale.y, model.scale.z);
      
      if (typeof model.rotation !== undefined) _object.rotation.set(THREE.MathUtils.degToRad(model.rotation.x), THREE.MathUtils.degToRad(model.rotation.y), THREE.MathUtils.degToRad(model.rotation.z));
    } else {
      let m = core.objectsConfig.models[core.objectsConfig.setupIndex];
      //console.log("Applying config for index", core.objectsConfig.setupIndex, m);
      if (typeof m.position !== "undefined")
        _object.position.set(m.position.x, m.position.y, m.position.z);
      if (typeof m.scale !== "undefined")
      _object.scale.set(m.scale.x, m.scale.y, m.scale.z);
      if (typeof m.rotation !== "undefined")
        _object.rotation.set(THREE.MathUtils.degToRad(m.rotation.x), THREE.MathUtils.degToRad(m.rotation.y), THREE.MathUtils.degToRad(m.rotation.z));
    }
    
    _object.needsUpdate = true;
    if (typeof _object.geometry !== "undefined") {
      _object.geometry.computeBoundingBox();
      _object.geometry.computeBoundingSphere();
    }
    _object.updateMatrix();
    _object.updateMatrixWorld(true);
  } else {
    var boundingBox = new THREE.Box3();
    if (Array.isArray(_object)) {
      for (let i = 0; i < _object.length; i++) {
        boundingBox.setFromObject(_object[i]);
        _object[i].position.set(
          -(boundingBox.min.x + boundingBox.max.x) / 2,
          -boundingBox.min.y,
          -(boundingBox.min.z + boundingBox.max.z) / 2
        );
        _object[i].needsUpdate = true;
        if (typeof _object[i].geometry !== "undefined") {
          _object[i].geometry.computeBoundingBox();
          _object[i].geometry.computeBoundingSphere();
        }
        _object[i].updateMatrixWorld();
      }
    } /*else if (_object.isGroup && fileObject.extension == "fbx") {
      //workaround for specific FBX case
      boundingBox.setFromObject(_object);
      var _obj = new THREE.Object3D();
      _obj.attach(_object);
      //_obj.position.set(-(boundingBox.min.x+boundingBox.max.x)/2, -boundingBox.min.y, -(boundingBox.min.z+boundingBox.max.z)/2);
      _obj.updateMatrixWorld();
      _object = _obj;
    }*/ else {
      boundingBox.setFromObject(_object);
      _object.position.set(
        -(boundingBox.min.x + boundingBox.max.x) / 2,
        -boundingBox.min.y,
        -(boundingBox.min.z + boundingBox.max.z) / 2
      );
      _object.updateMatrixWorld();
      //_object.position.set (0, 0, 0);
      _object.needsUpdate = true;
      if (typeof _object.geometry !== "undefined") {
        _object.geometry.computeBoundingBox();
        _object.geometry.computeBoundingSphere();
      }
    }
  }
  core.cameraLight.position.set(
    core.camera.position.x,
    core.camera.position.y,
    core.camera.position.z
  );
  if (Array.isArray(_object)) {
    core.cameraLightTarget.position.set(
      _object[0].position.x,
      _object[0].position.y,
      _object[0].position.z
    );
  } else {
    core.cameraLightTarget.position.set(
      _object.position.x,
      _object.position.y,
      _object.position.z
    );
  }
  core.cameraLight.target.updateMatrixWorld();
  core.objectsConfig.setupIndex++;
}

export async function setupCamera (_object, _light, _config, _helperObjects) {
  if (core.objectsConfig /*&& CONFIG.entity.metadata.source !== ''*/) {
    if (typeof core.objectsConfig.camera.position !== "undefined") {
      core.camera.position.set(core.objectsConfig.camera.position.x, core.objectsConfig.camera.position.y, core.objectsConfig.camera.position.z);
    } else {
      setupEmptyCamera(core.camera, _object, _helperObjects);
    }
    if (typeof core.objectsConfig.camera.target !== "undefined") {
      core.controls.target.set(core.objectsConfig.camera.target.x, core.objectsConfig.camera.target.y, core.objectsConfig.camera.target.z);
    } else {
      setupEmptyCamera(core.camera, _object, _helperObjects);
    }
  
    // Setup lights
    core.objectsConfig.scene.lights.forEach(light => {
      switch (light.type) {
        case "directional":
          //console.log("Directional light with color", light.color);
          _light.position.set(light.position.x, light.position.y, light.position.z);
          _light.rotation.set(light.target.x, light.target.y, light.target.z);
          
          _light.color = new THREE.Color().setHex(light.color);
          core.colors["DirectionalLight"] = light.color;
          
          core.intensity.startIntensityDir = _light.intensity = light.intensity;

        break;
        case "ambient":
          //console.log("Ambient light with color", light.color);
          core.ambientLight.color = new THREE.Color().setHex(light.color);
          core.colors["AmbientLight"] = light.color;
          core.intensity.startIntensityAmbient = core.ambientLight.intensity = light.intensity;
        break;
        case "point":
          //console.log("Point light at", light.position);
          core.cameraLight.color = new THREE.Color().setHex(light.color);
          core.colors["CameraLight"] = light.color;
          core.intensity.startIntensityCamera = core.cameraLight.intensity = light.intensity;
        break;
      }
    });
    // Setup background
    let _foundScene = false;
    if (core.objectsConfig.scenes !== undefined && Array.isArray(core.objectsConfig.scenes)) {
        core.objectsConfig.scenes.forEach(_scene => {
          if (_scene.background !== null) {
            if ("red" in _scene.background && "green" in _scene.background && "blue" in _scene.background) {
              var newBackground = new THREE.Color(`rgb(${_scene.background.red}, ${_scene.background.green}, ${_scene.background.blue})`);
              if (newBackground !== null) {
                _foundScene = true;
                changeBackground("linear", "#" + newBackground.getHexString());
                //console.log("Setting up scene background", _scene.background);
              }
            }
          }
        });
    } 
    if (!_foundScene) {
      if (core.objectsConfig.scene.background === null) {
        changeBackground("linear", "#ffffff", "#ffffff");
      } else {
        const gradient = parseGradient(core.objectsConfig.scene.background);
        const newBackground0 = new THREE.Color(`rgb(${gradient.colors[0].r}, ${gradient.colors[0].g}, ${gradient.colors[0].b})`);
        const newBackground1 = new THREE.Color(`rgb(${gradient.colors[1].r}, ${gradient.colors[1].g}, ${gradient.colors[1].b})`);
        changeBackground(gradient.type, "#" + newBackground0.getHexString(), "#" + newBackground1.getHexString());
      }
    }
    core.camera.updateProjectionMatrix();
    core.controls.update();
    fitCameraToCenteredObject(_object, 2.5, false, _helperObjects);
  } else {
    setupEmptyCamera(core.camera, _object, _helperObjects);
  }
}

function fitCameraToCenteredObject(object, add_offset, _fit, _helperObjects) {
  const boundingBox = new THREE.Box3();
  if (Array.isArray(object)) {
    for (let i = 0; i < object.length; i++) {
      const box = new THREE.Box3().setFromObject(object[i]);
      boundingBox.union(box);
    }
  } else {
    boundingBox.setFromObject(object);
  }

  var size = new THREE.Vector3(), center = new THREE.Vector3();
  boundingBox.getSize(size);
  boundingBox.getCenter(center); // center point
  // ground
  var distance1 = new THREE.Vector3(
    Math.abs(boundingBox.max.x - boundingBox.min.x),
    Math.abs(boundingBox.max.y - boundingBox.min.y),
    Math.abs(boundingBox.max.z - boundingBox.min.z)
  );
  core.gridSize = Math.max(distance1.x, distance1.y, distance1.z);

  core.dirLightTarget = new THREE.Object3D();
  core.dirLightTarget.position.set(0, 0, 0);

  core.lightHelper = new THREE.DirectionalLightHelper(core.dirLight, core.gridSize);
  core.scene.add(core.lightHelper);
  core.lightHelper.visible = false;

  core.scene.add(core.dirLightTarget);
  core.dirLight.target = core.dirLightTarget;
  core.dirLight.target.updateMatrixWorld();

  var gridSizeScale = core.gridSize * 2.5;
  if (core.basicGrid !== undefined) core.scene.remove(core.basicGrid);
  core.basicGrid = new THREE.Group();
  var planeMesh = new THREE.Mesh(
    new THREE.PlaneGeometry(gridSizeScale, gridSizeScale),
    new THREE.MeshPhongMaterial({
      color: 0xefefef,
      depthWrite: false,
      transparent: true,
      opacity: 0.65,
    })
  );
  planeMesh.rotation.x = -Math.PI / 2;
  planeMesh.position.set(0, 0, 0);
  planeMesh.receiveShadow = true;
  core.basicGrid.add(planeMesh);

  const axesHelper = new THREE.AxesHelper(core.gridSize);
  axesHelper.position.set(0, 0, 0);
  core.basicGrid.add(axesHelper);

  const grid = new THREE.GridHelper(gridSizeScale, 25, 0xaeaeae, 0x000000);
  grid.material.opacity = 0.1;
  grid.material.transparent = true;
  grid.position.set(0, 0, 0);
  core.basicGrid.add(grid);

  core.scene.add(core.basicGrid);

  // Half size of the object
  const halfHeight = size.y / 2;
  const halfWidth = size.x / 2;

  // Camera distance from object center (along z-axis or camera direction)
  const fitHeightDistance = halfHeight / Math.tan(THREE.MathUtils.degToRad(core.camera.fov / 2));
  const fitWidthDistance = halfWidth / Math.tan(THREE.MathUtils.degToRad(core.camera.fov / 2)) / core.camera.aspect;

  const distance = Math.max(fitHeightDistance, fitWidthDistance) * add_offset;

  // Compute camera position
  const direction = new THREE.Vector3(0, 0, 1); // camera looks along -Z by default
  core.camera.position.copy(center).add(direction.multiplyScalar(distance));
  core.camera.lookAt(center);
  let cameraZ = core.camera.position.z;

  core.cameraCoords = {
    x: core.camera.position.x,
    y: core.camera.position.y,
    z: cameraZ * 0.85,
  };
  core.tween = new TWEEN.Tween(core.cameraCoords)
    .to({ z: core.camera.position.z }, 1500)
    .onUpdate(() => {
      core.camera.position.set(-core.cameraCoords.x*1.2, core.cameraCoords.y*1.7, core.cameraCoords.z*1.1);
      core.cameraLight.position.set(core.cameraCoords.x, core.cameraCoords.y, core.cameraCoords.z);
      core.camera.updateProjectionMatrix();
      core.controls.update();
    })
    .start();

  // set the far plane of the camera so that it easily encompasses the whole object
  const minZ = boundingBox.min.z;
  const cameraToFarEdge = minZ < 0 ? -minZ + cameraZ : cameraZ - minZ;

  //camera.far = cameraToFarEdge * 3;
  core.camera.updateProjectionMatrix();
  if (core.controls !== undefined && _fit) {
    // set camera to rotate around the center
    core.controls.target = new THREE.Vector3(0, offset.y, 0);

    // prevent camera from zooming out far enough to create far plane cutoff
    core.controls.maxDistance = cameraToFarEdge * 2;
  }
  core.controls.update();

  if (_fit) {
    var rotateMetadata = new THREE.Vector3(
      THREE.MathUtils.radToDeg(_helperObjects[0].rotation.x),
      THREE.MathUtils.radToDeg(_helperObjects[0].rotation.y),
      THREE.MathUtils.radToDeg(_helperObjects[0].rotation.z)
    );
    originalMetadata = {
      objPosition: [object.position.x, object.position.y, object.position.z],
      objRotation: [rotateMetadata.x, rotateMetadata.y, rotateMetadata.z],
      objScale: [
        _helperObjects[0].scale.x,
        _helperObjects[0].scale.y,
        _helperObjects[0].scale.z,
      ],
      cameraPosition: [core.camera.position.x, core.camera.position.y, core.camera.position.z],
      controlsTarget: [core.controls.target.x, core.controls.target.y, core.controls.target.z],
    };
  }
  setupClippingPlanes(object, core.gridSize, {x: boundingBox.max.x*1.1, y: boundingBox.max.y*1.1, z: boundingBox.max.z*1.1});
}

function parseGradient(str) {
  // Match "radial-gradient" or "linear-gradient"
  const typeMatch = str.match(/(radial|linear)-gradient\s*\(([^,]+)/i);
  const gradientType = typeMatch ? typeMatch[1].toLowerCase() : null;
  const shapeOrDirection = typeMatch ? typeMatch[2].trim() : null;

  // Match all rgb(...) values
  const colors = [...str.matchAll(/rgb\((\d+),\s*(\d+),\s*(\d+)\)/g)]
    .map(([, r, g, b]) => ({
      r: +r,
      g: +g,
      b: +b,
    }));

  return {
    type: gradientType,       // "radial" or "linear"
    shapeOrDirection,         // e.g. "circle" or "to right"
    colors,                   // array of { r, g, b }
  };
}

function changeBackgroundHelper(_color1, _color2) {
  core.mainCanvas.style.setProperty(
    "background",
    "-moz-radial-gradient(circle, " + _color1 + " 0%, " + _color2 + " 100%)"
  );
  core.mainCanvas.style.setProperty(
    "background",
    "-webkit-radial-gradient(circle, " + _color1 + " 0%, " + _color2 + " 100%)"
  );
  core.mainCanvas.style.setProperty(
    "background",
    "radial-gradient(circle, " + _color1 + " 0%, " + _color2 + " 100%)"
  );
}

function changeBackground(_type, _color1, _color2) {
  switch (_type) {
    case "linear":
      changeBackgroundHelper(_color1, _color1);
      break;
    case "gradient":
    case "radial":
      changeBackgroundHelper(_color1, _color2);
      break;
  }
}

function setupClippingPlanes(_geom, _size, _distance) {
  /*var _geometry;
  if (_geom.isGroup)
    _geometry = _geom.children;
  else
    _geometry = _geom.geometry.clone();*/
  core.clippingPlanes[0].constant = _distance.x;
  core.clippingPlanes[1].constant = _distance.y;
  core.clippingPlanes[2].constant = _distance.z;

  core.scene.add(core.transformControlClippingPlaneX.getHelper());
  core.scene.add(core.transformControlClippingPlaneY.getHelper());
  core.scene.add(core.transformControlClippingPlaneZ.getHelper());
  let planeColor = new THREE.Color(0xffffff).getHexString();
  if (core.scene.background != null) planeColor = core.scene.background.getHexString();

  core.planeHelpers = core.clippingPlanes.map(
    (p) => new THREE.PlaneHelper(p, _size * 2, invertHexColor(planeColor))
  );
  core.planeHelpers.forEach((ph) => {
    ph.visible = false;
    ph.name = "PlaneHelper";
    core.scene.add(ph);
  });

  core.distanceGeometry = _distance;
  let displayHelper = {x: getOrAddGuiController(core.clippingFolder, core.planeParams.planeX, "displayHelperX"), constantX: getOrAddGuiController(core.clippingFolder, core.planeParams.planeX, "constantX"), y: getOrAddGuiController(core.clippingFolder, core.planeParams.planeY, "displayHelperY"), constantY: getOrAddGuiController(core.clippingFolder, core.planeParams.planeY, "constantY"), z: getOrAddGuiController(core.clippingFolder, core.planeParams.planeZ, "displayHelperZ"), constantZ: getOrAddGuiController(core.clippingFolder, core.planeParams.planeZ, "constantZ"), outline: getOrAddGuiController(core.clippingFolder, core.planeParams.outline, "visible")};
  displayHelper.x.onChange((v) => {
      core.planeParams.clippingMode.x = core.planeHelpers[0].visible = v;
      if (v) {
        core.transformControlClippingPlaneX.attach(core.planeHelpers[0]);
        if (core.planeParams.outline.visible) outlineClipping.visible = true;
      } else {
        core.transformControlClippingPlaneX.detach();
        if (
          !core.planeParams.clippingMode.y &&
          !core.planeParams.clippingMode.z &&
          !core.planeParams.outline.visible
        )
          outlineClipping.visible = false;
      }
    });

    displayHelper.constantX
      .min(-core.distanceGeometry.x)
      .max(core.distanceGeometry.x)
      .setValue(core.distanceGeometry.x)
      .step(_size / 100)
      .listen()
      .onChange((d) => (core.clippingPlanes[0].constant = d));

    displayHelper.y.onChange((v) => {
      core.planeParams.clippingMode.y = core.planeHelpers[1].visible = v;
      if (v) {
        core.transformControlClippingPlaneY.attach(core.planeHelpers[1]);
        if (core.planeParams.outline.visible) outlineClipping.visible = true;
      } else {
        core.transformControlClippingPlaneY.detach();
        if (
          !core.planeParams.clippingMode.x &&
          !core.planeParams.clippingMode.z &&
          !core.planeParams.outline.visible
        )
          outlineClipping.visible = false;
      }
    });
    displayHelper.constantY
      .min(-core.distanceGeometry.y)
      .max(core.distanceGeometry.y)
      .setValue(core.distanceGeometry.y)
      .step(_size / 100)
      .listen()
      .onChange((d) => (core.clippingPlanes[1].constant = d));
  
    displayHelper.z.onChange((v) => {
      core.planeParams.clippingMode.z = core.planeHelpers[2].visible = v;
      if (v) {
        core.transformControlClippingPlaneZ.attach(core.planeHelpers[2]);
        if (core.planeParams.outline.visible) outlineClipping.visible = true;
      } else {
        core.transformControlClippingPlaneZ.detach();
        if (
          !core.planeParams.clippingMode.x &&
          !core.planeParams.clippingMode.y &&
          !core.planeParams.outline.visible
        )
          outlineClipping.visible = false;
      }
    });
    displayHelper.constantZ
      .min(-core.distanceGeometry.z)
      .max(core.distanceGeometry.z)
      .setValue(core.distanceGeometry.z)
      .step(_size / 100)
      .listen()
      .onChange((d) => (core.clippingPlanes[2].constant = d));

    displayHelper.outline.onChange((v) => {
      outlineClipping.visible = v;
    });
}


// Color helpers
export function invertHexColor(hexTripletColor) {
  let color = hexTripletColor.substring(1);
  color = parseInt(color, 16);
  color = 0xffffff ^ color;
  color = color.toString(16);
  color = ("000000" + color).slice(-6);
  return "#" + color;
}

export function getOrAddGuiController(folder, object, prop) {
  let controller = folder.controllers.find(c => c._name === prop);
  if (controller) return controller;

  for (const subfolder of folder.folders) {
    const found = getOrAddController(subfolder, object, prop);
    if (found) return found;
  }
  return folder.add(object, prop);
}