import pandas as pd
import sys
import os

def drupal_to_dntrade(csv_file, xls_file):
    # Читаем CSV с разделителем ';'
    df = pd.read_csv(csv_file, delimiter=';', encoding='utf-8-sig')
    
    # Получаем имя группы из названия файла (без расширения)
    file_name = os.path.basename(csv_file)
    group_name = os.path.splitext(file_name)[0]  # "Взуття.csv" → "Взуття"
    
    # Создаём список для результирующих строк
    dntrade_rows = []
    
    # Переменные для хранения текущего родительского товара
    current_parent = None
    
    for idx, row in df.iterrows():
        # Проверяем, является ли строка родительским товаром
        if pd.notna(row['ID(product)']):
            # ========== РОДИТЕЛЬСКИЙ ТОВАР (базовая строка) ==========
            # Определяем статус для родителя из поля "Немає у наявності"
            try:
                field_available = int(row['Немає у наявності'])
                parent_status = "Активний" if field_available == 0 else "Деактивований"
            except:
                parent_status = "Активний"
            
            # Создаём запись родительского товара (основную)
            parent_row = {
                "Код": int(row['ID(product)']),
                "Ім'я товару": row['Заголовок'],
                "Артикул": str(row['ID(product)']).strip(),
                "Група товарів": group_name,
                "Категорія": row['Категорія'],
                "Ціна": "",
                "Ваговий товар": "ні",
                "Модифікації": "так",
                "Код товару модифікації": "",
                "**Розмір": "",  # Исправлено: двойная звездочка
                "Статус": parent_status
            }
            dntrade_rows.append(parent_row)
            
            current_parent = parent_row
            
            # ========== ПЕРВАЯ ВАРИАЦИЯ ИЗ СТРОКИ РОДИТЕЛЯ ==========
            # Если в строке родителя есть данные вариации
            if pd.notna(row['ID(variation)']):
                # Берем последнее поле "Опубліковано(status)" - для вариации
                # Это будет самое правое значение в строке
                variation_status_val = None
                
                # Ищем значение статуса вариации (последнее в строке)
                # Преобразуем строку в словарь и ищем ключи с нужным названием
                row_dict = row.to_dict()
                # Находим все ключи с "Опубліковано(status)"
                status_keys = [k for k in row_dict.keys() if 'Опубліковано(status)' in str(k)]
                
                if status_keys:
                    # Берем последний ключ (самый правый в CSV)
                    last_status_key = status_keys[-1]
                    try:
                        variation_status_val = int(row[last_status_key])
                    except:
                        variation_status_val = 1  # По умолчанию
                else:
                    variation_status_val = 1
                
                variation_status = "Активний" if variation_status_val == 1 else "Деактивований"
                
                # Создаём запись вариации из строки родителя
                variation_row = {
                    "Код": "",
                    "Ім'я товару": "",
                    "Артикул": str(row['SKU']).strip() if pd.notna(row.get('SKU')) else "",
                    "Група товарів": group_name,
                    "Категорія": "",
                    "Ціна": float(row['Price']) if pd.notna(row.get('Price')) else "",
                    "Ваговий товар": "ні",
                    "Модифікації": "",
                    "Код товару модифікації": current_parent['Код'],
                    "**Розмір": str(row['Розмір:']).strip() if pd.notna(row.get('Розмір:')) else "",
                    "Статус": variation_status
                }
                dntrade_rows.append(variation_row)
        
        # ========== ДОПОЛНИТЕЛЬНЫЕ ВАРИАЦИИ ==========
        # Если это отдельная строка с вариацией
        elif pd.notna(row.get('ID(variation)')) and current_parent:
            # Берем последнее поле "Опубліковано(status)" для вариации
            variation_status_val = None
            
            # Ищем значение статуса вариации (последнее в строке)
            row_dict = row.to_dict()
            status_keys = [k for k in row_dict.keys() if 'Опубліковано(status)' in str(k)]
            
            if status_keys:
                last_status_key = status_keys[-1]
                try:
                    variation_status_val = int(row[last_status_key])
                except:
                    variation_status_val = 1
            else:
                variation_status_val = 1
            
            variation_status = "Активний" if variation_status_val == 1 else "Деактивований"
            
            # Создаём запись вариации
            variation_row = {
                "Код": "",
                "Ім'я товару": "",
                "Артикул": str(row['SKU']).strip() if pd.notna(row.get('SKU')) else "",
                "Група товарів": group_name,
                "Категорія": "",
                "Ціна": float(row['Price']) if pd.notna(row.get('Price')) else "",
                "Ваговий товар": "ні",
                "Модифікації": "",
                "Код товару модифікації": current_parent['Код'],
                "**Розмір": str(row['Розмір:']).strip() if pd.notna(row.get('Розмір:')) else "",
                "Статус": variation_status
            }
            dntrade_rows.append(variation_row)
    
    # Создаём DataFrame для DNTrade
    dntrade_df = pd.DataFrame(dntrade_rows)
    
    # Определяем порядок столбцов
    column_order = [
        "Код",
        "Ім'я товару", 
        "Артикул", 
        "Група товарів", 
        "Категорія",
        "Ціна", 
        "Ваговий товар", 
        "Модифікації", 
        "Код товару модифікації", 
        "**Розмір",  # Исправлено: двойная звездочка
        "Статус"
    ]
    
    # Переупорядочиваем столбцы
    dntrade_df = dntrade_df[column_order]
    
    # Сохраняем в Excel
    with pd.ExcelWriter(xls_file, engine='openpyxl') as writer:
        dntrade_df.to_excel(writer, sheet_name='Лист1', index=False)
    
    print(f"✅ Преобразование завершено! Файл сохранён как: {xls_file}")
    print(f"✅ Обработано строк: {len(dntrade_rows)}")
    print(f"✅ Група товарів взята из имени файла: {group_name}")

# Запуск скрипта
if __name__ == "__main__":
    # Укажите пути к файлам
    input_csv = "Взуття.csv"  # ваш CSV файл
    output_xls = "DNTrade_import.xlsx"  # выходной файл
    
    # Если файлы переданы как аргументы командной строки
    if len(sys.argv) >= 3:
        input_csv = sys.argv[1]
        output_xls = sys.argv[2]
    
    drupal_to_dntrade(input_csv, output_xls)
  