<?php

namespace Drupal\dntrade\Batch;

use Drupal\dntrade\DntradeSyncService;

/**
 * Batch processor for DNTrade synchronization.
 */
class DntradeBatch {
  
  /**
   * Process a batch operation for full sync.
   */
  public static function processFullSync(array &$context): void {
    if (!isset($context['sandbox']['offset'])) {
      $context['sandbox']['offset'] = 0;
      $context['sandbox']['total_processed'] = 0;
      $context['sandbox']['has_more'] = true;
      
      $context['results'] = [
        'processed' => 0,
        'created' => 0,
        'updated' => 0,
        'skipped' => 0,
        'failed' => 0,
        'processed_ids' => [],
      ];
    }
    
    /** @var \Drupal\dntrade\DntradeSyncService $syncService */
    $syncService = \Drupal::service('dntrade.sync_service');
    
    // Process one batch (100 products)
    $limit = 100;
    $offset = $context['sandbox']['offset'];
    
    try {
      // Fetch products
      $apiProducts = $syncService->fetchProducts($limit, $offset);
      
      if (empty($apiProducts)) {
        $context['sandbox']['has_more'] = false;
        $context['finished'] = 1;
        return;
      }
      
      // Process batch
      $batchResults = $syncService->processProductBatch($apiProducts);
      
      // Update context
      $context['sandbox']['offset'] += $limit;
      $context['sandbox']['total_processed'] += $batchResults['processed'];
      
      $context['results']['processed'] += $batchResults['processed'];
      $context['results']['created'] += $batchResults['created'];
      $context['results']['updated'] += $batchResults['updated'];
      $context['results']['skipped'] += $batchResults['skipped'];
      $context['results']['failed'] += $batchResults['failed'];
      $context['results']['processed_ids'] = array_merge(
        $context['results']['processed_ids'],
        $batchResults['processed_ids']
      );
      
      // Update progress message
      $context['message'] = t('Processed @total products so far...', [
        '@total' => $context['sandbox']['total_processed'],
      ]);
      
      // Estimate progress (we don't know total, so use arbitrary logic)
      $context['finished'] = $context['sandbox']['has_more'] ? 
        min(0.95, $context['sandbox']['total_processed'] / 10000) : 1;
        
    } catch (\Exception $e) {
      $context['results']['failed']++;
      \Drupal::logger('dntrade')->error('Batch processing error: @error', [
        '@error' => $e->getMessage(),
      ]);
      
      $context['message'] = t('Error processing batch: @error', [
        '@error' => $e->getMessage(),
      ]);
      
      // Continue despite error
      $context['sandbox']['offset'] += $limit;
    }
  }
  
  /**
   * Process a batch operation for incremental sync.
   */
  public static function processIncrementalSync(array &$context): void {
    if (!isset($context['sandbox']['offset'])) {
      $context['sandbox']['offset'] = 0;
      $context['sandbox']['total_processed'] = 0;
      $context['sandbox']['has_more'] = true;
      
      // Get last sync time
      $lastSyncTime = \Drupal::state()->get('dntrade.last_sync_time', 0);
      $context['sandbox']['modified_from'] = date('Y-m-d H:i:s', $lastSyncTime);
      
      $context['results'] = [
        'processed' => 0,
        'created' => 0,
        'updated' => 0,
        'skipped' => 0,
        'failed' => 0,
        'processed_ids' => [],
      ];
    }
    
    /** @var \Drupal\dntrade\DntradeSyncService $syncService */
    $syncService = \Drupal::service('dntrade.sync_service');
    
    // Process one batch (100 products)
    $limit = 100;
    $offset = $context['sandbox']['offset'];
    
    try {
      // Fetch modified products
      $apiProducts = $syncService->fetchModifiedProducts(
        $limit, 
        $offset, 
        $context['sandbox']['modified_from']
      );
      
      if (empty($apiProducts)) {
        $context['sandbox']['has_more'] = false;
        $context['finished'] = 1;
        return;
      }
      
      // Process batch
      $batchResults = $syncService->processProductBatch($apiProducts);
      
      // Update context
      $context['sandbox']['offset'] += $limit;
      $context['sandbox']['total_processed'] += $batchResults['processed'];
      
      $context['results']['processed'] += $batchResults['processed'];
      $context['results']['created'] += $batchResults['created'];
      $context['results']['updated'] += $batchResults['updated'];
      $context['results']['skipped'] += $batchResults['skipped'];
      $context['results']['failed'] += $batchResults['failed'];
      $context['results']['processed_ids'] = array_merge(
        $context['results']['processed_ids'],
        $batchResults['processed_ids']
      );
      
      // Update progress message
      $context['message'] = t('Processed @total modified products...', [
        '@total' => $context['sandbox']['total_processed'],
      ]);
      
      // Estimate progress
      $context['finished'] = $context['sandbox']['has_more'] ? 
        min(0.95, $context['sandbox']['total_processed'] / 1000) : 1;
        
    } catch (\Exception $e) {
      $context['results']['failed']++;
      \Drupal::logger('dntrade')->error('Incremental batch processing error: @error', [
        '@error' => $e->getMessage(),
      ]);
      
      $context['message'] = t('Error processing batch: @error', [
        '@error' => $e->getMessage(),
      ]);
      
      // Continue despite error
      $context['sandbox']['offset'] += $limit;
    }
  }
  
  /**
   * Batch finished callback for full sync.
   */
  public static function finishFullSync(bool $success, array $results, array $operations): void {
    if ($success) {
      /** @var \Drupal\dntrade\DntradeSyncService $syncService */
      $syncService = \Drupal::service('dntrade.sync_service');
      
      // Mark missing products as unavailable
      $availabilityResults = $syncService->markMissingProductsAsUnavailable($results['processed_ids']);
      
      // Merge results
      $finalResults = array_merge($results, $availabilityResults);
      
      // Log results
      $message = t(
        'Full sync completed: @processed processed, @created created, @updated updated, ' .
        '@skipped skipped, @failed failed. Availability: @available available, ' .
        '@unavailable unavailable. Variations: @unpublished unpublished, @published published.',
        [
          '@processed' => $finalResults['processed'],
          '@created' => $finalResults['created'],
          '@updated' => $finalResults['updated'],
          '@skipped' => $finalResults['skipped'],
          '@failed' => $finalResults['failed'],
          '@available' => $finalResults['marked_available'],
          '@unavailable' => $finalResults['marked_unavailable'],
          '@unpublished' => $finalResults['unpublished_variations'],
          '@published' => $finalResults['published_variations'],
        ]
      );
      
      \Drupal::messenger()->addStatus($message);
      \Drupal::logger('dntrade')->info($message);
      
    } else {
      $message = t('Full sync failed with errors.');
      \Drupal::messenger()->addError($message);
      \Drupal::logger('dntrade')->error($message);
    }
  }
  
  /**
   * Batch finished callback for incremental sync.
   */
  public static function finishIncrementalSync(bool $success, array $results, array $operations): void {
    if ($success) {
      // Update last sync time
      \Drupal::state()->set('dntrade.last_sync_time', time());
      
      // Log results
      $message = t(
        'Incremental sync completed: @processed processed, @created created, ' .
        '@updated updated, @skipped skipped, @failed failed.',
        [
          '@processed' => $results['processed'],
          '@created' => $results['created'],
          '@updated' => $results['updated'],
          '@skipped' => $results['skipped'],
          '@failed' => $results['failed'],
        ]
      );
      
      \Drupal::messenger()->addStatus($message);
      \Drupal::logger('dntrade')->info($message);
      
    } else {
      $message = t('Incremental sync failed with errors.');
      \Drupal::messenger()->addError($message);
      \Drupal::logger('dntrade')->error($message);
    }
  }
  
  /**
   * Initialize full sync batch.
   */
  public static function initFullSync(): array {
    $batch = [
      'title' => t('Synchronizing products from DNTrade (Full Sync)'),
      'operations' => [
        [[self::class, 'processFullSync'], []],
      ],
      'finished' => [self::class, 'finishFullSync'],
      'file' => \Drupal::service('extension.list.module')->getPath('dntrade') . '/src/Batch/DntradeBatch.php',
    ];
    
    return $batch;
  }
  
  /**
   * Initialize incremental sync batch.
   */
  public static function initIncrementalSync(): array {
    $batch = [
      'title' => t('Synchronizing modified products from DNTrade (Incremental Sync)'),
      'operations' => [
        [[self::class, 'processIncrementalSync'], []],
      ],
      'finished' => [self::class, 'finishIncrementalSync'],
      'file' => \Drupal::service('extension.list.module')->getPath('dntrade') . '/src/Batch/DntradeBatch.php',
    ];
    
    return $batch;
  }
}
