<?php

namespace Drupal\dntrade\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\dntrade\DntradeSyncService;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for DNTrade synchronization.
 */
class DntradeSyncController extends ControllerBase {
  
  /**
   * DNTrade sync service.
   *
   * @var \Drupal\dntrade\DntradeSyncService
   */
  protected $syncService;
  
  /**
   * Constructor.
   */
  public function __construct(DntradeSyncService $syncService) {
    $this->syncService = $syncService;
  }
  
  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('dntrade.sync_service')
    );
  }
  
  /**
   * Admin page for DNTrade synchronization.
   */
  public function adminPage() {
    $build = [];
    
    // Get sync status
    $status = $this->syncService->getSyncStatus();
    
    /** @var \Drupal\dntrade\SyncStatus $syncStatus */
    $syncStatus = \Drupal::service('dntrade.sync_status');
    $syncEnabled = $syncStatus->isEnabled();
    
    // Sync Status header
    $build['sync_status_header'] = [
      '#type' => 'details',
      '#title' => $this->t('Synchronization Status'),
      '#open' => TRUE,
    ];
    
    if ($syncEnabled) {
      $build['sync_status_header']['status'] = [
        '#markup' => '<div class="messages messages--status">' . 
          '<h3>✓ ' . $this->t('Synchronization is ENABLED') . '</h3>' .
          '<p>' . $this->t('Automatic and manual synchronizations are allowed.') . '</p>' .
          '</div>',
      ];
      
      $build['sync_status_header']['disable'] = [
        '#type' => 'link',
        '#title' => $this->t('Disable Synchronization'),
        '#url' => \Drupal\Core\Url::fromRoute('dntrade.sync_disable'),
        '#attributes' => [
          'class' => ['button', 'button--danger'],
        ],
      ];
    } else {
      $build['sync_status_header']['status'] = [
        '#markup' => '<div class="messages messages--warning">' . 
          '<h3>✗ ' . $this->t('Synchronization is DISABLED') . '</h3>' .
          '<p>' . $this->t('Automatic synchronizations are paused. Manual sync may still work.') . '</p>' .
          '</div>',
      ];
      
      $build['sync_status_header']['enable'] = [
        '#type' => 'link',
        '#title' => $this->t('Enable Synchronization'),
        '#url' => \Drupal\Core\Url::fromRoute('dntrade.sync_enable'),
        '#attributes' => [
          'class' => ['button', 'button--primary'],
        ],
      ];
    }
    
    // API Configuration status
    $config = $this->config('dntrade.settings');
    $apiConfigured = !empty($config->get('base_uri')) && !empty($config->get('api_key'));
    
    $build['api_status'] = [
      '#type' => 'details',
      '#title' => $this->t('API Configuration Status'),
      '#open' => TRUE,
    ];
    
    if ($apiConfigured) {
      $build['api_status']['status'] = [
        '#markup' => '<p class="text-success">✓ ' . $this->t('API is configured.') . '</p>',
      ];
      
      // Test connection button
      $build['api_status']['test'] = [
        '#type' => 'link',
        '#title' => $this->t('Test Connection'),
        '#url' => \Drupal\Core\Url::fromRoute('dntrade.sync_test'),
        '#attributes' => [
          'class' => ['button'],
        ],
      ];
    } else {
      $build['api_status']['status'] = [
        '#markup' => '<p class="text-error">✗ ' . $this->t('API is not configured.') . '</p>',
      ];
      
      $build['api_status']['configure'] = [
        '#type' => 'link',
        '#title' => $this->t('Configure API Settings'),
        '#url' => \Drupal\Core\Url::fromRoute('dntrade.settings_form'),
        '#attributes' => [
          'class' => ['button', 'button--primary'],
        ],
      ];
    }
    
    // Status information
    $build['status'] = [
      '#type' => 'details',
      '#title' => $this->t('Synchronization Status'),
      '#open' => TRUE,
    ];
    
    $items = [
      $this->t('Last sync: @time', ['@time' => $status['last_sync_human']]),
      $this->t('Log file: @size', ['@size' => $this->formatBytes($status['log_file_size'])]),
    ];
    
    if (!empty($status['last_sync_results'])) {
      $results = $status['last_sync_results'];
      $items[] = $this->t('Last sync results: @processed processed, @created created, @updated updated', [
        '@processed' => $results['processed'] ?? 0,
        '@created' => $results['created'] ?? 0,
        '@updated' => $results['updated'] ?? 0,
      ]);
    }
    
    $build['status']['list'] = [
      '#theme' => 'item_list',
      '#items' => $items,
    ];
    
    // Schedule information
    try {
      /** @var \Drupal\dntrade\DntradeCron $cron */
      $cron = \Drupal::service('dntrade.cron');
      $schedule = $cron->getScheduleInfo();
      
      $build['schedule'] = [
        '#type' => 'details',
        '#title' => $this->t('Synchronization Schedule'),
        '#open' => FALSE,
      ];
      
      $schedule_items = [
        $this->t('Full sync interval: @interval', ['@interval' => $this->formatInterval($schedule['full_sync_interval'])]),
        $this->t('Last full sync: @time', ['@time' => $schedule['last_full_sync_human']]),
        $this->t('Next full sync: @time', ['@time' => $schedule['next_full_sync_human']]),
        $this->t('Incremental sync interval: @interval', ['@interval' => $this->formatInterval($schedule['incremental_sync_interval'])]),
        $this->t('Last incremental sync: @time', ['@time' => $schedule['last_incremental_sync_human']]),
        $this->t('Next incremental sync: @time', ['@time' => $schedule['next_incremental_sync_human']]),
      ];
      
      $build['schedule']['list'] = [
        '#theme' => 'item_list',
        '#items' => $schedule_items,
      ];
    } catch (\Exception $e) {
      // Service might not be available
    }
    
    // Action buttons
    $build['actions'] = [
      '#type' => 'details',
      '#title' => $this->t('Synchronization Actions'),
      '#open' => TRUE,
    ];
    
    $build['actions']['full_sync'] = [
      '#type' => 'link',
      '#title' => $this->t('Run Full Synchronization'),
      '#url' => \Drupal\Core\Url::fromRoute('dntrade.sync_full'),
      '#attributes' => [
        'class' => ['button', 'button--primary'],
      ],
    ];
    
    $build['actions']['incremental_sync'] = [
      '#type' => 'link',
      '#title' => $this->t('Run Incremental Synchronization'),
      '#url' => \Drupal\Core\Url::fromRoute('dntrade.sync_incremental'),
      '#attributes' => [
        'class' => ['button'],
      ],
    ];
    
    $build['actions']['clear_state'] = [
      '#type' => 'link',
      '#title' => $this->t('Clear Sync State'),
      '#url' => \Drupal\Core\Url::fromRoute('dntrade.sync_clear_state'),
      '#attributes' => [
        'class' => ['button', 'button--danger'],
      ],
    ];
    
    // Quick links
    $build['quick_links'] = [
      '#type' => 'details',
      '#title' => $this->t('Configuration Links'),
      '#open' => FALSE,
    ];
    
    $build['quick_links']['api_settings'] = [
      '#type' => 'link',
      '#title' => $this->t('API Settings'),
      '#url' => \Drupal\Core\Url::fromRoute('dntrade.settings_form'),
      '#attributes' => [
        'class' => ['button'],
      ],
    ];
    
    $build['quick_links']['sync_settings'] = [
      '#type' => 'link',
      '#title' => $this->t('Sync Settings'),
      '#url' => \Drupal\Core\Url::fromRoute('dntrade.sync_settings_form'),
      '#attributes' => [
        'class' => ['button'],
      ],
    ];
    
    return $build;
  }
    
  /**
   * Run full synchronization.
   */
  public function runFullSync() {
    batch_set(\Drupal\dntrade\Batch\DntradeBatch::initFullSync());
    return batch_process('/admin/config/services/dntrade');
  }
  
  /**
   * Run incremental synchronization.
   */
  public function runIncrementalSync() {
    batch_set(\Drupal\dntrade\Batch\DntradeBatch::initIncrementalSync());
    return batch_process('/admin/config/services/dntrade');
  }
  
  /**
   * Clear sync state.
   */
  public function clearSyncState() {
    $this->syncService->clearSyncState();
    $this->messenger()->addStatus($this->t('Sync state cleared.'));
    return $this->redirect('dntrade.admin');
  }
  
  /**
   * Test API connection.
   */
  public function testConnection() {
    $result = $this->syncService->testConnection();
    
    if ($result['success']) {
      $this->messenger()->addStatus($this->t('Connection test successful: @message', [
        '@message' => $result['message'],
      ]));
    } else {
      $this->messenger()->addError($this->t('Connection test failed: @message', [
        '@message' => $result['message'],
      ]));
    }
    
    return $this->redirect('dntrade.admin');
  }
  
  /**
   * API endpoint for external sync trigger (e.g., from cron).
   */
  public function apiSync(Request $request) {
    // Check for API key or other authentication
    $apiKey = $request->query->get('key');
    $config = $this->config('dntrade.settings');
    
    if ($apiKey !== $config->get('sync_api_key')) {
      return new JsonResponse(['error' => 'Access denied'], 403);
    }
    
    $syncType = $request->query->get('type', 'incremental');
    
    try {
      if ($syncType === 'full') {
        $results = $this->syncService->runFullSync();
      } else {
        $results = $this->syncService->runIncrementalSync();
      }
      
      return new JsonResponse([
        'success' => true,
        'type' => $syncType,
        'results' => $results,
        'timestamp' => time(),
      ]);
      
    } catch (\Exception $e) {
      return new JsonResponse([
        'success' => false,
        'error' => $e->getMessage(),
        'timestamp' => time(),
      ], 500);
    }
  }
  
  /**
   * API endpoint for sync status.
   */
  public function apiStatus(Request $request) {
    // Check for API key or other authentication
    $apiKey = $request->query->get('key');
    $config = $this->config('dntrade.settings');
    
    if ($apiKey !== $config->get('sync_api_key')) {
      return new JsonResponse(['error' => 'Access denied'], 403);
    }
    
    $status = $this->syncService->getSyncStatus();
    
    return new JsonResponse([
      'success' => true,
      'status' => $status,
      'timestamp' => time(),
    ]);
  }
  
  /**
   * Format interval in seconds to human readable.
   */
  private function formatInterval(int $seconds): string {
    if ($seconds < 60) {
      return "$seconds seconds";
    } elseif ($seconds < 3600) {
      $minutes = floor($seconds / 60);
      return "$minutes minutes";
    } elseif ($seconds < 86400) {
      $hours = floor($seconds / 3600);
      return "$hours hours";
    } else {
      $days = floor($seconds / 86400);
      return "$days days";
    }
  }
  
  /**
   * Format bytes to human readable.
   */
  private function formatBytes(int $bytes, int $precision = 2): string {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);
    $bytes /= pow(1024, $pow);
    
    return round($bytes, $precision) . ' ' . $units[$pow];
  }
  
  /**
   * Enable synchronization.
   */
  public function enableSync() {
    /** @var \Drupal\dntrade\SyncStatus $syncStatus */
    $syncStatus = \Drupal::service('dntrade.sync_status');
    $syncStatus->enableWithReason('Enabled via admin interface');
    
    $this->messenger()->addStatus($this->t('Synchronization has been enabled.'));
    return $this->redirect('dntrade.admin');
  }

  /**
   * Disable synchronization.
   */
  public function disableSync() {
    /** @var \Drupal\dntrade\SyncStatus $syncStatus */
    $syncStatus = \Drupal::service('dntrade.sync_status');
    $syncStatus->disableWithReason('Disabled via admin interface');
    
    $this->messenger()->addStatus($this->t('Synchronization has been disabled.'));
    return $this->redirect('dntrade.admin');
  }

  /**
   * Toggle synchronization status.
   */
  public function toggleSync() {
    /** @var \Drupal\dntrade\SyncStatus $syncStatus */
    $syncStatus = \Drupal::service('dntrade.sync_status');
    $newStatus = $syncStatus->toggle();
    
    if ($newStatus) {
      $this->messenger()->addStatus($this->t('Synchronization has been enabled.'));
    } else {
      $this->messenger()->addStatus($this->t('Synchronization has been disabled.'));
    }
    
    return $this->redirect('dntrade.admin');
  }
  
}
