<?php

namespace Drupal\dntrade;

use GuzzleHttp\Client;

/**
 * Class DntradeClient Provides client for interacting with the Dntrade API.
 *
 * @package Drupal\dntrade
 *
 * @method mixed getCenters(array $args)
 * @method mixed getSites(array $args)
 * @method mixed getActivities(array $args)
 * @method mixed getActivityTypes(array $args)
 * @method mixed getActivityOtherCategories(array $args)
 * @method mixed getFlexRegPrograms(array $args)
 * @method mixed getFlexRegProgramTypes(array $args)
 * @method mixed getMembershipPackages(array $args)
 * @method mixed getMembershipCategories(array $args)
 * @method mixed getActivityDetail(int $id)
 * @method mixed getStoresIds()
 * @method mixed getPartnersList()
 * @method mixed getOrdersStatus()
 * @method mixed createOrder(array $orderData)
 * @method mixed getProductsList(array $args)
 */
class DntradeClient extends Client implements DntradeClientInterface {

  /**
   * Settings.
   *
   * @var array
   */
  protected $apisettings;

  /**
   * DntradeClient constructor.
   *
   * @param array $config
   *   Guzzle client configuration.
   */
  public function __construct(array $config = []) {
    parent::__construct($config);
  }

  /**
   * Set API settings.
   *
   * @param array $apisettings
   *   The api config settings.
   */
  public function setApi(array $apisettings) {
    $this->apisettings = $apisettings;
  }

  /**
   * Wrapper for 'request' method.
   *
   * @param string $method
   *   HTTP Method.
   * @param string $uri
   *   Dntrade URI.
   * @param array $parameters
   *   Arguments.
   *
   * @return mixed
   *   Data from Dntrade.
   *
   * @throws \Drupal\dntrade\DntradeClientException
   */
  private function makeRequest($method, $uri, array $parameters = []) {
    try {
      $response = $this->request($method, $uri, $parameters);

      $statusCode = $response->getStatusCode();
      $body = $response->getBody();
      $contents = $body->getContents();

      // Логируем полный ответ для отладки
      \Drupal::logger('dntrade')->debug('Full API response - Status: @status, Body: @body', [
        '@status' => $statusCode,
        '@body' => $contents
      ]);

      if (200 != $statusCode) {
        throw new DntradeClientException(sprintf('Got non 200 response code for the uri %s. Status: %d, Body: %s', $uri, $statusCode, $contents));
      }

      if (!$body) {
        throw new DntradeClientException(sprintf('Failed to get response body for the uri %s.', $uri));
      }

      if (!$contents) {
        throw new DntradeClientException(sprintf('Failed to get body contents for the uri: %s.', $uri));
      }

      // ВРЕМЕННО: логируем сырой ответ для отладки
      // \Drupal::logger('dntrade')->debug('Raw API response: @response', ['@response' => $contents]);

      $object = json_decode($contents);

      // ВРЕМЕННО: логируем структуру объекта
      // \Drupal::logger('dntrade')->debug('Decoded object: @object', ['@object' => print_r($object, TRUE)]);
      // Проверяем статус ответа API
      if (isset($object->status) && $object->status != 1 && $object->status != 'success') {
        $errorMessage = isset($object->message) ? $object->message : 'Unknown API error';
        throw new DntradeClientException(sprintf('API returned error: %s', $errorMessage));
      }
      
      // Проверяем различные форматы ответа API
      if (isset($object->data)) {
        return $object->data;  // Основной формат: {"status":1, "data": [...]}
      }
      elseif (isset($object->body)) {
        return $object->body;
      }
      elseif (isset($object->result)) {
        return $object->result;
      }
      elseif (is_array($object)) {
        return $object;
      }
      elseif (is_object($object)) {
        return $object;
      }

      // Если ни один из ожидаемых форматов не найден, возвращаем как есть
      return $contents;

    }
    catch (\Exception $e) {
      throw new DntradeClientException(sprintf('Failed to make a request for uri %s with message %s.', $uri, $e->getMessage()));
    }
  }

  /**
   * Magic call method.
   *
   * @param string $method
   *   Method.
   * @param mixed $args
   *   Arguments.
   *
   * @return mixed
   *   Data.
   *
   * @throws DntradeClientException.
   */
  public function __call($method, $args) {
    if (!$this->apisettings) {
      throw new DntradeClientException('Please inject api settings using "$this->setAPI($apisettings)".');
    }

    $base_uri = $this->apisettings['base_uri'];
    
    // методы, предоставляемые этим сервисом
    switch ($method) {
      case 'makeRequest':
       throw new DntradeClientException(sprintf('Please, extend Dntrade client!', $method));

      case 'getStoresIds': // GET uuid склада
        return $this->makeRequest('get', $base_uri . 'products/stores');
      
      case 'getPartnersList': // GET контрагентов / покупателей
        return $this->makeRequest('get', $base_uri . 'partners/list');
      
      case 'getOrdersStatus': // GET список статусов заказов
        $request_params = [];
        if (!empty($args[0])) {
          $request_params['query'] = $args[0];
        }
        return $this->makeRequest('get', $base_uri . 'orders/statuslist', $request_params);

      case 'getProductsList': // POST геттер списка товаров
        $request_params = [];
        
        if (!empty($args[0])) {
          // Разделяем параметры: простые в query, фильтры в json body
          $simple_params = [];
          $filter_params = [];
          
          foreach ($args[0] as $key => $value) {
            // Простые параметры (limit, website_synch) идут в URL
            if (in_array($key, ['limit', 'offset', 'website_synch', 'store_id', 'code'])) {
              $simple_params[$key] = $value;
            } 
            // Фильтры (code, product_id и т.д.) идут в тело запроса
            else {
              $filter_params[$key] = $value;
            }
          }
          
          // Добавляем простые параметры в URL
          if (!empty($simple_params)) {
            $request_params['query'] = $simple_params;
          }
          
          // Добавляем фильтры в тело запроса
          if (!empty($filter_params)) {
            $request_params['json'] = $filter_params;
          }
        }
        
        return $this->makeRequest('post', $base_uri . 'products/list', $request_params);
        
      case 'createOrder': // POST создание заказа
        if (empty($args[0]) || !is_array($args[0])) {
          throw new DntradeClientException('Order data is required and must be an array.');
        }
        $orderData = $args[0];
        $request_params = [
          'json' => $orderData,
          'headers' => [
            'Content-Type' => 'application/json',
          ],
        ];
        return $this->makeRequest('post', $base_uri . 'orders/upload', $request_params);

      default:
        throw new DntradeClientException(sprintf('Method %s not implemented yet.', $method));
    }
  }

  /**
   * Retrieves details of an activity by its ID.
   *
   * @param int $id
   *   The ID of the activity to retrieve details for.
   *
   * @return mixed
   *   Activity details.
   *
   * @throws DntradeClientException
   */
  public function getActivityDetail(int $id) {
    if (!$this->apisettings) {
      throw new DntradeClientException('Please inject api settings using "$this->setAPI($apisettings)".');
    }

    $base_uri = $this->apisettings['base_uri'];
    return $this->makeRequest('get', $base_uri . 'activities/' . $id);
  }

  /**
   * Creates a new order.
   *
   * @param array $orderData
   *   Order data in the required format.
   *
   * @return mixed
   *   Response from the API.
   *
   * @throws DntradeClientException
   */
  public function createOrder(array $orderData) {
    return $this->__call('createOrder', [$orderData]);
  }

  /**
   * Gets products list.
   *
   * @param array $productsData
   *   Products filter data.
   *
   * @return mixed
   *   Products list.
   *
   * @throws DntradeClientException
   */
  public function getProductsList(array $productsData) {
    return $this->__call('getProductsList', [$productsData]);
  }

}
