<?php

namespace Drupal\dntrade;

use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Cron handler for DNTrade synchronization.
 * НЕ реализует CronInterface, т.к. вызывается через hook_cron()
 */
class DntradeCron {
  
  /**
   * DNTrade sync service.
   *
   * @var \Drupal\dntrade\DntradeSyncService
   */
  protected $syncService;
  
  /**
   * Logger.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;
  
  /**
   * State API.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;
  
  /**
   * Config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;
  
  /**
   * Sync status service.
   *
   * @var \Drupal\dntrade\SyncStatus
   */
  protected $syncStatus;
  
  /**
   * Constructor.
   */
  public function __construct(
    DntradeSyncService $syncService,
    LoggerChannelInterface $logger,
    StateInterface $state,
    ConfigFactoryInterface $configFactory,
    SyncStatus $syncStatus
  ) {
    $this->syncService = $syncService;
    $this->logger = $logger;
    $this->state = $state;
    $this->configFactory = $configFactory;
    $this->syncStatus = $syncStatus;
  }
  
  /**
   * Synchronize products based on schedule.
   * Вызывается из hook_cron(), а НЕ из Drupal Cron системы.
   */
  public function synchronizeProducts(): array {
    $config = $this->configFactory->get('dntrade.settings');
    
    // Check if sync is enabled
    if (!$this->syncStatus->shouldRun()) {
      $this->logger->debug('Synchronization skipped - disabled or not allowed');
      return [];
    }
    
    // Get sync intervals
    $fullSyncInterval = $config->get('full_sync_interval') ?? 86400; // 24 hours
    $incrementalSyncInterval = $config->get('incremental_sync_interval') ?? 900; // 15 minutes
    
    // Get last sync times
    $lastFullSync = $this->state->get('dntrade.last_full_sync', 0);
    $lastIncrementalSync = $this->state->get('dntrade.last_incremental_sync', 0);
    
    $currentTime = time();
    $results = [];
    
    // Check if we should run full sync
    $shouldRunFullSync = ($currentTime - $lastFullSync >= $fullSyncInterval);
    $shouldRunIncrementalSync = ($currentTime - $lastIncrementalSync >= $incrementalSyncInterval);
    
    // Log schedule info
    $this->logger->debug('Sync schedule check:', [
      'last_full_sync' => $lastFullSync,
      'full_interval' => $fullSyncInterval,
      'should_full' => $shouldRunFullSync,
      'last_inc_sync' => $lastIncrementalSync,
      'inc_interval' => $incrementalSyncInterval,
      'should_inc' => $shouldRunIncrementalSync,
    ]);
    
    // Run full sync if scheduled
    if ($shouldRunFullSync) {
      $this->logger->info('Starting scheduled full synchronization');
      $results['full_sync'] = $this->runFullSync();
      $this->state->set('dntrade.last_full_sync', $currentTime);
    }
    
    // Run incremental sync if scheduled
    if ($shouldRunIncrementalSync) {
      $this->logger->info('Starting scheduled incremental synchronization');
      $results['incremental_sync'] = $this->runIncrementalSync();
      $this->state->set('dntrade.last_incremental_sync', $currentTime);
    }
    
    // Log sync summary
    $this->logSyncSummary($results);
    
    return $results;
  }
  
  /**
   * Run full synchronization.
   */
  private function runFullSync(): array {
    $this->logger->info('Starting full synchronization from DNTrade');
    
    try {
      $results = $this->syncService->runFullSync();
      
      $this->logger->info('Full synchronization completed: @processed processed, @created created, @updated updated', [
        '@processed' => $results['processed'] ?? 0,
        '@created' => $results['created'] ?? 0,
        '@updated' => $results['updated'] ?? 0,
      ]);
      
      // Send notification if configured
      $this->sendNotification('full', $results);
      
      return $results;
      
    } catch (\Exception $e) {
      $this->logger->error('Full synchronization failed: @error', [
        '@error' => $e->getMessage(),
      ]);
      
      // Send error notification
      $this->sendErrorNotification('full', $e);
      
      return [
        'success' => false,
        'error' => $e->getMessage(),
      ];
    }
  }
  
  /**
   * Run incremental synchronization.
   */
  private function runIncrementalSync(): array {
    $this->logger->info('Starting incremental synchronization from DNTrade');
    
    try {
      $results = $this->syncService->runIncrementalSync();
      
      $this->logger->info('Incremental synchronization completed: @processed processed, @created created, @updated updated', [
        '@processed' => $results['processed'] ?? 0,
        '@created' => $results['created'] ?? 0,
        '@updated' => $results['updated'] ?? 0,
      ]);
      
      // Only send notification if there were changes
      if (($results['processed'] ?? 0) > 0) {
        $this->sendNotification('incremental', $results);
      }
      
      return $results;
      
    } catch (\Exception $e) {
      $this->logger->error('Incremental synchronization failed: @error', [
        '@error' => $e->getMessage(),
      ]);
      
      // Only send error notification for critical errors
      if (!$this->isTransientError($e)) {
        $this->sendErrorNotification('incremental', $e);
      }
      
      return [
        'success' => false,
        'error' => $e->getMessage(),
      ];
    }
  }
  
/**
   * Check if error is transient (rate limit, network issues).
   */
  private function isTransientError(\Exception $e): bool {
    $errorMessage = $e->getMessage();
    
    // Check for rate limiting errors
    if (strpos($errorMessage, '429') !== false ||
        strpos($errorMessage, 'rate limit') !== false ||
        strpos($errorMessage, 'too many requests') !== false) {
      return true;
    }
    
    // Check for network errors
    if (strpos($errorMessage, 'connection') !== false ||
        strpos($errorMessage, 'timeout') !== false ||
        strpos($errorMessage, 'network') !== false) {
      return true;
    }
    
    return false;
  }
  
  /**
   * Send success notification.
   */
  private function sendNotification(string $syncType, array $results): void {
    $config = $this->configFactory->get('dntrade.settings');
    
    // Check if notifications are enabled
    if (!$config->get('enable_notifications')) {
      return;
    }
    
    $recipients = $config->get('notification_emails');
    if (empty($recipients)) {
      return;
    }
    
    $recipients = explode(',', $recipients);
    
    // Prepare email
    $subject = $this->t('DNTrade @type Sync Completed', ['@type' => ucfirst($syncType)]);
    
    $message = $this->t('
DNTrade synchronization completed successfully.

Sync Type: @type
Time: @time

Results:
- Processed: @processed products
- Created: @created products
- Updated: @updated products
- Skipped: @skipped products
- Failed: @failed products

@availability
@variations

Log: @log_url
    ', [
      '@type' => $syncType,
      '@time' => date('Y-m-d H:i:s'),
      '@processed' => $results['processed'] ?? 0,
      '@created' => $results['created'] ?? 0,
      '@updated' => $results['updated'] ?? 0,
      '@skipped' => $results['skipped'] ?? 0,
      '@failed' => $results['failed'] ?? 0,
      '@availability' => isset($results['marked_available']) ? 
        sprintf("Availability: %d marked available, %d marked unavailable", 
          $results['marked_available'], $results['marked_unavailable']) : '',
      '@variations' => isset($results['unpublished_variations']) ?
        sprintf("Variations: %d unpublished, %d published",
          $results['unpublished_variations'], $results['published_variations']) : '',
      '@log_url' => \Drupal::request()->getSchemeAndHttpHost() . '/admin/reports/dblog?type=dntrade',
    ]);
    
    // Send email to each recipient
    foreach ($recipients as $email) {
      $email = trim($email);
      if (!empty($email) && filter_var($email, FILTER_VALIDATE_EMAIL)) {
        \Drupal::service('plugin.manager.mail')->mail(
          'dntrade',
          'sync_notification',
          $email,
          'en',
          [
            'subject' => $subject,
            'body' => $message,
          ]
        );
      }
    }
  }
  
  /**
   * Send error notification.
   */
  private function sendErrorNotification(string $syncType, \Exception $e): void {
    $config = $this->configFactory->get('dntrade.settings');
    
    // Check if error notifications are enabled
    if (!$config->get('enable_error_notifications')) {
      return;
    }
    
    $recipients = $config->get('error_notification_emails');
    if (empty($recipients)) {
      return;
    }
    
    $recipients = explode(',', $recipients);
    
    // Prepare email
    $subject = $this->t('DNTrade @type Sync Failed', ['@type' => ucfirst($syncType)]);
    
    $message = $this->t('
DNTrade synchronization failed.

Sync Type: @type
Time: @time
Error: @error

Stack trace:
@trace

Please check the logs for more details.
Log: @log_url
    ', [
      '@type' => $syncType,
      '@time' => date('Y-m-d H:i:s'),
      '@error' => $e->getMessage(),
      '@trace' => $e->getTraceAsString(),
      '@log_url' => \Drupal::request()->getSchemeAndHttpHost() . '/admin/reports/dblog?type=dntrade',
    ]);
    
    // Send email to each recipient
    foreach ($recipients as $email) {
      $email = trim($email);
      if (!empty($email) && filter_var($email, FILTER_VALIDATE_EMAIL)) {
        \Drupal::service('plugin.manager.mail')->mail(
          'dntrade',
          'error_notification',
          $email,
          'en',
          [
            'subject' => $subject,
            'body' => $message,
          ]
        );
      }
    }
  }
  
  /**
   * Log sync summary.
   */
  private function logSyncSummary(array $results): void {
    $summary = [];
    
    if (isset($results['full_sync'])) {
      $full = $results['full_sync'];
      if (isset($full['success']) && $full['success'] === false) {
        $summary[] = $this->t('Full sync failed: @error', ['@error' => $full['error']]);
      } else {
        $summary[] = $this->t('Full sync: @processed processed', [
          '@processed' => $full['processed'] ?? 0,
        ]);
      }
    }
    
    if (isset($results['incremental_sync'])) {
      $incremental = $results['incremental_sync'];
      if (isset($incremental['success']) && $incremental['success'] === false) {
        $summary[] = $this->t('Incremental sync failed: @error', ['@error' => $incremental['error']]);
      } else {
        $summary[] = $this->t('Incremental sync: @processed processed', [
          '@processed' => $incremental['processed'] ?? 0,
        ]);
      }
    }
    
    if (!empty($summary)) {
      $this->logger->info('Sync summary: ' . implode('; ', $summary));
    }
  }
  
  /**
   * Get sync schedule information.
   */
  public function getScheduleInfo(): array {
    $config = $this->configFactory->get('dntrade.settings');
  
    $lastFullSync = $this->state->get('dntrade.last_full_sync', 0);
    $lastIncrementalSync = $this->state->get('dntrade.last_incremental_sync', 0);
    
    $fullSyncInterval = $config->get('full_sync_interval') ?? 86400;
    $incrementalSyncInterval = $config->get('incremental_sync_interval') ?? 900;
    
    // Получаем последнюю полную синхронизацию из общего статуса
    $lastSyncStats = $this->state->get('dntrade.last_sync_stats', []);
    $lastFullSyncFromStats = $lastSyncStats['time'] ?? 0;
    
    // Используем последнее доступное время
    $effectiveLastFullSync = max($lastFullSync, $lastFullSyncFromStats);
    
    return [
      'last_full_sync' => $effectiveLastFullSync,
      'last_full_sync_human' => $effectiveLastFullSync ? date('Y-m-d H:i:s', $effectiveLastFullSync) : 'Never',
      'next_full_sync' => $effectiveLastFullSync + $fullSyncInterval,
      'next_full_sync_human' => date('Y-m-d H:i:s', $effectiveLastFullSync + $fullSyncInterval),
      'full_sync_interval' => $fullSyncInterval,
      'last_incremental_sync' => $lastIncrementalSync,
      'last_incremental_sync_human' => $lastIncrementalSync ? date('Y-m-d H:i:s', $lastIncrementalSync) : 'Never',
      'next_incremental_sync' => $lastIncrementalSync + $incrementalSyncInterval,
      'next_incremental_sync_human' => date('Y-m-d H:i:s', $lastIncrementalSync + $incrementalSyncInterval),
      'incremental_sync_interval' => $incrementalSyncInterval,
    ];
  }
  
  /**
   * Manually trigger synchronization.
   */
  public function triggerSync(string $type = 'incremental'): array {
    switch ($type) {
      case 'full':
        return $this->runFullSync();
        
      case 'incremental':
        return $this->runIncrementalSync();
        
      default:
        throw new \InvalidArgumentException('Invalid sync type: ' . $type);
    }
  }
  
  /**
   * Reset sync schedule.
   */
  public function resetSchedule(): void {
    $this->state->delete('dntrade.last_full_sync');
    $this->state->delete('dntrade.last_incremental_sync');
    $this->logger->info('Sync schedule reset');
  }

}
