<?php

namespace Drupal\dntrade;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\commerce_product\Entity\Product;
use Drupal\commerce_product\Entity\ProductVariation;
use Drupal\commerce_product\Entity\ProductAttribute;
use Drupal\commerce_product\Entity\ProductAttributeValue;
use Drupal\commerce_price\Price;
use Drupal\file\Entity\File;
use Symfony\Component\Yaml\Yaml;

/**
 * Product creation service for DNTrade API integration.
 */
class DntradeProductCreator {
  
  use StringTranslationTrait;

  /**
   * The DNTrade client.
   *
   * @var \Drupal\dntrade\DntradeClientInterface
   */
  protected $client;

  /**
   * Rate limiter service.
   *
   * @var \Drupal\dntrade\RateLimiter
   */
  protected $rateLimiter;

  /**
   * Array of DNTrade product IDs processed in current sync.
   *
   * @var array
   */
  protected $processedApiIds = [];

  /**
   * Constructor.
   */
  public function __construct(DntradeClientInterface $client) {
    $this->client = $client;
    $this->rateLimiter = new RateLimiter();
  }

  /**
   * Process a batch of products from API response.
   */
  public function processApiBatch(array $apiProducts, array &$context): void {
    // Group products by base product (variations grouping)
    $groupedProducts = $this->groupProductsByBase($apiProducts);
    
    foreach ($groupedProducts as $baseCode => $products) {
      $this->processProductGroup($baseCode, $products, $context);
    }
    
    // Collect processed API IDs for availability tracking
    foreach ($apiProducts as $product) {
      if (isset($product->code) && is_numeric($product->code)) {
        $this->processedApiIds[] = (int)$product->code;
      }
    }
  }

  /**
   * Group API products by base product code.
   */
  private function groupProductsByBase(array $apiProducts): array {
      $groups = [];
      
      // Сначала создаем массив всех товаров, сгруппированных по базовому названию
      $productsByBaseTitle = [];
      
      foreach ($apiProducts as $product) {
          if (empty($product->code)) {
              continue;
          }
          
          $baseTitle = $this->extractBaseTitle($product->title);
          
          if (!isset($productsByBaseTitle[$baseTitle])) {
              $productsByBaseTitle[$baseTitle] = [];
          }
          
          $productsByBaseTitle[$baseTitle][] = $product;
      }
      
      // Теперь группируем по коду базового товара
      foreach ($productsByBaseTitle as $baseTitle => $products) {
          // Если только один товар с таким названием
          if (count($products) === 1) {
              $product = reset($products);
              $groups[(int)$product->code] = [$product];
              continue;
          }
          
          // Если несколько товаров с одним названием
          // Ищем базовый товар (без скобок в названии и с максимальным code)
          $baseProduct = null;
          $variations = [];
          
          // Сначала пытаемся найти товар БЕЗ скобок
          foreach ($products as $product) {
              if (!$this->isVariation($product)) {
                  // Если нашли несколько без скобок, берем с меньшим code как базовый
                  if (!$baseProduct || (int)$product->code < (int)$baseProduct->code) {
                      $baseProduct = $product;
                  }
              } else {
                  $variations[] = $product;
              }
          }
          
          // Если не нашли без скобок, используем товар с наименьшим code
          if (!$baseProduct && !empty($products)) {
              usort($products, function($a, $b) {
                  return (int)$a->code - (int)$b->code;
              });
              $baseProduct = reset($products);
              // Убираем его из вариаций
              $variations = array_filter($products, function($p) use ($baseProduct) {
                  return $p->code !== $baseProduct->code;
              });
          }
          
          if ($baseProduct) {
              $groups[(int)$baseProduct->code] = array_merge([$baseProduct], $variations);
          }
      }
      
      // Отладочная информация
      foreach ($groups as $baseCode => $products) {
          $variationCount = count($products) - 1;
          \Drupal::logger('dntrade')->debug('Group @code: @title (@count товаров, @variations вариаций)', [
              '@code' => $baseCode,
              '@title' => !empty($products[0]) ? $this->extractBaseTitle($products[0]->title) : 'unknown',
              '@count' => count($products),
              '@variations' => $variationCount,
          ]);
          
          foreach ($products as $index => $product) {
              \Drupal::logger('dntrade')->debug('  [@index] @title (code: @code, variation: @isVariation)', [
                  '@index' => $index,
                  '@title' => $product->title,
                  '@code' => $product->code,
                  '@isVariation' => $this->isVariation($product) ? 'yes' : 'no',
              ]);
          }
      }
      
      return $groups;
  }

  /**
   * Determine base product code from API product data.
   */
  private function getBaseProductCode(\stdClass $product): int {
    
    // Если продукт является вариацией, ищем базовый продукт по названию
    if ($this->isVariation($product)) {
      // Try to extract base code from title or find by base title
      $baseTitle = $this->extractBaseTitle($product->title);
      
      // Look for existing product with this base title and dntrade code
      /*$existingProducts = $this->findProductsByBaseTitle($baseTitle);
      
      if (!empty($existingProducts)) {
        $existingProduct = reset($existingProducts);
        // Get dntrade_code from first variation
        $variations = $existingProduct->getVariations();
        if (!$variations->isEmpty()) {
          $variation = $variations->first();
          if ($dntradeCode = $variation->get('field_dntrade_code')->value) {
            return (int)$dntradeCode;
          }
        }
      }*/
    }
    
    // If not a variation or base not found, use product's own code
    return (int)$product->code;
  }

  /**
   * Find products by base title that have dntrade code.
   */
  private function findProductsByBaseTitle(string $baseTitle): array {
    // Find all products with this base title
    $products = \Drupal::entityTypeManager()
      ->getStorage('commerce_product')
      ->loadByProperties(['title' => $baseTitle]);
    
    // Filter only products that have variations with dntrade code
    $filteredProducts = [];
    foreach ($products as $product) {
      $variations = $product->getVariations();
      foreach ($variations as $variation) {
        if ($variation->get('field_dntrade_code')->value) {
          $filteredProducts[] = $product;
          break;
        }
      }
    }
    
    return $filteredProducts;
  }

  /**
   * Check if product is a variation.
   */
  private function isVariation(\stdClass $product): bool {
    // Variation has attributes in parentheses in title
    $hasParentheses = preg_match('/\s*\([^)]+\)/', $product->title);
    // Также проверяем, что это не просто пустые скобки
    if ($hasParentheses) {
      preg_match('/\((.*?)\)/', $product->title, $matches);
      return !empty($matches[1]);
    }
    return false;
  }

  /**
   * Check if product should be treated as base product for variations.
   */
  private function isBaseForVariations(\stdClass $product, array $allProducts): bool {
      $baseTitle = $this->extractBaseTitle($product->title);
      
      // Ищем другие товары с таким же базовым названием
      foreach ($allProducts as $otherProduct) {
          if ($otherProduct === $product) {
              continue;
          }
          
          $otherBaseTitle = $this->extractBaseTitle($otherProduct->title);
          
          if ($otherBaseTitle === $baseTitle) {
              // Нашли другой товар с таким же базовым названием
              // Если этот товар без скобок, а другой со скобками - это базовый товар
              if ($this->isVariation($otherProduct) && !$this->isVariation($product)) {
                  return true;
              }
          }
      }
      
      return false;
  }


  /**
   * Extract base title (without attributes in parentheses).
   */
  private function extractBaseTitle(string $title): string {
    // return trim(preg_replace('/\s*\([^)]*\)/', '', $title));
    // Удаляем все, что в скобках
    $baseTitle = trim(preg_replace('/\s*\([^)]*\)/', '', $title));
    
    // Также удаляем лишние пробелы
    $baseTitle = preg_replace('/\s+/', ' ', $baseTitle);
    
    return $baseTitle;
  }

  /**
   * Extract attributes from variation title - расширенные правила для размеров
   */
  private function extractAttributesFromTitle(string $title): array {
    $attributes = ['size' => null, 'color' => null];
    
    // Расширенная конфигурация правил для размеров
    $rules = [
      'size' => [
        'patterns' => [
          // Буквенные размеры
          '/^(XS|S|M|L|XL|XXL|XXXL|XLL|2XL|3XL|4XL)$/i',
          
          // Числовые размеры
          '/^\d{1,3}$/', // 36, 42, 100 и т.д.
          '/^\d{1,3}\s*л$/', // 20л
          '/^\d{1,3}\s*см$/i', // 100см
          
          // Дробные размеры и слэши
          // '/^\d{1,3}[\\\/]\d{1,3}$/', // 41/5, 42\5, 26\28
          // '/^\d{1,3}[\/\\\\]\d{1,3}$/', // 41/5, 42\5, 26\28
          '/^\d{1,3}[\\/]\d{1,3}$/', // используем \/ вместо [\/]
          // '/^\d{1,3}\s*[\\\/]\s*\d{1,3}$/', // с пробелами
          '/^\d{1,3}\s*[\/\\\\]\s*\d{1,3}$/', // с пробелами
    
          // Размеры с миллиметрами
          '/^\d{1,3}\s*\(\d{1,4}мм\)$/', // 36 (233мм)
          '/^\d{1,3}\(\d{1,4}мм\)$/', // 36(233мм)
          
          // UK/EU размеры
          // '/^UK\s*\d+(\.\d+)?\s*[\\\/]\s*EU\s*\d+(\s*\d+[\\\/]\d+)?$/i', // UK 6.5 / EU 40
          '/^UK\s*\d+(\.\d+)?\s*[\/\\\\]\s*EU\s*\d+(\s*\d+[\/\\\\]\d+)?$/i',
          // '/^EU\s*\d+(\s*\d+[\\\/]\d+)?\s*[\\\/]\s*UK\s*\d+(\.\d+)?$/i', // EU 40 / UK 6.5
          '/^EU\s*\d+(\s*\d+[\/\\\\]\d+)?\s*[\/\\\\]\s*UK\s*\d+(\.\d+)?$/i',
          
          // Комбинированные буквенные размеры
          // '/^(XS|S|M|L|XL|XXL|XXXL|2XL|3XL|4XL)[\\\/\\\](S|M|L|R)$/i', // S\S, M\R, XL\L
          '/^(XS|S|M|L|XL|XXL|XXXL|2XL|3XL|4XL)[\/\\\\](S|M|L|R)$/i', // S\S, M\R, XL\L
          // '/^(XS|S|M|L|XL|XXL|XXXL|2XL|3XL|4XL)[\\\/\\\](XS|S|M|L|XL|XXL|XXXL)$/i', // XS/S, S/M
          '/^(XS|S|M|L|XL|XXL|XXXL|2XL|3XL|4XL)[\/\\\\](XS|S|M|L|XL|XXL|XXXL)$/i', // XS/S, S/M
          
          // Длина
          '/^(long|regular|короткий|довгий)$/i',
        ],
        
        // Известные значения размеров из вашего файла
        'known_values' => [
          // Числовые
          '1', '20л', '36', '37', '38', '39', '40', '41', '42', '43', '44', 
          '45', '46', '47', '48', '55', '56', '57', '58', '59', '60', '61',
          
          // Дробные
          '41/5', '42\5', '43\5', '44\5',
          
          // Буквенные
          'XS', 'S', 'M', 'L', 'XL', '2XL', '3XL', '4XL', 'XLL',
          
          // С миллиметрами (полные строки)
          '36 (233мм)', '38(247мм)', '39(253мм)', '40(260мм)', '41(267мм)', 
          '42(273мм)', '43(280мм)', '44(287мм)', '45(293мм)', '46(300мм)', '47(307мм)',
          
          // UK/EU
          'UK 6.5 / EU 40', 'UK 7 / EU 40 2/3', 'UK 7.5 / EU 41 1/3', 
          'UK 8 / EU 42', 'UK 8.5 / EU 42 2/3', 'UK 9 / EU 43 1/3', 
          'UK 9.5 / EU 44', 'UK 10 / EU 44 2/3', 'UK 10.5 / EU 45 1/3', 
          'UK 11.5 / EU 46 2/3',
          
          // Комбинированные
          'S\S', 'M\S', 'L\S', 'XL\S', '2XL\S', '3XL\S', '4XL\S',
          'S\R', 'M\R', 'L\R', 'XL\R', '2XL\R', '3XL\R', '4XL\R',
          'XS\L', 'S\L', 'M\L', 'L\L', 'XL\L', '2XL\L', '3XL\L', '4XL\L',
          
          // Двойные
          'XS/S', 'S/M', 'M/L', 'L/XL', 'XL/2XL', 'S\M', 'L\XL',
          
          // Числовые комбинации
          '26\28', '26\30', '26\32', '28\30', '28\32', '28\34', '30\30',
          '30\32', '30\34', '30\36', '32\30', '32\32', '32\34', '32\36',
          '34\30', '34\32', '34\34', '34\36', '36\30', '36\32', '36\34',
          '36\36', '38\30', '38\32', '38\34', '38\36', '39\42', '40\30',
          '40\32', '40\34', '40\36', '42\30', '42\32', '42\36', '43\46',
          '44\32', '44\34', '44\36',
          
          // Длина
          'long', 'regular', 'Long', 'Regular',
        ],
        
        'validators' => [
          'is_numeric_size', // Пользовательский валидатор
          'is_size_with_slash',
          'is_size_with_backslash',
        ],
        
        'priority' => 1, // Высокий приоритет
      ],
      
      'color' => [
        'patterns' => [
          // Украинские/русские названия цветов
          '/[а-яїієґ]{2,}/iu',
          '/(ий$|ый$|ой$|а$|я$|евой$|овый$|елый$|ный$|овий$)/iu',
          
          // Английские названия цветов
          '/^(black|white|red|blue|green|yellow|brown|gray|pink|purple|orange|navy|beige)$/i',
          
          // Составные цвета
          '/^[а-яa-z]+\s*[-\s]*[а-яa-z]+$/iu', // темно-синий, light-blue
        ],
        
        'known_values' => [
          // Украинские
          'чорний', 'білий', 'червоний', 'синій', 'зелений', 'жовтий',
          'коричневий', 'сірий', 'рожевий', 'фіолетовий', 'помаранчевий',
          'блакитний', 'бірюзовий', 'бордовий', 'малиновий', 'оливковий',
          'салатовий', 'бірюзовий', 'кремовий', 'золотий', 'срібний',
          
          // Русские
          'черный', 'белый', 'красный', 'синий', 'зеленый', 'желтый',
          'коричневый', 'серый', 'розовый', 'фиолетовый', 'оранжевый',
          'голубой', 'бирюзовый', 'бордовый', 'малиновый', 'оливковый',
          'салатовый', 'кремовый', 'золотой', 'серебряный',
          
          // Дополнительные
          'хаки', 'бежевый', 'бежевий', 'терракотовий', 'терракотовый',
          'молочний', 'молочный', 'пісочний', 'песочный',
        ],
        
        'priority' => 2, // Средний приоритет
      ],
    ];
    
    if (preg_match('/\((.*?)\)/', $title, $matches)) {
      $attrs = explode(',', $matches[1]);
      $attrs = array_map('trim', $attrs);
      
      // Собираем атрибуты в порядке приоритета
      $collected = [];
      
      foreach ($attrs as $attr) {
        $type = $this->determineAttributeType($attr, $rules);
        
        if ($type && !isset($collected[$type])) {
          $collected[$type] = $attr;
          $attributes[$type] = $attr;
          
          \Drupal::logger('dntrade')->debug('Attribute "@attr" classified as @type for title: "@title"', [
            '@attr' => $attr,
            '@type' => $type,
            '@title' => $title,
          ]);
        } elseif ($type) {
          // Уже есть атрибут этого типа - логируем конфликт
          \Drupal::logger('dntrade')->warning(
            'Multiple @type attributes found: "@existing" and "@new" in title: "@title"',
            [
              '@type' => $type,
              '@existing' => $collected[$type],
              '@new' => $attr,
              '@title' => $title,
            ]
          );
        } else {
          // Неизвестный атрибут - пробуем умную догадку
          $guessedType = $this->guessAttributeType($attr);
          if ($guessedType && !isset($collected[$guessedType])) {
            $collected[$guessedType] = $attr;
            $attributes[$guessedType] = $attr;
            
            \Drupal::logger('dntrade')->info('Guessed attribute "@attr" as @type for title: "@title"', [
              '@attr' => $attr,
              '@type' => $guessedType,
              '@title' => $title,
            ]);
          } else {
            // Неизвестный атрибут
            \Drupal::logger('dntrade')->debug('Unknown attribute: "@attr" in title: "@title"', [
              '@attr' => $attr,
              '@title' => $title,
            ]);
          }
        }
      }
    }
    
    return $attributes;
  }

  /**
   * Determine attribute type based on rules
   */
  private function determineAttributeType(string $value, array $rules): ?string {
    // Проверяем в порядке приоритета
    $priorities = [];
    foreach ($rules as $type => $rule) {
      $priorities[$rule['priority'] ?? 999] = $type;
    }
    ksort($priorities);
    
    foreach ($priorities as $type) {
      $rule = $rules[$type];
      
      // 1. Проверяем известные значения (точное совпадение)
      foreach ($rule['known_values'] ?? [] as $known_value) {
        // Чувствительное к регистру сравнение для точных значений
        if ($value === $known_value) {
          return $type;
        }
        // Нечувствительное к регистру для удобства
        if (strcasecmp($value, $known_value) === 0) {
          return $type;
        }
      }
      
      // 2. Проверяем паттерны
      foreach ($rule['patterns'] ?? [] as $pattern) {
        if (preg_match($pattern, $value)) {
          return $type;
        }
      }
      
      // 3. Проверяем валидаторы
      foreach ($rule['validators'] ?? [] as $validator) {
        if ($this->runValidator($validator, $value)) {
          return $type;
        }
      }
    }
    
    return null;
  }

  /**
   * Run specific validator
   */
  private function runValidator(string $validator, string $value): bool {
    switch ($validator) {
      case 'is_numeric_size':
        // Проверяем, является ли числовым размером (только цифры)
        return preg_match('/^\d{1,3}$/', $value);
        
      case 'is_size_with_slash':
        // Проверяем размеры с косой чертой
        return preg_match('/^\d{1,3}\/\d{1,3}$/', $value);
        
      case 'is_size_with_backslash':
        // Проверяем размеры с обратной косой чертой
        return preg_match('/^\d{1,3}\\\d{1,3}$/', $value);
        
      case 'is_numeric':
        return is_numeric($value);
        
      case 'strlen_less_than_5':
        return strlen($value) <= 4;
        
      default:
        return false;
    }
  }

  /**
   * Умная догадка для атрибутов, не попавших под правила
   */
  private function guessAttributeType(string $value): ?string {
    // 1. Проверяем на размер (приоритет)
    
    // Числовое значение
    if (is_numeric($value)) {
      return 'size';
    }
    
    // Короткое значение (1-4 символа) без кириллицы
    if (strlen($value) <= 4 && !preg_match('/[а-яїієґ]/iu', $value)) {
      // Проверяем, не является ли это аббревиатурой цвета
      $colorAbbrs = ['blk', 'wht', 'red', 'blu', 'grn', 'ylw', 'brn', 'gry'];
      if (!in_array(strtolower($value), $colorAbbrs)) {
        return 'size';
      }
    }
    
    // Содержит цифры
    if (preg_match('/\d/', $value)) {
      return 'size';
    }
    
    // 2. Проверяем на цвет
    
    // Содержит кириллицу (скорее всего цвет)
    if (preg_match('/[а-яїієґ]{2,}/iu', $value)) {
      return 'color';
    }
    
    // Английские названия цветов (полные или сокращенные)
    $englishColors = [
      'black', 'white', 'red', 'blue', 'green', 'yellow', 'brown', 'gray',
      'pink', 'purple', 'orange', 'navy', 'beige', 'cream', 'gold', 'silver',
      'khaki', 'olive', 'maroon', 'burgundy', 'coral', 'turquoise', 'teal',
      'lime', 'magenta', 'violet', 'indigo', 'cyan',
    ];
    
    if (in_array(strtolower($value), $englishColors)) {
      return 'color';
    }
    
    // 3. Не удалось определить
    return null;
  }

  /**
   * Дополнительный метод для очистки и нормализации значений размеров
   * Без нормализации:
   * Размеры "S\R" и "S/R" будут разными атрибутами
   * "XL" и "Xl" будут разными размерами
   * Фильтрация по размерам будет работать некорректно
   */
  private function normalizeSizeValue(string $size): string {
    if (empty(trim($size))) {
      return $size;
    }
    
    $original = $size;
    
    // 1. Удаляем лишние пробелы
    $size = trim($size);
    
    // 2. Приводим к верхнему регистру для буквенных размеров
    $upperSize = strtoupper($size);
    
    // 3. Маппинг буквенных размеров
    $sizeMappings = [
      // Стандартизация форматов
      '2XL' => ['XXL', '2 XL', '2-XL'],
      '3XL' => ['XXXL', '3 XL', '3-XL', 'XLL'], // XLL → 3XL
      '4XL' => ['XXXXL', '4 XL', '4-XL'],
      
      // Приведение к единому регистру
      'XS' => ['xs', 'Xs', 'xS'],
      'S'  => ['s'],
      'M'  => ['m'],
      'L'  => ['l'],
      'XL' => ['xl', 'Xl', 'xL'],
    ];
    
    // Проверяем маппинг
    foreach ($sizeMappings as $standard => $variations) {
      if (in_array($upperSize, $variations) || $upperSize === $standard) {
        return $standard;
      }
    }
    
    // 4. Стандартизируем разделители
    $size = str_replace(['\\', '／', '¦', '|'], '/', $size); // Все разделители → /
    
    // 5. Удаляем лишние пробелы вокруг разделителей
    $size = preg_replace('/\s*\/\s*/', '/', $size);
    $size = preg_replace('/\s*-\s*/', '-', $size);
    
    // 6. UK/EU форматы
    if (preg_match('/uk\s*(\d+(\.\d+)?)\s*[\/]\s*eu\s*(\d+(\s*\d+\/\d+)?)/i', $size, $matches)) {
      return 'UK ' . $matches[1] . ' / EU ' . $matches[3];
    }
    
    // 7. Размеры с миллиметрами - убираем лишние пробелы
    if (preg_match('/(\d+)\s*\(\s*(\d+)мм\s*\)/', $size, $matches)) {
      return $matches[1] . ' (' . $matches[2] . 'мм)';
    }
    
    // 8. Числовые комбинации (талия/длина)
    if (preg_match('/^(\d+)\s*[\/]\s*(\d+)$/', $size, $matches)) {
      return $matches[1] . '/' . $matches[2];
    }
    
    // Логируем нормализацию
    if ($original !== $size) {
      \Drupal::logger('dntrade')->debug('Size normalized: "@original" → "@normalized"', [
        '@original' => $original,
        '@normalized' => $size,
      ]);
    }
    
    return $size;
  }

  /**
   * Process a group of products (base + variations).
   */
  private function processProductGroup(int $baseCode, array $products, array &$context): void {
    
    \Drupal::logger('dntrade')->debug('=== Processing product group ===');
    \Drupal::logger('dntrade')->debug('Base code: @code, Products in group: @count', [
        '@code' => $baseCode,
        '@count' => count($products),
    ]);
    
    // Определяем, есть ли в группе вариации
    $hasVariations = false;
    foreach ($products as $product) {
        if ($this->isVariation($product)) {
            $hasVariations = true;
            break;
        }
    }
    
    // Если есть вариации, создаем атрибуты
    if ($hasVariations) {
        \Drupal::logger('dntrade')->debug('Group has variations, creating attributes...');
        $this->createAttributesFromVariations($products);
    } else {
        \Drupal::logger('dntrade')->debug('Group has NO variations');
    }
    
    // 1. Найти базовый товар в группе
    $baseProduct = $this->findBaseProduct($products);
    
    if (!$baseProduct) {
      \Drupal::logger('dntrade')->warning('No base product found in group with code: @code', [
        '@code' => $baseCode,
      ]);
      return;
    }
    
    \Drupal::logger('dntrade')->debug('Base product found. Title: @title, Code: @code', [
      '@title' => $baseProduct->title,
      '@code' => $baseProduct->code,
    ]);
    
    // Check if product already exists by dntrade code
    $existingProduct = $this->findExistingProductByDntradeCode($baseProduct);
    
    if ($existingProduct) {
      // Обновляем существующий товар с field_dntrade_code
      \Drupal::logger('dntrade')->debug('Existing product found by dntrade code with pid: @id', [
        '@id' => $existingProduct->id(),
      ]);
      
      // Обновляем существующий товар с field_dntrade_code
      $result = $this->updateExistingProduct($existingProduct, $products);
    } else {
      // Ищем товар по SKU или названию
      $existingProduct = $this->findProductBySkuOrTitle($baseProduct);
      
      if ($existingProduct) {        
        // Нашли товар по SKU/названию - у него нет field_dntrade_code, но есть в DNTrade
        \Drupal::logger('dntrade')->debug('Existing product found by SKU/title: @id', [
          '@id' => $existingProduct->id(),
        ]);
        // Заполняем field_dntrade_code и обновляем
        $result = $this->updateAndLinkProduct($existingProduct, $products, $baseProduct);
      } else {
        // Не нашли - создаем новый товар
        \Drupal::logger('dntrade')->debug('Creating new product');
        $result = $this->createNewProduct($baseProduct, $products);
      }
    }
    
    // Update context results
    if ($result['status'] === 'success') {
      $context['results']['created']++;
    } elseif ($result['status'] === 'updated') {
      $context['results']['updated']++;
    } elseif ($result['status'] === 'skipped') {
      $context['results']['skipped']++;
    } else {
      $context['results']['failed']++;
      \Drupal::logger('dntrade')->error('Failed to process product group: @error', [
        '@error' => $result['message'],
      ]);
    }
  }
  
  /**
   * Find base product in a group.
   */
  private function findBaseProduct(array $products): ?\stdClass {
      // Сначала ищем товар без скобок
      $nonVariationProducts = [];
      foreach ($products as $product) {
          if (!$this->isVariation($product)) {
              $nonVariationProducts[] = $product;
          }
      }
      
      // Если нашли товары без скобок
      if (!empty($nonVariationProducts)) {
          // Если несколько, берем с наименьшим code
          usort($nonVariationProducts, function($a, $b) {
              return (int)$a->code - (int)$b->code;
          });
          return reset($nonVariationProducts);
      }
      
      // Если все товары с скобками, берем первый
      return !empty($products) ? reset($products) : null;
  }

  /**
   * Find existing Drupal product by DNTrade code.
   */
  private function findExistingProductByDntradeCode(\stdClass $apiProduct): ?Product {
    if (empty($apiProduct->code)) {
      return null;
    }
    
    // Try to find by dntrade code in variations
    $variations = \Drupal::entityTypeManager()
      ->getStorage('commerce_product_variation')
      ->loadByProperties(['field_dntrade_code' => (int)$apiProduct->code]);
    
    if (!empty($variations)) {
      $variation = reset($variations);
      return $variation->getProduct();
    }
    
    return null;
  }

  /**
   * Find product by SKU or title.
   */
  private function findProductBySkuOrTitle(\stdClass $apiProduct): ?Product {
    // First try by SKU
    if (!empty($apiProduct->sku)) {
      $variations = \Drupal::entityTypeManager()
        ->getStorage('commerce_product_variation')
        ->loadByProperties(['sku' => $apiProduct->sku]);
      
      if (!empty($variations)) {
        $variation = reset($variations);
        $product = $variation->getProduct();
        
        // Check if product already has dntrade code
        if ($product) {
          $hasDntradeCode = false;
          $productVariations = $product->getVariations();
          foreach ($productVariations as $pv) {
            if ($pv->get('field_dntrade_code')->value) {
              $hasDntradeCode = true;
              break;
            }
          }
          
          // Если у товара уже есть dntrade code, не связываем его с другим кодом
          if (!$hasDntradeCode) {
            return $product;
          }
        }
      }
    }
    
    // Try by title (base title for variations)
    /*$title = $this->isVariation($apiProduct) 
      ? $this->extractBaseTitle($apiProduct->title)
      : $apiProduct->title;*/
    $title = $this->extractBaseTitle($apiProduct->title);
    
    // Для товаров с вариациями ищем по базовому названию
    $products = \Drupal::entityTypeManager()
      ->getStorage('commerce_product')
      ->loadByProperties(['title' => $title]);
    
    foreach ($products as $product) {
      // Check if product already has dntrade code
      $hasDntradeCode = false;
      $productVariations = $product->getVariations();
      foreach ($productVariations as $variation) {
        if ($variation->get('field_dntrade_code')->value) {
          $hasDntradeCode = true;
          break;
        }
      }
      
      // Если у товара нет dntrade code, можно его использовать
      if (!$hasDntradeCode) {
        return $product;
      }
    }
    
    return null;
  }

  /**
   * Create new product with variations.
   */
  private function createNewProduct(\stdClass $baseProduct, array $allProducts): array {
    try {
       \Drupal::logger('dntrade')->debug('Creating new product. Base product title: @title, Total variations: @count', [
        '@title' => $baseProduct->title,
        '@count' => count($allProducts),
      ]);
      
      // Определяем, является ли это товаром с вариациями
      $hasVariations = count($allProducts) > 1;
      
      // Если это товар с вариациями, проверяем, является ли первый товар базовым
      if ($hasVariations && !$this->isVariation($baseProduct)) {
          // Это базовый товар для вариаций
          \Drupal::logger('dntrade')->debug('Creating product with variations. Base product: @title', [
              '@title' => $baseProduct->title,
          ]);
          
          // Создаем атрибуты из вариаций
          $this->createAttributesFromVariations($allProducts);
      }
      
      // Determine product and viariation type from category
      // $productType = $this->getProductTypeFromCategory($baseProduct->category);
      $typeInfo = $this->getProductTypeFromCategory($baseProduct->category);
      
      /*if ($productType === false) {
        return [
          'status' => 'skipped',
          'message' => $this->t('Product skipped because it belongs to "Різне" category.'),
        ];
      }*/
      if ($typeInfo['skip_reason'] !== null) {
        $this->logSkippedProduct($baseProduct, $typeInfo);
        return [
          'status' => 'skipped',
          'message' => $this->t('Product skipped: @reason', [
            '@reason' => $this->getSkipReasonMessage($typeInfo['skip_reason'], $typeInfo),
          ]),
        ];
      }
      
      $productType = $typeInfo['product_type'];
      $variationType = $typeInfo['variation_type'];
      
      // Create base product
      $product = Product::create([
        'uid' => 1,
        'type' => $productType,
        'title' => $this->extractBaseTitle($baseProduct->title),
        'body' => [
          'value' => $baseProduct->description ?? '',
          'format' => filter_default_format(),
        ],
        'stores' => [\Drupal\commerce_store\Entity\Store::load(1)],
      ]);
      
      // Set category if available
      if (!empty($baseProduct->category->title)) {
        
        /* $categoryTerm = $this->getOrCreateCategoryTerm(
          $baseProduct->category->title,  // Название категории (напр. "Кеди")
          $productType,
          $baseProduct->category->parent->title ?? ''   // Родительская категория (напр. "Взуття")
        );*/
        $categoryTerm = $this->findCategoryTerm(
          $baseProduct->category->title,  // Название категории (например "Кеди")
          $productType                    // Тип товара (определен из bundles.yml)
        );
        
        if ($categoryTerm) {
          $product->set('field_ref_cat', ['target_id' => $categoryTerm->id()]);
          \Drupal::logger('dntrade')->info('Linked product "@title" to category "@category" (ID: @id)', [
            '@title' => $product->getTitle(),
            '@category' => $categoryTerm->getName(),
            '@id' => $categoryTerm->id(),
          ]);
        }else {
          \Drupal::logger('dntrade')->warning('Category "@category" not found for product "@title". Skipping category link.', [
            '@category' => $baseProduct->category->title,
            '@title' => $product->getTitle(),
          ]);
        }
      }
      
      // Create variations using the correct variation type from bundles.yml
      $variations = [];
      foreach ($allProducts as $apiProduct) {
        
        /*\Drupal::logger('dntrade')->debug('Processing variation. Title: @title, SKU: @sku, Code: @code', [
          '@title' => $apiProduct->title,
          '@sku' => $apiProduct->sku ?? 'none',
          '@code' => $apiProduct->code ?? 'none',
        ]);*/
        
        // $variation = $this->createVariation($apiProduct, $productType);
        $variation = $this->createVariation($apiProduct, $variationType);
        if ($variation) {
         /* \Drupal::logger('dntrade')->debug('Variation created successfully. ID: @id', [
            '@id' => $variation->id(),
          ]);*/
          
          $variations[] = $variation;
          $product->addVariation($variation);
        } else {
          \Drupal::logger('drate')->error('Failed to create variation for product: @title', [
            '@title' => $apiProduct->title,
          ]);
        }
        
      }
      
      /*\Drupal::logger('dntrade')->debug('Total variations created: @count', [
        '@count' => count($variations),
      ]);*/
      
      if (empty($variations)) {
        return [
          'status' => 'error',
          'message' => $this->t('No valid variations created for product.'),
        ];
      }
      
      // Process images
      $this->processProductImages($product, $baseProduct);
      
      $product->save();
      
      // После сохранения проверим сколько вариаций у товара
      /*\Drupal::entityTypeManager()->getStorage('commerce_product')->resetCache([$product->id()]);
      $product = Product::load($product->id());
      $actualVariations = $product->getVariations();
      \Drupal::logger('dntrade')->debug('Product saved. Actual variations count: @count', [
        '@count' => count($actualVariations),
      ]);*/
      
      return [
        'status' => 'success',
        'message' => $this->t('Created product "@title" with @count variations.', [
          '@title' => $product->getTitle(),
          '@count' => count($variations),
        ]),
      ];
      
    } catch (\Exception $e) {
      \Drupal::logger('dntrade')->error('Error creating product: @error', [
        '@error' => $e->getMessage(),
      ]);
      return [
        'status' => 'error',
        'message' => $this->t('Error creating product: @error', [
          '@error' => $e->getMessage(),
        ]),
      ];
    }
  }
  
  /**
   * Create attributes from variations.
   */
  private function createAttributesFromVariations(array $allProducts): void {
      $sizeValues = [];
      $colorValues = [];
      
      // Собираем все атрибуты из всех вариаций
      foreach ($allProducts as $product) {
          if ($this->isVariation($product)) {
              $attributes = $this->extractAttributesFromTitle($product->title);
              
              if (!empty($attributes['size'])) {
                  $normalizedSize = $this->normalizeSizeValue($attributes['size']);
                  $sizeValues[$normalizedSize] = $normalizedSize;
              }
              
              if (!empty($attributes['color'])) {
                  $colorValues[$attributes['color']] = $attributes['color'];
              }
          }
      }
      
      // Создаем атрибуты размера
      if (!empty($sizeValues)) {
          foreach ($sizeValues as $sizeValue) {
              $this->ensureAttributeWithValue('size', $sizeValue);
          }
          \Drupal::logger('dntrade')->debug('Created size attributes: @values', [
              '@values' => implode(', ', $sizeValues),
          ]);
      }
      
      // Создаем атрибуты цвета
      if (!empty($colorValues)) {
          foreach ($colorValues as $colorValue) {
              $this->ensureAttributeWithValue('color', $colorValue);
          }
          \Drupal::logger('dntrade')->debug('Created color attributes: @values', [
              '@values' => implode(', ', $colorValues),
          ]);
      }
}

  /**
   * Log skipped product information.
   */
  private function logSkippedProduct(\stdClass $product, array $typeInfo): void {
    $context = [
      '@title' => $product->title,
      '@code' => $product->code ?? 'N/A',
      '@category' => $product->category->title ?? 'N/A',
      '@parent' => $product->category->parent->title ?? 'N/A',
      '@reason' => $typeInfo['skip_reason'],
    ];
    
    switch ($typeInfo['skip_reason']) {
      case 'category_is_rizne':
        \Drupal::logger('dntrade')->info('Product "@title" (code: @code) skipped - belongs to "Різне" category', $context);
        break;
        
      case 'category_not_mapped':
        $context['@parent_category'] = $typeInfo['category'] ?? 'N/A';
        \Drupal::logger('dntrade')->warning('Product "@title" (code: @code) skipped - parent category "@parent_category" not mapped in bundles.yml', $context);
        break;
        
      case 'product_type_not_exists':
        $context['@product_type'] = $typeInfo['product_type'] ?? 'N/A';
        \Drupal::logger('dntrade')->error('Product "@title" (code: @code) skipped - product type "@product_type" does not exist in Drupal', $context);
        break;
    }
  }

  /**
   * Get user-friendly skip reason message.
   */
  private function getSkipReasonMessage(string $reason, array $typeInfo): string {
    switch ($reason) {
      case 'category_is_rizne':
        return $this->t('Belongs to "Різне" category');
        
      case 'category_not_mapped':
        return $this->t('Parent category "@category" not mapped', [
          '@category' => $typeInfo['category'] ?? 'unknown',
        ]);
        
      case 'product_type_not_exists':
        return $this->t('Product type "@type" does not exist', [
          '@type' => $typeInfo['product_type'] ?? 'unknown',
        ]);
        
      default:
        return $this->t('Unknown reason');
    }
  }
  
  /**
   * Update existing product with new variations (already has dntrade code).
   */
  private function updateExistingProduct(Product $product, array $apiProducts): array {
    try {
      $updatedCount = 0;
      $createdCount = 0;
      $linkedCount = 0;
      
      foreach ($apiProducts as $apiProduct) {
        $result = $this->createOrUpdateVariation($apiProduct, $product);
        
        if ($result['status'] === 'created') {
          $createdCount++;
        } elseif ($result['status'] === 'updated') {
          $updatedCount++;
        } elseif ($result['status'] === 'linked') {
          $linkedCount++;
        }
      }
      
      // Update product images
      $baseProduct = $this->findBaseProduct($apiProducts);
      if ($baseProduct) {
        $this->processProductImages($product, $baseProduct);
      }
      $this->updateProductCategory($product, $baseProduct);
      
      $product->save();
      
      return [
        'status' => 'updated',
        'message' => $this->t('Updated product "@title": @created new, @updated updated, @linked linked variations.', [
          '@title' => $product->getTitle(),
          '@created' => $createdCount,
          '@updated' => $updatedCount,
          '@linked' => $linkedCount,
        ]),
      ];
      
    } catch (\Exception $e) {
      return [
        'status' => 'error',
        'message' => $this->t('Error updating product: @error', [
          '@error' => $e->getMessage(),
        ]),
      ];
    }
  }

  /**
   * Update product found by SKU/title and link it with dntrade code.
   * 
   * 
   */
  private function updateAndLinkProduct(Product $product, array $apiProducts, \stdClass $baseApiProduct): array {
    try {
      $updatedCount = 0;
      $createdCount = 0;
      $linkedCount = 0;
      
      // Обрабатываем ВАРИАЦИИ
      foreach ($apiProducts as $apiProduct) {
        $result = $this->linkOrCreateVariation($apiProduct, $product);
        
        if ($result['status'] === 'created') {
          $createdCount++;
        } elseif ($result['status'] === 'updated') {
          $updatedCount++;
        } elseif ($result['status'] === 'linked') {
          $linkedCount++;
        }
      }
      
      // Обновляем изображения
      if ($baseApiProduct) {
        $this->processProductImages($product, $baseApiProduct);
      }
      
      // связываем с категорией
      $this->updateProductCategory($product, $baseApiProduct);
      
      $product->save();
      
      return [
        'status' => 'updated', // Меняем на 'updated' вместо 'success'
        'message' => $this->t('Linked product "@title" with DNTrade: @created new, @updated updated, @linked linked variations.', [
          '@title' => $product->getTitle(),
          '@created' => $createdCount,
          '@updated' => $updatedCount,
          '@linked' => $linkedCount,
        ]),
      ];
      
    } catch (\Exception $e) {
      return [
        'status' => 'error',
        'message' => $this->t('Error linking product: @error', [
          '@error' => $e->getMessage(),
        ]),
      ];
    }
  }

  /**
   * Create or update variation for a product (already has dntrade code).
   */
  private function createOrUpdateVariation(\stdClass $apiProduct, Product $product): array {
    // Try to find existing variation by dntrade code
    if (!empty($apiProduct->code)) {
      $existingVariations = \Drupal::entityTypeManager()
        ->getStorage('commerce_product_variation')
        ->loadByProperties([
          'field_dntrade_code' => (int)$apiProduct->code,
          'product_id' => $product->id(),
        ]);
      
      if (!empty($existingVariations)) {
        $variation = reset($existingVariations);
        $this->updateVariation($variation, $apiProduct);
        return ['status' => 'updated'];
      }
    }
    
    // Try to find by SKU
    if (!empty($apiProduct->sku)) {
      $existingVariations = \Drupal::entityTypeManager()
        ->getStorage('commerce_product_variation')
        ->loadByProperties([
          'sku' => $apiProduct->sku,
          'product_id' => $product->id(),
        ]);
      
      if (!empty($existingVariations)) {
        $variation = reset($existingVariations);
        // Update dntrade code if not set
        if (empty($variation->get('field_dntrade_code')->value) && !empty($apiProduct->code)) {
          $variation->set('field_dntrade_code', (int)$apiProduct->code);
          $this->updateVariation($variation, $apiProduct);
          return ['status' => 'linked'];
        } else {
          $this->updateVariation($variation, $apiProduct);
          return ['status' => 'updated'];
        }
      }
    }
    
    // Create new variation with correct variation type
    $productType = $product->bundle();
    $variationType = $this->getVariationTypeForProductType($productType);
    
    $variation = $this->createVariation($apiProduct, $variationType);
    if ($variation) {
      $product->addVariation($variation);
      
      \Drupal::logger('dntrade')->debug(
        'Created variation with type: @type for product: @product (product type: @product_type)',
        [
          '@type' => $variationType,
          '@product' => $product->getTitle(),
          '@product_type' => $productType,
        ]
      );
      
      return ['status' => 'created'];
    }
    
    return ['status' => 'skipped'];
  }

  /**
   * Link or create variation for a product (found by SKU/title, needs dntrade code).
   */
  private function linkOrCreateVariation(\stdClass $apiProduct, Product $product): array {
    // Try to find by SKU first
    if (!empty($apiProduct->sku)) {
      $existingVariations = \Drupal::entityTypeManager()
        ->getStorage('commerce_product_variation')
        ->loadByProperties([
          'sku' => $apiProduct->sku,
          'product_id' => $product->id(),
        ]);
      
      if (!empty($existingVariations)) {
        $variation = reset($existingVariations);
        // Set dntrade code if not set
        if (empty($variation->get('field_dntrade_code')->value) && !empty($apiProduct->code)) {
          $variation->set('field_dntrade_code', (int)$apiProduct->code);
          $this->updateVariation($variation, $apiProduct);
          return ['status' => 'linked'];
        } else {
          $this->updateVariation($variation, $apiProduct);
          return ['status' => 'updated'];
        }
      }
    }
    
    // Try to find by title/attributes
    if ($this->isVariation($apiProduct)) {
      $attributes = $this->extractAttributesFromTitle($apiProduct->title);
      $variation = $this->findVariationByAttributes($product, $attributes);
      
      if ($variation) {
        // Set dntrade code if not set
        if (empty($variation->get('field_dntrade_code')->value) && !empty($apiProduct->code)) {
          $variation->set('field_dntrade_code', (int)$apiProduct->code);
          $this->updateVariation($variation, $apiProduct);
          return ['status' => 'linked'];
        } else {
          $this->updateVariation($variation, $apiProduct);
          return ['status' => 'updated'];
        }
      }
    }
    
    // Create new variation with correct variation type and dntrade code
    $productType = $product->bundle();
    $variationType = $this->getVariationTypeForProductType($productType);
    
    $variation = $this->createVariation($apiProduct, $variationType);
    if ($variation) {
      $product->addVariation($variation);
      
      \Drupal::logger('dntrade')->debug(
        'Linked variation with type: @type for product: @product',
        ['@type' => $variationType, '@product' => $product->getTitle()]
      );
      
      return ['status' => 'created'];
    }
    
    return ['status' => 'skipped'];
  }

  /**
   * Find variation by attributes (size, color).
   */
  private function findVariationByAttributes(Product $product, array $attributes): ?ProductVariation {
    $variations = $product->getVariations();
    
    foreach ($variations as $variation) {
      $match = true;
      
      // Check size
      if (!empty($attributes['size']) && $variation->hasField('attribute_size')) {
        $sizeValue = $variation->get('attribute_size')->entity;
        if (!$sizeValue || $sizeValue->getName() !== $attributes['size']) {
          $match = false;
        }
      }
      
      // Check color
      if (!empty($attributes['color']) && $variation->hasField('attribute_color')) {
        $colorValue = $variation->get('attribute_color')->entity;
        if (!$colorValue || $colorValue->getName() !== $attributes['color']) {
          $match = false;
        }
      }
      
      if ($match) {
        return $variation;
      }
    }
    
    return null;
  }

  /**
   * Create variation from API product data.
   */
  private function createVariation(\stdClass $apiProduct, string $variationType): ?ProductVariation {
    try {
       \Drupal::logger('dntrade')->debug('Creating variation for variation type: @type', [
        '@type' => $variationType,
      ]);
      
      $variationData = [
        'type' => $variationType, // Используем промаппленный в bundles.yml тип вариации
        'title' => $apiProduct->title,
        'sku' => $apiProduct->sku ?? '',
        'price' => $this->getPrice($apiProduct),
        'status' => ($apiProduct->website_synch == 1 && $apiProduct->status == 1) ? 1 : 0,
      ];
      
      // Add dntrade code
      if (!empty($apiProduct->code)) {
        $variationData['field_dntrade_code'] = (int)$apiProduct->code;
      }
      
      // Add attributes if variation
      if ($this->isVariation($apiProduct)) {
        $attributes = $this->extractAttributesFromTitle($apiProduct->title);
        
       /* \Drupal::logger('dntrade')->debug('Variation attributes detected: @attributes', [
          '@attributes' => json_encode($attributes),
        ]);*/
        
        if (!empty($attributes['size'])) {
          $attributeId = $this->ensureAttributeWithValue('size', $attributes['size']);
          if ($attributeId) {
            $variationData['attribute_size'] = $attributeId;
            /*\Drupal::logger('dntrade')->debug('Size attribute set: @id', [
              '@id' => $attributeId,
            ]);*/
          }
        }
        
        if (!empty($attributes['color'])) {
          $attributeId = $this->ensureAttributeWithValue('color', $attributes['color']);
          if ($attributeId) {
            $variationData['attribute_color'] = $attributeId;
            /*\Drupal::logger('dntrade')->debug('Color attribute set: @id', [
              '@id' => $attributeId,
            ]);*/
          }
        }
      }else {
        \Drupal::logger('dntrade')->debug('Not a variation');
      }
      
      // 2do: Add stock information if available
      // if (isset($apiProduct->balance)) {
        // $variationData['field_stock'] = (int)$apiProduct->balance;
      // }
      
      $variation = ProductVariation::create($variationData);
      $variation->save();
      
      /*\Drupal::logger('dntrade')->debug('Variation created. ID: @id', [
        '@id' => $variation->id(),
      ]);*/
      
      return $variation;
      
    } catch (\Exception $e) {
      \Drupal::logger('dntrade')->error('Error creating variation: @error', [
        '@error' => $e->getMessage(),
      ]);
      return null;
    }
  }

  /**
   * Update existing variation with API data.
   */
  private function updateVariation(ProductVariation $variation, \stdClass $apiProduct): void {
    try {
      // Update price if changed
      $newPrice = $this->getPrice($apiProduct);
      $currentPrice = $variation->getPrice();
      
      if (!$currentPrice->equals($newPrice)) {
        $variation->setPrice($newPrice);
        \Drupal::logger('dntrade')->info('Updated price for variation @sku: @old -> @new', [
          '@sku' => $variation->getSku(),
          '@old' => $currentPrice->getNumber(),
          '@new' => $newPrice->getNumber(),
        ]);
      }
      
      // Update status
      $currentStatus = $variation->isPublished();
      $newStatus = ($apiProduct->website_synch == 1 && $apiProduct->status == 1) ? 1 : 0;
      if ($currentStatus != $newStatus) {
        $variation->setPublished($newStatus);
        \Drupal::logger('dntrade')->info('Updated status for variation @sku: @old -> @new', [
          '@sku' => $variation->getSku(),
          '@old' => $currentStatus ? 'published' : 'unpublished',
          '@new' => $newStatus ? 'published' : 'unpublished',
        ]);
      }
      
      // 2do Update stock quantity
      /*if (isset($apiProduct->balance)) {
        $currentStock = (int)$variation->get('field_stock')->value;
        $newStock = (int)$apiProduct->balance;
        if ($currentStock != $newStock) {
          $variation->set('field_stock', $newStock);
          \Drupal::logger('dntrade')->info('Updated stock for variation @sku: @old -> @new', [
            '@sku' => $variation->getSku(),
            '@old' => $currentStock,
            '@new' => $newStock,
          ]);
        }
      }*/
      
      // Update title if changed
      $currentTitle = $variation->getTitle();
      if ($currentTitle != $apiProduct->title) {
        $variation->setTitle($apiProduct->title);
      }
      
      // Update SKU if changed and not empty
      if (!empty($apiProduct->sku) && $variation->getSku() != $apiProduct->sku) {
        $variation->setSku($apiProduct->sku);
      }
      
      // ОБНОВЛЯЕМ АТРИБУТЫ если это вариация
      if ($this->isVariation($apiProduct)) {
        $attributes = $this->extractAttributesFromTitle($apiProduct->title);
        
        // Обновляем размер
        if (!empty($attributes['size']) && $variation->hasField('attribute_size')) {
          $currentSize = $variation->get('attribute_size')->entity;
          $newSizeValue = $this->normalizeSizeValue($attributes['size']);
          
          // Проверяем, изменился ли размер
          if (!$currentSize || $currentSize->getName() !== $newSizeValue) {
            $attributeId = $this->ensureAttributeWithValue('size', $newSizeValue);
            if ($attributeId) {
              $variation->set('attribute_size', $attributeId);
              \Drupal::logger('dntrade')->info('Updated size for variation @sku: @old -> @new', [
                '@sku' => $variation->getSku(),
                '@old' => $currentSize ? $currentSize->getName() : 'none',
                '@new' => $newSizeValue,
              ]);
            }
          }
        }
        
        // Обновляем цвет
        if (!empty($attributes['color']) && $variation->hasField('attribute_color')) {
          $currentColor = $variation->get('attribute_color')->entity;
          
          // Проверяем, изменился ли цвет
          if (!$currentColor || $currentColor->getName() !== $attributes['color']) {
            $attributeId = $this->ensureAttributeWithValue('color', $attributes['color']);
            if ($attributeId) {
              $variation->set('attribute_color', $attributeId);
              \Drupal::logger('dntrade')->info('Updated color for variation @sku: @old -> @new', [
                '@sku' => $variation->getSku(),
                '@old' => $currentColor ? $currentColor->getName() : 'none',
                '@new' => $attributes['color'],
              ]);
            }
          }
        }
        
        // Если атрибуты были удалены в DNTrade (стала базовая вариация)
        // Удаляем атрибуты у вариации
        if (!$this->isVariation($apiProduct)) {
          if ($variation->hasField('attribute_size') && !$variation->get('attribute_size')->isEmpty()) {
            $variation->set('attribute_size', null);
          }
          if ($variation->hasField('attribute_color') && !$variation->get('attribute_color')->isEmpty()) {
            $variation->set('attribute_color', null);
          }
        }
      }
      
      $variation->save();
      
    } catch (\Exception $e) {
      \Drupal::logger('dntrade')->error('Error updating variation: @error', [
        '@error' => $e->getMessage(),
      ]);
    }
  }

  /**
   * Get price from API product data.
   */
  private function getPrice(\stdClass $apiProduct): Price {
    // Use retail price if available
    $priceValue = '0.00';
    $currency = 'UAH';
    
    if (isset($apiProduct->price) && is_numeric($apiProduct->price)) {
      $priceValue = (string)$apiProduct->price;
    }
    
    if (!empty($apiProduct->currency)) {
      $currency = $this->normalizeCurrency($apiProduct->currency);
    }
    
    return new Price($priceValue, $currency);
  }

  /**
   * Normalize currency code.
   */
  private function normalizeCurrency(string $currency): string {
    $currency = trim($currency);
    
    if (strpos($currency, 'грн') !== false) {
      return 'UAH';
    }
    
    if (strpos($currency, '$') !== false || strpos($currency, 'USD') !== false) {
      return 'USD';
    }
    
    if (strpos($currency, '€') !== false || strpos($currency, 'EUR') !== false) {
      return 'EUR';
    }
    
    return $currency;
  }

  /**
   * Determine product type from DNTrade category with validation.
   * Если тип вариации не указан, используется тип товара как fallback
   */
  private function getProductTypeFromCategory(\stdClass $category = null): ?array {
    if (!$category) {
      return [
        'product_type' => 'default',
        'variation_type' => 'default',
        'skip_reason' => null,
      ];
    }
    
    $categoryTitle = $category->title;
    // Skip "Різне" category
    if ($categoryTitle === 'Різне') {
      return [
        'product_type' => null,
        'variation_type' => null,
        'skip_reason' => 'category_is_rizne',
      ];
    }
    
    // Check parent category for mapping
    $parentTitle = $category->parent->title ?? '';
    
    $mappings = $this->getBundleMappings();
    
    // Check if parent category is mapped
    if (!isset($mappings['product_types'][$parentTitle])) {
      return [
        'product_type' => null,
        'variation_type' => null,
        'skip_reason' => 'category_not_mapped',
        'category' => $parentTitle,
      ];
    }
    
    $productType = $mappings['product_types'][$parentTitle];
    
    // Check if product type exists in Drupal
    $productTypeStorage = \Drupal::entityTypeManager()->getStorage('commerce_product_type');
    $existingType = $productTypeStorage->load($productType);
    
    if (!$existingType) {
      return [
        'product_type' => null,
        'variation_type' => null,
        'skip_reason' => 'product_type_not_exists',
        'product_type_name' => $productType,
      ];
    }
    
    // Get variation type - either specific mapping or fallback to product type
    $variationType = $mappings['variation_types'][$parentTitle] ?? $productType;
    
    // Check if variation type exists
    $variationTypeStorage = \Drupal::entityTypeManager()->getStorage('commerce_product_variation_type');
    $existingVariationType = $variationTypeStorage->load($variationType);
    
    if (!$existingVariationType) {
      \Drupal::logger('dntrade')->warning(
        'Variation type "@type" not found, falling back to product type',
        ['@type' => $variationType]
      );
      $variationType = $productType;
    }
    
    return [
      'product_type' => $productType,
      'variation_type' => $variationType,
      'skip_reason' => null,
    ];
  }
  
  /**
   * Get product bundle mapping with variation types.
   */
  private function getBundleMappings(): array {
    static $mappings = null;
    
    if ($mappings === null) {
      $module_path = \Drupal::service('extension.list.module')->getPath('dntrade');
      $file_path = DRUPAL_ROOT . '/' . $module_path . '/settings/bundles.yml';
      
      $defaultMappings = [
        'product_types' => [],
        'variation_types' => [],
      ];
      
      if (file_exists($file_path)) {
        $file_contents = file_get_contents($file_path);
        $parsedMappings = Yaml::parse($file_contents);
        
        // Ensure both arrays exist
        $mappings = [
          'product_types' => $parsedMappings['product_types'] ?? [],
          'variation_types' => $parsedMappings['variation_types'] ?? [],
        ];
      } else {
        $mappings = $defaultMappings;
      }
      
      // Log available mappings for debugging
      \Drupal::logger('dntrade')->debug('Loaded product type mappings: @count', [
        '@count' => count($mappings['product_types']),
      ]);
      \Drupal::logger('dntrade')->debug('Loaded variation type mappings: @count', [
        '@count' => count($mappings['variation_types']),
      ]);
    }
    
    return $mappings;
  }
  
  /**
   * Get variation type for a product type based on bundles.yml mapping.
   * Если тип вариации не указан, используется тип товара как fallback
   */
  private function getVariationTypeForProductType(string $productType): string {
    $mappings = $this->getBundleMappings();
    
    // Default to product type if no specific variation type is mapped
    $variationType = $productType;
    
    // Look for the product type in variation_types mapping
    // We need to find which parent category maps to this product type first
    foreach ($mappings['product_types'] as $parentCategory => $mappedProductType) {
      if ($mappedProductType === $productType) {
        // Found the parent category that maps to this product type
        // Now check if we have a specific variation type for this category
        if (isset($mappings['variation_types'][$parentCategory])) {
          $mappedVariationType = $mappings['variation_types'][$parentCategory];
          
          // Verify the variation type exists
          $variationTypeStorage = \Drupal::entityTypeManager()
            ->getStorage('commerce_product_variation_type');
          
          $existingType = $variationTypeStorage->load($mappedVariationType);
          if ($existingType) {
            
            // $variationType = $mappedVariationType;
           /* \Drupal::logger('dntrade')->debug(
              'Using mapped variation type: @variation_type for product type: @product_type (category: @category)',
              [
                '@variation_type' => $mappedVariationType,
                '@product_type' => $productType,
                '@category' => $parentCategory,
              ]
            );*/
            // Found valid mapped variation type
            return $mappedVariationType;
            
          } else {
            // Mapped variation type doesn't exist, fallback to product type
            \Drupal::logger('dntrade')->warning(
              'Mapped variation type "@mapped" not found, falling back to product type "@product_type"',
              ['@mapped' => $mappedVariationType, '@product_type' => $productType]
            );
            return $productType;
          }
        }
        break; // Found the mapping, no need to continue
      }
    }
    
    return $productType;
  }
    
  /**
   * Get mapped product type from bundles.yml.
   */
  private function getMappedProductType(string $productType): string {
    $module_path = \Drupal::service('extension.list.module')->getPath('dntrade');
    $file_path = DRUPAL_ROOT . '/' . $module_path . '/settings/bundles.yml';
    
    if (file_exists($file_path)) {
      $file_contents = file_get_contents($file_path);
      $mapping = Yaml::parse($file_contents);
      
      return $mapping[$productType] ?? $productType;
    }
    
    return $productType;
  }

  /**
   * Get or create category term.
   * 2do: not used
   */
  private function getOrCreateCategoryTerm(string $categoryName, string $productType, string $parentCategory = ''): ?\Drupal\taxonomy\Entity\Term {
    \Drupal::logger('dntrade')->debug('getOrCreateCategoryTerm called: category="@category", productType="@type", parent="@parent"', [
      '@category' => $categoryName,
      '@type' => $productType,
      '@parent' => $parentCategory,
    ]);
    
    if (empty($categoryName)) {
      return null;
    }
    
    // Get vocabulary from field configuration
    $field_config = \Drupal\field\Entity\FieldConfig::load('commerce_product.' . $productType . '.field_ref_cat');
    if (!$field_config) {
      return null;
    }
    
    $settings = $field_config->getSettings();
    $target_bundles = $settings['handler_settings']['target_bundles'] ?? [];
    
    if (empty($target_bundles)) {
      return null;
    }
    
    $vocabulary = reset($target_bundles);
    
    // Try to find existing term
    $terms = \Drupal::entityTypeManager()
      ->getStorage('taxonomy_term')
      ->loadByProperties([
        'name' => $categoryName,
        'vid' => $vocabulary,
      ]);
    
    if (!empty($terms)) {
      return reset($terms);
    }
    
    // Create new term
    try {
      $termData = [
        'vid' => $vocabulary,
        'name' => $categoryName,
      ];
      
      // Add parent if specified
      if (!empty($parentCategory)) {
        $parentTerms = \Drupal::entityTypeManager()
          ->getStorage('taxonomy_term')
          ->loadByProperties([
            'name' => $parentCategory,
            'vid' => $vocabulary,
          ]);
        
        if (!empty($parentTerms)) {
          $parentTerm = reset($parentTerms);
          $termData['parent'] = ['target_id' => $parentTerm->id()];
        }
      }
      
      $term = \Drupal\taxonomy\Entity\Term::create($termData);
      $term->save();
      
      if ($term) {
        \Drupal::logger('dntrade')->debug('Term found/created: ID=@id, Name="@name"', [
          '@id' => $term->id(),
          '@name' => $term->getName(),
        ]);
      } else {
        \Drupal::logger('dntrade')->warning('Term not created/found for category "@category"', [
          '@category' => $categoryName,
        ]);
      }
      
      return $term;
      
    } catch (\Exception $e) {
      \Drupal::logger('dntrade')->error('Error creating category term: @error', [
        '@error' => $e->getMessage(),
      ]);
      return null;
    }
  }
  /**
   * Find existing category term (DO NOT CREATE NEW).
   */
  private function findCategoryTerm(string $categoryName, string $productType): ?\Drupal\taxonomy\Entity\Term {
    if (empty($categoryName)) {
      return null;
    }
    
    \Drupal::logger('dntrade')->debug('Looking for category term: "@category" for product type "@type"', [
      '@category' => $categoryName,
      '@type' => $productType,
    ]);
    
    // Get vocabulary from field configuration
    $field_config = \Drupal\field\Entity\FieldConfig::load('commerce_product.' . $productType . '.field_ref_cat');
    if (!$field_config) {
      \Drupal::logger('dntrade')->warning('Field config not found for commerce_product.@type.field_ref_cat', [
        '@type' => $productType,
      ]);
      return null;
    }
    
    $settings = $field_config->getSettings();
    $target_bundles = $settings['handler_settings']['target_bundles'] ?? [];
    
    if (empty($target_bundles)) {
      \Drupal::logger('dntrade')->warning('No target bundles found for field_ref_cat on product type @type', [
        '@type' => $productType,
      ]);
      return null;
    }
    
    $vocabulary = reset($target_bundles);
    
    // Try to find existing term by name
    $terms = \Drupal::entityTypeManager()
      ->getStorage('taxonomy_term')
      ->loadByProperties([
        'name' => $categoryName,
        'vid' => $vocabulary,
      ]);
    
    if (!empty($terms)) {
      $term = reset($terms);
      \Drupal::logger('dntrade')->debug('Found category term: ID=@id, Name="@name"', [
        '@id' => $term->id(),
        '@name' => $term->getName(),
      ]);
      return $term;
    }
    
    // Also try case-insensitive search
    $query = \Drupal::entityTypeManager()->getStorage('taxonomy_term')->getQuery();
    $query->accessCheck(FALSE)
      ->condition('vid', $vocabulary)
      ->condition('name', $categoryName, '=');
    
    $tids = $query->execute();
    
    if (!empty($tids)) {
      $term = \Drupal\taxonomy\Entity\Term::load(reset($tids));
      \Drupal::logger('dntrade')->debug('Found category term (case-insensitive): ID=@id, Name="@name"', [
        '@id' => $term->id(),
        '@name' => $term->getName(),
      ]);
      return $term;
    }
    
    \Drupal::logger('dntrade')->warning('Category term "@category" not found in vocabulary "@vocab"', [
      '@category' => $categoryName,
      '@vocab' => $vocabulary,
    ]);
    
    return null;
  }
  private function updateProductCategory(Product $product, \stdClass $baseProduct): void {
    if (empty($baseProduct->category->title)) {
      return;
    }
    
    $productType = $product->bundle();
    $categoryTerm = $this->findCategoryTerm($baseProduct->category->title, $productType);
    
    if ($categoryTerm) {
      // Проверяем, не привязана ли уже эта категория
      $currentCategory = $product->get('field_ref_cat')->target_id;
      
      if ($currentCategory != $categoryTerm->id()) {
        $product->set('field_ref_cat', ['target_id' => $categoryTerm->id()]);
        \Drupal::logger('dntrade')->info('Updated category for product "@title" to "@category"', [
          '@title' => $product->getTitle(),
          '@category' => $categoryTerm->getName(),
        ]);
      }
    }
  }
  
  /**
   * Process product images from API data.
   */
  private function processProductImages(Product $product, \stdClass $apiProduct): void {
    if (empty($apiProduct->images) || !is_array($apiProduct->images)) {
      return;
    }
    
    $images = [];
    foreach ($apiProduct->images as $imageUrl) {
      if (empty($imageUrl)) {
        continue;
      }
      
      $file = $this->downloadAndCreateFile($imageUrl, $product->bundle());
      if ($file) {
        $images[] = ['target_id' => $file->id()];
      }
    }
    
    if (!empty($images)) {
      $product->set('field_prodphoto', $images);
    }
  }

  /**
   * Download image and create file entity.
   */
  private function downloadAndCreateFile(string $imageUrl, string $bundle): ?File {
    try {
      // Extract filename from URL
      $filename = basename(parse_url($imageUrl, PHP_URL_PATH));
      if (empty($filename)) {
        $filename = md5($imageUrl) . '.jpg';
      }
      
      // Check if file already exists
      $files = \Drupal::entityTypeManager()
        ->getStorage('file')
        ->loadByProperties(['filename' => $filename]);
      
      if (!empty($files)) {
        return reset($files);
      }
      
      // Get destination path from field settings
      $field_config = \Drupal::config('field.field.commerce_product.' . $bundle . '.field_prodphoto');
      $file_directory = $field_config->get('third_party_settings.filefield_paths.file_path.value') ?: 'prodimg-dntrade';
      $file_directory = str_replace("_", "", $file_directory);
      $uri_scheme = $field_config->get('settings.uri_scheme') ?: 'public';
      $destination = $uri_scheme . '://' . $file_directory . '/' . $filename;
      
      // Download file
      $client = \Drupal::httpClient();
      $response = $client->get($imageUrl);
      
      if ($response->getStatusCode() !== 200) {
        return null;
      }
      
      // Get the file contents
      $data = $response->getBody()->getContents();
      
      // Prepare directory
      $directory = dirname($destination);
      /** @var \Drupal\Core\File\FileSystemInterface $file_system */
      $file_system = \Drupal::service('file_system');
      $file_system->prepareDirectory($directory, \Drupal\Core\File\FileSystemInterface::CREATE_DIRECTORY | \Drupal\Core\File\FileSystemInterface::MODIFY_PERMISSIONS);
      
      // Save file
      $file_path = $file_system->saveData($data, $destination, \Drupal\Core\File\FileSystemInterface::EXISTS_RENAME);
      
      if (!$file_path) {
        return null;
      }
      
      // Get MIME type
      $mime_type = \Drupal::service('file.mime_type.guesser')->guessMimeType($file_path);
      
      // Create file entity
      $file = File::create([
        'uid' => 1,
        'filename' => basename($file_path),
        'uri' => $file_path,
        'status' => 1,
        'filemime' => $mime_type,
      ]);
      
      $file->save();
      return $file;
      
    } catch (\Exception $e) {
      \Drupal::logger('dntrade')->error('Error downloading image: @error', [
        '@error' => $e->getMessage(),
      ]);
      return null;
    }
  }
  
  /**
   * Ensure attribute with value exists.
   */
  private function ensureAttributeWithValue(string $attributeName, string $valueName, string $attributeLabel = null): ?int {
    try {
      
      // НОРМАЛИЗУЕМ значение перед созданием/поиском
      $normalizedValue = $valueName;
      
      if ($attributeName === 'size') {
        $normalizedValue = $this->normalizeSizeValue($valueName);
        
        \Drupal::logger('dntrade')->debug('Normalizing size: "@original" -> "@normalized"', [
          '@original' => $valueName,
          '@normalized' => $normalizedValue,
        ]);
      }
      
      $attribute_storage = \Drupal::entityTypeManager()->getStorage('commerce_product_attribute');
      $value_storage = \Drupal::entityTypeManager()->getStorage('commerce_product_attribute_value');
      
      if ($attributeLabel === null) {
        $attributeLabel = ucfirst(str_replace('_', ' ', $attributeName));
      }
      
      // Load or create attribute
      $attribute = $attribute_storage->load($attributeName);
      if (!$attribute) {
        $attribute = ProductAttribute::create([
          'id' => $attributeName,
          'label' => $attributeLabel,
        ]);
        $attribute->save();
      }
      
      // Load or create attribute value
      $existing_values = $value_storage->loadByProperties([
        'attribute' => $attributeName,
        'name' => $valueName,
      ]);
      
      if (!empty($existing_values)) {
        return reset($existing_values)->id();
      }
      
      $attribute_value = ProductAttributeValue::create([
        'attribute' => $attributeName,
        'name' => $valueName,
      ]);
      $attribute_value->save();
      
      return $attribute_value->id();
      
    } catch (\Exception $e) {
      \Drupal::logger('dntrade')->error('Error creating attribute @attr with value @value: @error', [
        '@attr' => $attributeName,
        '@value' => $valueName,
        '@error' => $e->getMessage(),
      ]);
      return null;
    }
  }

  /**
   * Mark missing products as unavailable based on DNTrade CRM data.
   * $processedApiIds - массив всех DNTrade ID (code), которые пришли в текущей синхронизации
   */
  public function markMissingProductsAsUnavailable(array $processedApiIds): array {
    $marked_unavailable = 0;
    $marked_available = 0;
    $unpublished_variations = 0;
    $published_variations = 0;
    
    try {
      // 1. Получаем все вариации, у которых есть field_dntrade_code
      $variationStorage = \Drupal::entityTypeManager()->getStorage('commerce_product_variation');
      $query = $variationStorage->getQuery()
        ->accessCheck(FALSE)
        ->condition('field_dntrade_code', 0, '>') // код > 0
        ->exists('field_dntrade_code');           // код существует (не NULL)
      
      $vids = $query->execute();
      $variations = $variationStorage->loadMultiple($vids);
      
      $productData = [];
      
      // 2. Обрабатываем каждую вариацию
      foreach ($variations as $variation) {
        $product = $variation->getProduct();
        $dntradeCode = $variation->get('field_dntrade_code')->value;
        
        // Проверяем, есть ли вариация в текущих данных от DNTrade
        $isInApi = in_array($dntradeCode, $processedApiIds);
        $isPublished = $variation->isPublished();
        
        // Обновляем статус публикации вариации
        if (!$isInApi && $isPublished) {
          // Вариации нет в CRM → снимаем с публикации
          $variation->setUnpublished();
          $variation->save();
          $unpublished_variations++;
        } elseif ($isInApi && !$isPublished) {
          // Вариация есть в CRM → публикуем
          $variation->setPublished();
          $variation->save();
          $published_variations++;
        }
        
        
        // 3. Теперь работаем с товарами (продуктами). Группируем данные по товарам для логики доступности
        if ($product) {
          $productId = $product->id();
          
          if (!isset($productData[$productId])) {
            $productData[$productId] = [
              'product' => $product,
              'has_variation_in_api' => false, // Есть ли хоть одна вариация в CRM
              'has_published_variation' => false, // Есть ли хоть одна опубликованная вариация
            ];
          }
          
          if ($isInApi) {
            $productData[$productId]['has_variation_in_api'] = true;
          }
          
          if ($variation->isPublished()) {
            $productData[$productId]['has_published_variation'] = true;
          }
        }
      }
      
      // 4. Обновляем доступность – field_available – товаров
      foreach ($productData as $data) {
        $product = $data['product'];
        
        if (!$product->hasField('field_available')) {
          continue;
        }
        
        $currentAvailability = (int)$product->get('field_available')->value;
        
        // Логика:
        // - has_variation_in_api = true → товар ЕСТЬ в CRM (field_available = 0)
        // - has_variation_in_api = false → товара НЕТ в CRM (field_available = 1)
        $shouldBeAvailableInCrm = $data['has_variation_in_api'];
        
        if ($shouldBeAvailableInCrm) {
          // Товар ЕСТЬ в CRM (есть хотя бы одна вариация)
          if ($currentAvailability != 0) {
            $product->set('field_available', 0); // Снимаем отметку "Немає у наявності"
            $product->save();
            $marked_available++;
          }
        } else {
          // Товара НЕТ в CRM (нет ни одной вариации)
          if ($currentAvailability != 1) {
            $product->set('field_available', 1); // Ставим отметку "Немає у наявності"
            $product->save();
            $marked_unavailable++;
          }
        }
        
        // Дополнительная проверка: если у товара нет опубликованных вариаций,
        // но он есть в CRM (например, все вариации отключены вручную),
        // всё равно отмечаем его как недоступный
        if (!$data['has_published_variation'] && $data['has_variation_in_api']) {
          if ($currentAvailability != 1) {
            $product->set('field_available', 1);
            $product->save();
          }
        }
      }
      
      // 5. Логируем результаты
      \Drupal::logger('dntrade')->info('Availability sync completed: @unavailable products marked unavailable, @available products marked available, @unpub variations unpublished, @pub variations published.', [
        '@unavailable' => $marked_unavailable,
        '@available' => $marked_available,
        '@unpub' => $unpublished_variations,
        '@pub' => $published_variations,
      ]);
      
    } catch (\Exception $e) {
      \Drupal::logger('dntrade')->error('Error marking missing products: @error', [
        '@error' => $e->getMessage(),
      ]);
    }
    
    return [
      'marked_unavailable' => $marked_unavailable,
      'marked_available' => $marked_available,
      'unpublished_variations' => $unpublished_variations,
      'published_variations' => $published_variations,
    ];
  }
  
  /**
   * Get processed API IDs for availability tracking.
   */
  public function getProcessedApiIds(): array {
    return $this->processedApiIds;
  }

  /**
   * Clear processed API IDs cache.
   */
  public function clearProcessedApiIds(): void {
    $this->processedApiIds = [];
  }

}
