<?php

namespace Drupal\dntrade;

use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Database\Connection;

/**
 * Service for managing DNTrade synchronization.
 */
class DntradeSyncService {
  
  /**
   * DNTrade client.
   *
   * @var \Drupal\dntrade\DntradeClientInterface
   */
  protected $client;
  
  /**
   * Product creator service.
   *
   * @var \Drupal\dntrade\DntradeProductCreator
   */
  protected $productCreator;
  
  /**
   * Logger.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;
  
  /**
   * State API.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;
  
  /**
   * Database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;
  
  /**
   * Rate limiter.
   *
   * @var \Drupal\dntrade\RateLimiter
   */
  protected $rateLimiter;
  
  /**
   * Sync log file path.
   *
   * @var string
   */
  protected $logFilePath;
  
  /**
   * Constructor.
   */
  public function __construct(
    DntradeClientInterface $client,
    DntradeProductCreator $productCreator,
    LoggerChannelInterface $logger,
    StateInterface $state,
    Connection $database,
    SyncStatus $syncStatus
  ) {
    $this->client = $client;
    $this->productCreator = $productCreator;
    $this->logger = $logger;
    $this->state = $state;
    $this->database = $database;
    $this->rateLimiter = new RateLimiter();
    $this->logFilePath = DRUPAL_ROOT . '/dntrade/dnt_posts.log';
    $this->syncStatus = $syncStatus;
  }
  
  /**
   * Run full synchronization.
   */
  public function runFullSync(): array {
    // Check if sync is enabled
    if (!$this->syncStatus->isEnabled()) {
      $this->logger->warning('Full sync attempted while synchronization is disabled');
      return [
        'success' => false,
        'error' => 'Synchronization is disabled',
        'skipped' => true,
      ];
    }

    $this->logSyncStart('FULL_SYNC');
    
    try {
      $results = [
        'processed' => 0,
        'created' => 0,
        'updated' => 0,
        'skipped' => 0,
        'failed' => 0,
        'processed_ids' => [],
      ];
      
      $offset = 0;
      $limit = 100;
      $hasMore = true;
      
      while ($hasMore) {
        $this->rateLimiter->check();
        
        // Fetch products from API
        $apiProducts = $this->fetchProducts($limit, $offset);
        
        if (empty($apiProducts)) {
          $hasMore = false;
          break;
        }
        
        // Process batch
        $batchResults = $this->processProductBatch($apiProducts);
        
        // Update overall results
        $results['processed'] += $batchResults['processed'];
        $results['created'] += $batchResults['created'];
        $results['updated'] += $batchResults['updated'];
        $results['skipped'] += $batchResults['skipped'];
        $results['failed'] += $batchResults['failed'];
        $results['processed_ids'] = array_merge(
          $results['processed_ids'],
          $batchResults['processed_ids']
        );
        
        // Check if we have more products
        if (count($apiProducts) < $limit) {
          $hasMore = false;
        } else {
          $offset += $limit;
        }
        
        // Log progress
        $this->logProgress($offset, $results);
      }
      
      // Mark missing products as unavailable
      $availabilityResults = $this->productCreator->markMissingProductsAsUnavailable($results['processed_ids']);
      
      // Save sync state
      $this->saveSyncState($results, $availabilityResults);
      
      // Log completion
      $this->logSyncComplete($results, $availabilityResults);
      
      return array_merge($results, $availabilityResults);
      
    } catch (\Exception $e) {
      $this->logger->error('Full sync failed: @error', ['@error' => $e->getMessage()]);
      $this->logToFile('ERROR: Full sync failed - ' . $e->getMessage());
      throw $e;
    }
  }
  
  /**
   * Run incremental synchronization.
   */
  public function runIncrementalSync(): array {
    // Check if sync is enabled
    if (!$this->syncStatus->isEnabled()) {
      $this->logger->warning('Full sync attempted while synchronization is disabled');
      return [
        'success' => false,
        'error' => 'Synchronization is disabled',
        'skipped' => true,
      ];
    }
    
    $this->logSyncStart('INCREMENTAL_SYNC');
    
    try {
      // Get last sync time
      $lastSyncTime = $this->state->get('dntrade.last_sync_time', 0);
      $modifiedFrom = date('Y-m-d H:i:s', $lastSyncTime);
      
      $results = [
        'processed' => 0,
        'created' => 0,
        'updated' => 0,
        'skipped' => 0,
        'failed' => 0,
        'processed_ids' => [],
      ];
      
      $offset = 0;
      $limit = 100;
      $hasMore = true;
      
      while ($hasMore) {
        $this->rateLimiter->check();
        
        // Fetch modified products from API
        $apiProducts = $this->fetchModifiedProducts($limit, $offset, $modifiedFrom);
        
        if (empty($apiProducts)) {
          $hasMore = false;
          break;
        }
        
        // Process batch
        $batchResults = $this->processProductBatch($apiProducts);
        
        // Update overall results
        $results['processed'] += $batchResults['processed'];
        $results['created'] += $batchResults['created'];
        $results['updated'] += $batchResults['updated'];
        $results['skipped'] += $batchResults['skipped'];
        $results['failed'] += $batchResults['failed'];
        $results['processed_ids'] = array_merge(
          $results['processed_ids'],
          $batchResults['processed_ids']
        );
        
        // Check if we have more products
        if (count($apiProducts) < $limit) {
          $hasMore = false;
        } else {
          $offset += $limit;
        }
        
        // Log progress
        $this->logProgress($offset, $results);
      }
      
      $this->saveSyncState($results, []);
      
      // Update last sync time
      $this->state->set('dntrade.last_sync_time', time());
      
      // Log completion
      $this->logSyncComplete($results, []);
      
      return $results;
      
    } catch (\Exception $e) {
      $this->logger->error('Incremental sync failed: @error', ['@error' => $e->getMessage()]);
      $this->logToFile('ERROR: Incremental sync failed - ' . $e->getMessage());
      throw $e;
    }
  }
  
  /**
   * Fetch products from API.
   */
  private function fetchProducts(int $limit, int $offset): array {
    try {
      $params = [
        'limit' => $limit,
        'offset' => $offset,
        'website_synch' => 1,
        'store_id' => '833a605c-fa32-46b6-9735-067239c68634', // uuid cклада: без него у товаров категории будут пустыми
      ];
      
      $response = $this->client->getProductsList($params);
      
      // Handle different response formats
      if (isset($response->products)) {
        return $response->products;
      } elseif (is_array($response)) {
        return $response;
      }
      
      return [];
      
    } catch (DntradeClientException $e) {
      // Check if it's a rate limit error
      if (strpos($e->getMessage(), '429') !== false) {
        $this->rateLimiter->handleRateLimitError();
        return $this->fetchProducts($limit, $offset); // Retry
      }
      
      throw $e;
    }
  }
  
  /**
   * Fetch modified products from API.
   */
  public function fetchModifiedProducts(int $limit, int $offset, string $modifiedFrom): array {
    try {
      $params = [
        'limit' => $limit,
        'offset' => $offset,
        'website_synch' => 1,
        'modified_from' => $modifiedFrom,
      ];
      
      $response = $this->client->getProductsList($params);
      
      // Handle different response formats
      if (isset($response->products)) {
        return $response->products;
      } elseif (is_array($response)) {
        return $response;
      }
      
      return [];
      
    } catch (DntradeClientException $e) {
      // Check if it's a rate limit error
      if (strpos($e->getMessage(), '429') !== false) {
        $this->rateLimiter->handleRateLimitError();
        return $this->fetchModifiedProducts($limit, $offset, $modifiedFrom); // Retry
      }
      
      throw $e;
    }
  }
  
  /**
   * Process a batch of products.
   */
  public function processProductBatch(array $apiProducts): array {
    $batchResults = [
      'processed' => count($apiProducts),
      'created' => 0,
      'updated' => 0,
      'skipped' => 0,
      'failed' => 0,
      'processed_ids' => [],
    ];
    
    $context = [
      'results' => [
        'created' => 0,
        'updated' => 0,
        'skipped' => 0,
        'failed' => 0,
      ],
    ];
    
    // Process products
    $this->productCreator->processApiBatch($apiProducts, $context);
    
    // Collect processed IDs
    foreach ($apiProducts as $product) {
      if (!empty($product->code)) {
        $batchResults['processed_ids'][] = (int)$product->code;
      }
    }
    
    // Update batch results
    $batchResults['created'] = $context['results']['created'];
    $batchResults['updated'] = $context['results']['updated'];
    $batchResults['skipped'] = $context['results']['skipped'];
    $batchResults['failed'] = $context['results']['failed'];
    
    return $batchResults;
  }
  
  /**
   * Log sync start.
   */
  private function logSyncStart(string $syncType): void {
    $this->logToFile(sprintf(
      "\n[--- %s START at %s ---]\n",
      $syncType,
      date('Y-m-d H:i:s')
    ));
    
    $this->logger->info('@type sync started', ['@type' => $syncType]);
  }
  
  /**
   * Log progress.
   */
  private function logProgress(int $offset, array $results): void {
    $message = sprintf(
      'Progress: offset %d, processed: %d, created: %d, updated: %d, skipped: %d, failed: %d',
      $offset,
      $results['processed'],
      $results['created'],
      $results['updated'],
      $results['skipped'],
      $results['failed']
    );
    
    $this->logToFile($message);
    
    // Log to Drupal logger less frequently
    if ($offset % 500 === 0) {
      $this->logger->info($message);
    }
  }
  
  /**
   * Log sync completion.
   */
  private function logSyncComplete(array $results, array $availabilityResults): void {
    $message = sprintf(
      "Sync completed: processed %d products, created %d, updated %d, skipped %d, failed %d\n" .
      "Availability: marked %d as available, %d as unavailable\n" .
      "Variations: unpublished %d, published %d\n" .
      "[--- SYNC END at %s ---]\n",
      $results['processed'],
      $results['created'],
      $results['updated'],
      $results['skipped'],
      $results['failed'],
      $availabilityResults['marked_available'] ?? 0,
      $availabilityResults['marked_unavailable'] ?? 0,
      $availabilityResults['unpublished_variations'] ?? 0,
      $availabilityResults['published_variations'] ?? 0,
      date('Y-m-d H:i:s')
    );
    
    $this->logToFile($message);
    
    $this->logger->info('Sync completed: @processed processed, @created created, @updated updated, @skipped skipped, @failed failed', [
      '@processed' => $results['processed'],
      '@created' => $results['created'],
      '@updated' => $results['updated'],
      '@skipped' => $results['skipped'],
      '@failed' => $results['failed'],
    ]);
  }
  
  /**
   * Save sync state.
   */
  private function saveSyncState(array $results, array $availabilityResults): void {
   /* $this->state->setMultiple([
      'dntrade.last_sync_time' => time(),
      'dntrade.last_sync_stats' => [
        'time' => time(),
        'results' => $results,
        'availability' => $availabilityResults,
      ],
    ]);*/
    $syncStats = [
      'time' => time(),
      'results' => $results,
      'availability' => $availabilityResults,
    ];
    
    $this->state->set('dntrade.last_sync_stats', $syncStats);
  }
  
  /**
   * Log to file.
   */
  private function logToFile(string $message): void {
    // Ensure directory exists
    $dir = dirname($this->logFilePath);
    if (!file_exists($dir)) {
      mkdir($dir, 0755, true);
    }
    
    // Rotate log if needed
    if (file_exists($this->logFilePath) && filesize($this->logFilePath) > 1048576) {
      $backup = $this->logFilePath . '.' . date('Y-m-d_H-i-s');
      rename($this->logFilePath, $backup);
    }
    
    // Write log
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[$timestamp] " . $message . "\n";
    file_put_contents($this->logFilePath, $logEntry, FILE_APPEND | LOCK_EX);
  }
  
  /**
   * Get sync status.
   */
  public function getSyncStatus(): array {
    $lastSync = $this->state->get('dntrade.last_sync_stats', []);
    
    // Безопасное получение времени
    $lastSyncTime = $lastSync['time'] ?? 0;
    
    return [
      'last_sync_time' => $lastSyncTime,
      'last_sync_human' => $lastSyncTime ? date('Y-m-d H:i:s', $lastSyncTime) : 'Never',
      'last_sync_results' => $lastSync['results'] ?? [],
      'log_file_exists' => file_exists($this->logFilePath),
      'log_file_size' => file_exists($this->logFilePath) ? filesize($this->logFilePath) : 0,
      'rate_limit_status' => [
        'requests_this_minute' => $this->rateLimiter->getRequestCount(),
      ],
    ];
  }
  
  /**
   * Clear sync state.
   */
  public function clearSyncState(): void {
    $this->state->delete('dntrade.last_sync_time');
    $this->state->delete('dntrade.last_sync_stats');
    $this->productCreator->clearProcessedApiIds();
    $this->rateLimiter->reset();
  }
  
  /**
   * Test API connection.
   */
  public function testConnection(): array {
    try {
      $this->rateLimiter->check();
      
      // Make a simple request to test connection
      $params = [
        'limit' => 1,
        'website_synch' => 1,
      ];
      
      $response = $this->client->getProductsList($params);
      
      return [
        'success' => true,
        'message' => 'Connection successful',
        'response_sample' => !empty($response),
      ];
      
    } catch (DntradeClientException $e) {
      return [
        'success' => false,
        'message' => 'Connection failed: ' . $e->getMessage(),
      ];
    } catch (\Exception $e) {
      return [
        'success' => false,
        'message' => 'Error: ' . $e->getMessage(),
      ];
    }
  }
}
