<?php

namespace Drupal\dntrade\Drush\Commands;

use Drush\Commands\DrushCommands;
use Drupal\dntrade\DntradeCron;
use Drupal\dntrade\DntradeSyncService;

/**
 * Drush commands for DNTrade module.
 */
class DntradeCommands extends DrushCommands {
  
  /**
   * DNTrade cron service.
   *
   * @var \Drupal\dntrade\DntradeCron
   */
  protected $cron;
  
  /**
   * DNTrade sync service.
   *
   * @var \Drupal\dntrade\DntradeSyncService
   */
  protected $syncService;
  
  /**
   * Constructor.
   */
  public function __construct(
    DntradeCron $cron,
    DntradeSyncService $syncService
  ) {
    $this->cron = $cron;
    $this->syncService = $syncService;
  }
  
  /**
   * Run DNTrade synchronization.
   *
   * @command dntrade:sync
   * @aliases dnt-sync, dntrade-sync
   * @option type Sync type (full, incremental)
   * @option force Force sync even if not scheduled
   * @option dry-run Dry run (don't save changes)
   * @usage dntrade:sync --type=full
   *   Run full synchronization.
   * @usage dntrade:sync --type=incremental
   *   Run incremental synchronization.
   * @usage dntrade:sync --type=full --dry-run
   *   Test full synchronization without saving.
   */
  public function sync(array $options = ['type' => 'incremental', 'force' => false, 'dry-run' => false]) {
    $type = $options['type'];
    
    $this->logger()->info('Starting manual @type synchronization', ['@type' => $type]);
    $this->output()->writeln("Starting $type synchronization...");
    
    // Set dry-run mode if specified
    if ($options['dry-run']) {
      $this->output()->writeln("<comment>DRY RUN: No changes will be saved.</comment>");
      // Здесь можно добавить логику dry-run если нужно
    }
    
    try {
      if ($type === 'full') {
        $results = $this->syncService->runFullSync();
      } else {
        $results = $this->syncService->runIncrementalSync();
      }
      
      // Display results
      $this->displayResults($results, $type);
      
    } catch (\Exception $e) {
      $this->logger()->error('Synchronization failed: @error', ['@error' => $e->getMessage()]);
      $this->output()->writeln("<error>Error: " . $e->getMessage() . "</error>");
      return 1;
    }
    
    return 0;
  }
  
  /**
   * Display synchronization results.
   */
  private function displayResults(array $results, string $type): void {
    $this->output()->writeln("\n<info>$type synchronization completed!</info>");
    $this->output()->writeln("Results:");
    $this->output()->writeln("  Processed: " . ($results['processed'] ?? 0));
    $this->output()->writeln("  Created: " . ($results['created'] ?? 0));
    $this->output()->writeln("  Updated: " . ($results['updated'] ?? 0));
    $this->output()->writeln("  Skipped: " . ($results['skipped'] ?? 0));
    $this->output()->writeln("  Failed: " . ($results['failed'] ?? 0));
    
    if (isset($results['marked_available'])) {
      $this->output()->writeln("\nAvailability:");
      $this->output()->writeln("  Marked as available: " . $results['marked_available']);
      $this->output()->writeln("  Marked as unavailable: " . $results['marked_unavailable']);
    }
    
    if (isset($results['unpublished_variations'])) {
      $this->output()->writeln("\nVariations:");
      $this->output()->writeln("  Unpublished: " . $results['unpublished_variations']);
      $this->output()->writeln("  Published: " . $results['published_variations']);
    }
  }
  
  /**
   * Run scheduled synchronization.
   *
   * @command dntrade:cron
   * @aliases dnt-cron, dntrade-cron
   * @usage dntrade:cron
   *   Run scheduled synchronization based on intervals.
   */
  public function cron() {
    $this->logger()->info('Running DNTrade cron synchronization');
    $this->output()->writeln("Running DNTrade scheduled synchronization...");
    
    $scheduleInfo = $this->cron->getScheduleInfo();
    
    $this->output()->writeln("Schedule info:");
    $this->output()->writeln("  Last full sync: " . $scheduleInfo['last_full_sync_human']);
    $this->output()->writeln("  Next full sync: " . $scheduleInfo['next_full_sync_human']);
    $this->output()->writeln("  Last incremental sync: " . $scheduleInfo['last_incremental_sync_human']);
    $this->output()->writeln("  Next incremental sync: " . $scheduleInfo['next_incremental_sync_human']);
    
    try {
      $results = $this->cron->synchronizeProducts();
      
      if (empty($results)) {
        $this->output()->writeln("<comment>No sync operations were scheduled at this time.</comment>");
      } else {
        foreach ($results as $type => $result) {
          $this->output()->writeln("\n<info>$type sync results:</info>");
          if (isset($result['success']) && $result['success'] === false) {
            $this->output()->writeln("<error>Failed: " . $result['error'] . "</error>");
          } else {
            $this->output()->writeln("  Processed: " . ($result['processed'] ?? 0));
            $this->output()->writeln("  Created: " . ($result['created'] ?? 0));
            $this->output()->writeln("  Updated: " . ($result['updated'] ?? 0));
          }
        }
      }
      
    } catch (\Exception $e) {
      $this->logger()->error('Cron synchronization failed: @error', ['@error' => $e->getMessage()]);
      $this->output()->writeln("<error>Error: " . $e->getMessage() . "</error>");
      return 1;
    }
    
    return 0;
  }
  
  /**
   * Show synchronization status.
   *
   * @command dntrade:status
   * @aliases dnt-status, dntrade-status
   * @usage dntrade:status
   *   Show current synchronization status.
   */
  public function status() {
    $status = $this->syncService->getSyncStatus();
    $schedule = $this->cron->getScheduleInfo();
    
    $this->output()->writeln("<info>DNTrade Synchronization Status</info>\n");
    
    $this->output()->writeln("Last Sync:");
    $this->output()->writeln("  Time: " . $status['last_sync_human']);
    
    if (!empty($status['last_sync_results'])) {
      $results = $status['last_sync_results'];
      $this->output()->writeln("  Processed: " . ($results['processed'] ?? 0));
      $this->output()->writeln("  Created: " . ($results['created'] ?? 0));
      $this->output()->writeln("  Updated: " . ($results['updated'] ?? 0));
    }
    
    $this->output()->writeln("\nSchedule:");
    $this->output()->writeln("  Full sync interval: " . $this->formatInterval($schedule['full_sync_interval']));
    $this->output()->writeln("  Last full sync: " . $schedule['last_full_sync_human']);
    $this->output()->writeln("  Next full sync: " . $schedule['next_full_sync_human']);
    $this->output()->writeln("  Incremental sync interval: " . $this->formatInterval($schedule['incremental_sync_interval']));
    $this->output()->writeln("  Last incremental sync: " . $schedule['last_incremental_sync_human']);
    $this->output()->writeln("  Next incremental sync: " . $schedule['next_incremental_sync_human']);
    
    $this->output()->writeln("\nLogs:");
    $this->output()->writeln("  Log file exists: " . ($status['log_file_exists'] ? 'Yes' : 'No'));
    $this->output()->writeln("  Log file size: " . $this->formatBytes($status['log_file_size']));
    
    $this->output()->writeln("\nAPI Rate Limit:");
    $this->output()->writeln("  Requests this minute: " . ($status['rate_limit_status']['requests_this_minute'] ?? 0));
    
    // Test connection
    $this->output()->writeln("\nConnection Test:");
    $connectionTest = $this->syncService->testConnection();
    if ($connectionTest['success']) {
      $this->output()->writeln("  <info>✓ Connected successfully</info>");
    } else {
      $this->output()->writeln("  <error>✗ Connection failed: " . $connectionTest['message'] . "</error>");
    }
  }
  
  /**
   * Test API connection.
   *
   * @command dntrade:test-connection
   * @aliases dnt-test, dntrade-test
   * @usage dntrade:test-connection
   *   Test connection to DNTrade API.
   */
  public function testConnection() {
    $this->output()->writeln("Testing connection to DNTrade API...");
    
    $result = $this->syncService->testConnection();
    
    if ($result['success']) {
      $this->output()->writeln("<info>✓ Connection successful!</info>");
      $this->output()->writeln("Message: " . $result['message']);
    } else {
      $this->output()->writeln("<error>✗ Connection failed!</error>");
      $this->output()->writeln("Error: " . $result['message']);
      return 1;
    }
    
    return 0;
  }
  
  /**
   * Reset synchronization schedule.
   *
   * @command dntrade:reset-schedule
   * @aliases dnt-reset, dntrade-reset
   * @usage dntrade:reset-schedule
   *   Reset synchronization schedule timers.
   */
  public function resetSchedule() {
    $this->cron->resetSchedule();
    $this->output()->writeln("<info>Synchronization schedule has been reset.</info>");
    $this->output()->writeln("Next cron run will trigger syncs based on intervals.");
  }
  
  /**
   * Clear all synchronization data.
   *
   * @command dntrade:clear
   * @aliases dnt-clear, dntrade-clear
   * @option products Also delete all imported products
   * @usage dntrade:clear
   *   Clear sync state and logs.
   * @usage dntrade:clear --products
   *   Clear sync state and delete all imported products.
   */
  public function clear(array $options = ['products' => false]) {
    $this->output()->writeln("<comment>Clearing DNTrade synchronization data...</comment>");
    
    // Clear sync state
    $this->syncService->clearSyncState();
    $this->cron->resetSchedule();
    
    // Clear log file
    $logFile = DRUPAL_ROOT . '/dntrade/dnt_posts.log';
    if (file_exists($logFile)) {
      unlink($logFile);
      $this->output()->writeln("  Log file deleted.");
    }
    
    // Delete products if requested
    if ($options['products']) {
      $this->output()->writeln("  Deleting imported products...");
      $deleted = $this->deleteImportedProducts();
      $this->output()->writeln("  Deleted $deleted products and variations.");
    }
    
    $this->output()->writeln("<info>✓ Synchronization data cleared successfully.</info>");
  }
  
  /**
   * Delete all imported products.
   */
  private function deleteImportedProducts(): int {
    $count = 0;
    
    // Find all products with dntrade variations
    $variationStorage = \Drupal::entityTypeManager()->getStorage('commerce_product_variation');
    $productStorage = \Drupal::entityTypeManager()->getStorage('commerce_product');
    
    // Find variations with dntrade code
    $query = $variationStorage->getQuery()
      ->accessCheck(FALSE)
      ->condition('field_dntrade_code', 0, '>');
    
    $vids = $query->execute();
    
    if (!empty($vids)) {
      $variations = $variationStorage->loadMultiple($vids);
      
      // Group by product
      $productIds = [];
      foreach ($variations as $variation) {
        if ($product = $variation->getProduct()) {
          $productIds[$product->id()] = $product->id();
        }
        $variation->delete();
        $count++;
      }
      
      // Delete products
      if (!empty($productIds)) {
        $products = $productStorage->loadMultiple($productIds);
        foreach ($products as $product) {
          $product->delete();
          $count++;
        }
      }
    }
    
    return $count;
  }
  
  /**
   * Format interval in seconds to human readable.
   */
  private function formatInterval(int $seconds): string {
    if ($seconds < 60) {
      return "$seconds seconds";
    } elseif ($seconds < 3600) {
      $minutes = floor($seconds / 60);
      return "$minutes minutes";
    } elseif ($seconds < 86400) {
      $hours = floor($seconds / 3600);
      return "$hours hours";
    } else {
      $days = floor($seconds / 86400);
      return "$days days";
    }
  }
  
  /**
   * Format bytes to human readable.
   */
  private function formatBytes(int $bytes, int $precision = 2): string {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);
    $bytes /= pow(1024, $pow);
    
    return round($bytes, $precision) . ' ' . $units[$pow];
  }
  
  /**
   * Enable synchronization.
   *
   * @command dntrade:enable
   * @aliases dnt-enable, dntrade-enable
   * @option reason Reason for enabling
   * @usage dntrade:enable
   *   Enable synchronization.
   * @usage dntrade:enable --reason="Maintenance completed"
   *   Enable synchronization with a reason.
   */
  public function enable(array $options = ['reason' => '']) {
    /** @var \Drupal\dntrade\SyncStatus $syncStatus */
    $syncStatus = \Drupal::service('dntrade.sync_status');
    
    $reason = $options['reason'] ?: 'Enabled via Drush command';
    $syncStatus->enableWithReason($reason, 'drush');
    
    $this->logger()->info('Synchronization enabled via Drush: @reason', ['@reason' => $reason]);
    $this->output()->writeln("<info>✓ Synchronization enabled.</info>");
    
    return 0;
  }

  /**
   * Disable synchronization.
   *
   * @command dntrade:disable
   * @aliases dnt-disable, dntrade-disable
   * @option reason Reason for disabling
   * @usage dntrade:disable
   *   Disable synchronization.
   * @usage dntrade:disable --reason="Maintenance in progress"
   *   Disable synchronization with a reason.
   */
  public function disable(array $options = ['reason' => '']) {
    /** @var \Drupal\dntrade\SyncStatus $syncStatus */
    $syncStatus = \Drupal::service('dntrade.sync_status');
    
    $reason = $options['reason'] ?: 'Disabled via Drush command';
    $syncStatus->disableWithReason($reason, 'drush');
    
    $this->logger()->info('Synchronization disabled via Drush: @reason', ['@reason' => $reason]);
    $this->output()->writeln("<info>✓ Synchronization disabled.</info>");
    
    return 0;
  }

  /**
   * Toggle synchronization status.
   *
   * @command dntrade:toggle
   * @aliases dnt-toggle, dntrade-toggle
   * @usage dntrade:toggle
   *   Toggle synchronization status.
   */
  public function toggle() {
    /** @var \Drupal\dntrade\SyncStatus $syncStatus */
    $syncStatus = \Drupal::service('dntrade.sync_status');
    
    $newStatus = $syncStatus->toggle();
    
    if ($newStatus) {
      $this->logger()->info('Synchronization toggled to ENABLED via Drush');
      $this->output()->writeln("<info>✓ Synchronization enabled.</info>");
    } else {
      $this->logger()->info('Synchronization toggled to DISABLED via Drush');
      $this->output()->writeln("<info>✓ Synchronization disabled.</info>");
    }
    
    return 0;
  }

  /**
   * Show synchronization status.
   *
   * @command dntrade:sync-status
   * @aliases dnt-sync-status, dntrade-status
   * @usage dntrade:sync-status
   *   Show detailed synchronization status.
   */
  public function syncStatus() {
    /** @var \Drupal\dntrade\SyncStatus $syncStatus */
    $syncStatus = \Drupal::service('dntrade.sync_status');
    $status = $syncStatus->getStatus();
    
    $this->output()->writeln("<info>DNTrade Synchronization Status</info>\n");
    
    $this->output()->writeln("Status: " . ($status['enabled'] ? "<info>ENABLED</info>" : "<error>DISABLED</error>"));
    
    if ($status['last_enabled_change']) {
      $this->output()->writeln("Last status change: " . date('Y-m-d H:i:s', $status['last_enabled_change']));
    }
    
    if ($status['disabled_by']) {
      $this->output()->writeln("Disabled by: " . $status['disabled_by']);
    }
    
    if ($status['disabled_reason']) {
      $this->output()->writeln("Reason: " . $status['disabled_reason']);
    }
    
    $this->output()->writeln("Manual sync allowed: " . ($status['allow_manual_sync'] ? "Yes" : "No"));
    
    if ($status['next_sync_time']) {
      $this->output()->writeln("Next scheduled sync: " . date('Y-m-d H:i:s', $status['next_sync_time']));
    }
    
    return 0;
  }
}
