<?php

namespace Drupal\dntrade\EventSubscriber;

use Drupal\dntrade\DntradeClientInterface;
use Drupal\dntrade\DntradeClientFactory;

use Drupal\state_machine\Event\WorkflowTransitionEvent;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\profile\Entity\ProfileInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Order transition event subscriber.
 */
class OrderPlaceSubscriber implements EventSubscriberInterface {

  /**
   * The DNTrade client factory service.
   *
   * @var \Drupal\dntrade\DntradeClientFactory
   */
  protected $dntradeClientFactory;

  /**
   * Constructs a new OrderPlaceSubscriber object.
   *
   * @param \Drupal\dntrade\DntradeClientFactory $dntrade_client_factory
   *   The DNTrade client factory service.
   */
  public function __construct(DntradeClientFactory $dntrade_client_factory) {
    $this->dntradeClientFactory = $dntrade_client_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      'commerce_order.place.pre_transition' => ['onPlaceTransition', -100],
    ];
  }

  /**
   * Handles the order place transition event.
   *
   * @param \Drupal\state_machine\Event\WorkflowTransitionEvent $event
   *   The transition event.
   */
  public function onPlaceTransition(WorkflowTransitionEvent $event) {
    /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
    $order = $event->getEntity();
    
    // if (!$order instanceof OrderInterface || $order->getState()->value !== 'validation') {
    if (!$order instanceof OrderInterface || $order->getState()->value !== 'completed') {
      return;
    }

    try {
      // Get DNTrade client instance
      $client = $this->dntradeClientFactory->get();
      
      // Get customer information for DNTrade order
      $customer_info = $this->getCustomerInfo($order); // name & phone
      
      // Prepare cart items for DNTrade and save UUIDs to product variations
      $cart_items = $this->prepareCartItems($order, $client);
      
      if (empty($cart_items)) {
        \Drupal::logger('dntrade')->warning('No cart items that match DNTrade` prod code found for order @order_id', ['@order_id' => $order->id()]);
        return;
      }

      // Create order data for DNTrade API
      $order_data = $this->createOrderData($order, $customer_info, $cart_items);
      
      
      // Send order to DNTrade
      $response = $client->createOrder($order_data);

      // Log successful order creation
      \Drupal::logger('dntrade')->info('Order @order_number successfully created in DNTrade with UUID: @uuid', [
        '@order_number' => $order->getOrderNumber(),
        '@uuid' => $response->id ?? 'unknown',
      ]);

      // save the DNTrade order UUID to the commerce order if needed
      $order->__set('field_dntrade_order_uuid', $response->id);
      // $order->save();
        // causes 'Drupal\Core\Entity\EntityStorageException: Update existing 'commerce_order' entity while changing the ID is not supported.'
        // + this order multiple records with same data (clones) in DNTrade side
        // data will be saved anyway

    } catch (\Exception $e) {
      \Drupal::logger('dntrade')->error('Error processing order @order_id: @message', [
        '@order_id' => $order->id(),
        '@message' => $e->getMessage(),
      ]);
      
      // Optionally, you can prevent the order transition if DNTrade integration fails
      // $event->setTransitionBlocked(TRUE);
      // $event->setMessage(t('Failed to create order in DNTrade: @error', ['@error' => $e->getMessage()]));
    }
  }

  /**
   * Extract customer information from order.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order entity.
   *
   * @return array
   *   Customer information array.
   */
  protected function getCustomerInfo(OrderInterface $order) {
    
    $customer = $order->getCustomer();
    
    $customer_info = [
      'name' => '',
      'phone' => '',
      'email' => '',
      'city' => '',
      'street' => '',
      'building' => '',
      'delivery_title' => '',
    ];

    // Get shipping profile
    // $shipping_profile = $order->getShippingProfile();
    // $billing_profile  = $order->getBillingProfile(); // null
    $shipping_profile = $this->getShippingProfile($order);
    if ($shipping_profile instanceof ProfileInterface) {
      
      // Get customer name
      if ($shipping_profile->hasField('address') && !$shipping_profile->get('address')->isEmpty()) {
        $address = $shipping_profile->get('address')->first();
        $customer_info['name'] = trim($address->getGivenName() . ' ' . $address->getFamilyName());
      }

      // Get customer telephone
      if ($shipping_profile->hasField('field_telephone') && !$shipping_profile->get('field_telephone')->isEmpty()) {
        $customer_info['phone'] = $shipping_profile->get('field_telephone')->getValue()[0]['value'];
      }
    }

    // shipment method that user have chosen
    $shipments = $order->get('shipments')->referencedEntities();
    if (!empty($shipments)) {
      $shipment = $shipments[0];
      $chosen_radio = $shipment->getShippingMethodId(); // 4 - input` value
      $shipping_method_id = $shipment->getShippingMethod();
      $customer_info['delivery_title'] = $shipping_method_id->getName(); // shipping_method_title like "Самовивіз з Нової Пошти"
      $pluginId = $shipping_method_id->getPlugin()->getPluginId();
    }
    
    $current_langcode = \Drupal::languageManager()->getCurrentLanguage()->getId();

    // custom values
    if($pluginId=='novaposhta' && in_array($chosen_radio,[2,5])){
      // $sklad_data = $address_field->get('warehouse_data')->getValue();
    }elseif(in_array($chosen_radio,[1,3])){
      if($current_langcode=='uk'){
        $city_name = "Дніпро";
      }elseif($current_langcode=='ru'){
        $city_name = "Днепр";
      }
    }
    
    // у DNTrade нет поля для региона / области – у Н. почты можем взять только город
    $shipment = $order->shipments->first()->entity;
    $shipping_profile = $shipment->shipping_profile->first()->entity;
    if(is_object($shipping_profile) && isset($pluginId) && $pluginId=='novaposhta'){
      $address_field = $shipping_profile->get('address_novaposhta')->first();
      // $region_data = $address_field->get('region_data')->getValue();
      $city_data = $address_field->get('city_data')->getValue();
      
      if($current_langcode=='uk'){
        // $region_name = $region_data['Description'];
        $city_name = $city_data['Description'];
        // if($sklad_data){
           // $sklad_name = $sklad_data['Description'];
        // }
      }elseif($current_langcode=='ru'){
        // $region_name = $region_data['DescriptionRu'];
        $city_name = $city_data['DescriptionRu'];
        // if($sklad_data){
           // $sklad_name = $sklad_data['DescriptionRu'];
        // }
      }
      
    }elseif(is_object($shipping_profile)){
       if(isset($shipping_profile->get('field_street')[0])){
          $customer_info['street'] = $shipping_profile->get('field_street')[0]->getValue()['value'];
        }
        
        // Будинок, Квартира
        if(isset($shipping_profile->get('field_house')[0])){
          $customer_info['building'] = $shipping_profile->get('field_house')[0]->getValue()['value'];
        }
        if(isset($shipping_profile->get('field_appartment')[0])){
          $customer_info['building'] = $shipping_profile->get('field_appartment')[0]->getValue()['value'];
        }
        // both
        if(isset($shipping_profile->get('field_house')[0]) && isset($shipping_profile->get('field_appartment')[0])){
          $customer_info['building'] = $shipping_profile->get('field_house')[0]->getValue()['value'] . ' ' . 
          $shipping_profile->get('field_appartment')[0]->getValue()['value'];
        }
    }

    if($city_name){
       $customer_info['city'] = $city_name;
    }   
    
    if (!empty($customer->getEmail())) {
        $customer_info['email'] = $customer->getEmail();
    }
    
    // Fallback to order email if name is empty
    if (empty($customer_info['name']) && $order->getEmail()) {
      $customer_info['name'] = $order->getEmail();
    }

    return $customer_info;
  }

  /**
   * Gets the shipping profile, if exists.
   *
   * The function safely checks for the existence of the 'shipments' field,
   * which is installed by commerce_shipping. If the field does not exist or is
   * empty, NULL will be returned.
   *
   * The shipping profile is assumed to be the same for all shipments.
   * Therefore, it is taken from the first found shipment, or created from
   * scratch if no shipments were found.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order entity.
   *
   * @return \Drupal\profile\Entity\ProfileInterface|null
   *   The shipping profile.
   */
  protected function getShippingProfile(OrderInterface $order) {
    if ($order->hasField('shipments')) {
      /** @var \Drupal\commerce_shipping\Entity\ShipmentInterface $shipment */
      foreach ($order->shipments->referencedEntities() as $shipment) {
        return $shipment->getShippingProfile();
      }
    }
    return NULL;
  }
  
  /**
   * Prepare cart items for DNTrade API and save UUIDs to product variations.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order entity.
   * @param \Drupal\dntrade\DntradeClientInterface $client
   *   The DNTrade client.
   *
   * @return array
   *   Array of cart items for DNTrade.
   */
  protected function prepareCartItems(OrderInterface $order, DntradeClientInterface $client) {
    $cart_items = [];
    $saved_uuids = [];
    $dntrade_store_uuid = '833a605c-fa32-46b6-9735-067239c68634'; // Your store UUID
    

    /** @var \Drupal\commerce_order\Entity\OrderItemInterface $order_item */
    foreach ($order->getItems() as $order_item) {
      $purchased_entity = $order_item->getPurchasedEntity();
      if (!$purchased_entity) {
        continue;
      }

      // We need required product uuid ("product_id"), so firstly let`s get it in a separate request on base of product 'code' value:
      
      // Get DNTrade code from product variation
      $dntrade_code = $this->getDntradeCode($purchased_entity);
      if (!$dntrade_code) {
        \Drupal::logger('dntrade')->warning('No DNTrade code found for product variation @variation_id', [
          '@variation_id' => $purchased_entity->id(),
        ]);
        continue;
      }

      // Get product UUID from DNTrade and save to product variation
      $product_uuid = $this->getProductUuid($dntrade_code, $client, $purchased_entity);
      if (!$product_uuid) {
        // to display 'uuid товару не знайдено на DNTrade' on an order view page
        $order_item->set('field_no_uuid', 1);
        $order_item->save();

        \Drupal::logger('dntrade')->warning('No UUID found for DNTrade code @code', [
          '@code' => $dntrade_code,
        ]);
        continue;
      }

      
      $cart_items[] = [
        'store_id' => $dntrade_store_uuid,
        'product_id' => $product_uuid,
        // 'price' => $this->convertPriceToCents($order_item->getUnitPrice()->getNumber()),
        'price' => $order_item->getUnitPrice()->getNumber(),
        'quantity' => (int) $order_item->getQuantity(),
        'product_bonus_sum' => 0,
      ];
    }

    return $cart_items;
  }

  /**
   * Get DNTrade code from product variation.
   *
   * @param mixed $purchased_entity
   *   The purchased entity (product variation).
   *
   * @return string|null
   *   DNTrade code or null if not found.
   */
  protected function getDntradeCode($purchased_entity) {
    if ($purchased_entity->hasField('field_dntrade_code') && !$purchased_entity->get('field_dntrade_code')->isEmpty()) {
      return $purchased_entity->get('field_dntrade_code')->getValue()[0]['value'];
    }
    
    return null;
  }

  /**
   * Get product UUID from DNTrade using product code and save to product variation.
   *
   * @param string $dntrade_code
   *   The DNTrade product code.
   * @param \Drupal\dntrade\DntradeClientInterface $client
   *   The DNTrade client.
   * @param \Drupal\commerce_product\Entity\ProductVariationInterface $product_variation
   *   The product variation entity.
   *
   * @return string|null
   *   Product UUID or null if not found.
   */
  protected function getProductUuid($dntrade_code, DntradeClientInterface $client, $product_variation) {

    try {
      $products_data = [
        'limit' => 1,
        'code' => $dntrade_code,
        'website_synch' => 0,
      ];

      $response = $client->getProductsList($products_data);
      
      $product_uuid = null;
      // Adjust this based on the actual response structure from getProductsList
      if (is_object($response) && !empty($response->products[0])) {
        $product = $response->products[0];
        $product_uuid = $product->id ?? $product->product_id ?? null;
      }

      // Save UUID to product variation if found and field exists
      if ($product_uuid && $product_variation->hasField('field_dntrade_prod_uuid')) {
        $current_uuid = $product_variation->get('field_dntrade_prod_uuid')->value;
        
        // Only save if UUID is different or empty
        if (empty($current_uuid) || $current_uuid !== $product_uuid) {
          $product_variation->set('field_dntrade_prod_uuid', $product_uuid);
          
          // Check if we need to save the product variation
          // We only save if the variation is not new and UUID has changed
          if (!$product_variation->isNew()) {
            $product_variation->save();
            
            \Drupal::logger('dntrade')->info('Saved DNTrade UUID @uuid for product variation @variation_id (code: @code)', [
              '@uuid' => $product_uuid,
              '@variation_id' => $product_variation->id(),
              '@code' => $dntrade_code,
            ]);
          }
        }
      }

      return $product_uuid;

    } catch (\Drupal\dntrade\DntradeClientException $e) {
      \Drupal::logger('dntrade')->error('Error getting product UUID for code @code: @message', [
        '@code' => $dntrade_code,
        '@message' => $e->getMessage(),
      ]);
      return null;
    }
  }

  /**
   * Convert price to cents (if required by DNTrade API).
   *
   * @param float $price
   *   The price in decimal format.
   *
   * @return int
   *   Price in cents.
   */
  protected function convertPriceToCents($price) {
    return (int) round($price * 100);
  }

  /**
   * Create order data for DNTrade API.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order entity.
   * @param array $customer_info
   *   Customer information.
   * @param array $cart_items
   *   Cart items array.
   *
   * @return array
   *   Formatted order data for DNTrade API.
   */
  protected function createOrderData(OrderInterface $order, array $customer_info, array $cart_items) {
    $timezone = \Drupal::config('system.date')->get('timezone.default');
    $now = new \DateTime('now', new \DateTimeZone($timezone));
    
    return [
      'id' => null,
      'number' => $order->getOrderNumber() ?: $order->id(),
      'date' => $now->format('Y-m-d H:i:s'),
      'status' => 5, // 1 – Підтверджено, 2 – На виконанні, 5 – Нове
      'channel' => 'pixel.org.ua',
      'cart' => $cart_items,
      'personal_info' => [
        'client_id' => '',
        'name' => $customer_info['name'],
        'city' => $customer_info['city'],
        'street' => $customer_info['street'],
        'building' => $customer_info['building'],
        'phone' => $customer_info['phone'],
        'email' => $customer_info['email'],
        'comment' => $this->getOrderComment($order),
        'delivery_title' => $customer_info['delivery_title'],
        // 'card_or_cash' => $this->getPaymentMethodType($order),
        'card_or_cash' => 1, // old API` legacy: value doesn't matter. In current dntrade.com.ua` admin interface it is absent at all.
        'bonus_sum' => 0,
        'writeoff_bonus' => 0,
      ],
    ];
  }

  /**
   * Get order comment if available.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order entity.
   *
   * @return string
   *   Order comment.
   */
  protected function getOrderComment(OrderInterface $order) {
    if ($order->hasField('field_order_notes') && !$order->get('field_order_notes')->isEmpty()) {
      return $order->get('field_order_notes')->getValue()[0]['value'];
    }
    
    return '';
  }

  /**
   * Get payment method type for DNTrade.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   The order entity.
   *
   * @return int
   *   1 for cash, 2 for card, etc.
   */
  protected function getPaymentMethodType(OrderInterface $order) {
    // Default to cash
    $payment_method_type = 1;
    
    // Get payment method from order
    $payments = $order->get('payment_method')->referencedEntities();
    if (!empty($payments)) {
      $payment_method = reset($payments);
      $payment_gateway = $payment_method->getPaymentGateway();
      
      if ($payment_gateway) {
        $gateway_id = $payment_gateway->id();
        // Map payment gateway to DNTrade payment type
        if (strpos($gateway_id, 'card') !== false || strpos($gateway_id, 'credit') !== false) {
          $payment_method_type = 2; // card
        }
      }
    }
    
    return $payment_method_type;
  }

}
