<?php

namespace Drupal\dntrade\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Settings Form for dntrade API configuration.
 */
class SettingsForm extends ConfigFormBase {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dntrade_api_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'dntrade.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('dntrade.settings');

    $form_state->setCached(FALSE);

    $form['api_configuration'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('API Configuration'),
      '#description' => $this->t('Configure connection to DNTrade API.'),
    ];

    $form['api_configuration']['base_uri'] = [
      '#type' => 'url',
      '#title' => $this->t('DNTrade API Base URI'),
      '#default_value' => $config->get('base_uri'),
      '#description' => $this->t('Add your DNTrade API base uri here. It follows the format of https://api.dntrade.com.ua/.'),
      '#required' => TRUE,
    ];

    $form['api_configuration']['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('DNTrade API Key'),
      '#default_value' => $config->get('api_key'),
      '#description' => $this->t('Add your DNTrade API key. Will be a long string provided by DNTrade.'),
      '#required' => TRUE,
    ];

    $form['sync_api'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Sync API Security'),
      '#description' => $this->t('Configure security for external sync triggers.'),
    ];

    $form['sync_api']['sync_api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Sync API Key'),
      '#default_value' => $config->get('sync_api_key'),
      '#description' => $this->t('API key for external sync triggers (cron, webhooks). Leave empty to disable external API.'),
    ];

    // Test connection button
    $form['actions']['test'] = [
      '#type' => 'submit',
      '#value' => $this->t('Test Connection'),
      '#submit' => ['::testConnection'],
      '#validate' => ['::validateTestConnection'],
      '#weight' => 5,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Validate test connection.
   */
  public function validateTestConnection(array &$form, FormStateInterface $form_state) {
    $base_uri = $form_state->getValue('base_uri');
    $api_key = $form_state->getValue('api_key');

    if (empty($base_uri)) {
      $form_state->setErrorByName('base_uri', $this->t('Base URI is required to test connection.'));
    }

    if (empty($api_key)) {
      $form_state->setErrorByName('api_key', $this->t('API key is required to test connection.'));
    }
  }

  /**
   * Test connection to DNTrade API.
   */
  public function testConnection(array &$form, FormStateInterface $form_state) {
    try {
      // Save temporary config for test
      $config = $this->configFactory->getEditable('dntrade.settings');
      $config
        ->set('base_uri', $form_state->getValue('base_uri'))
        ->set('api_key', $form_state->getValue('api_key'))
        ->save();

      // Test connection
      /** @var \Drupal\dntrade\DntradeSyncService $syncService */
      $syncService = \Drupal::service('dntrade.sync_service');
      $result = $syncService->testConnection();

      if ($result['success']) {
        $this->messenger()->addStatus($this->t('Connection successful! @message', [
          '@message' => $result['message'],
        ]));
      } else {
        $this->messenger()->addError($this->t('Connection failed: @message', [
          '@message' => $result['message'],
        ]));
      }

    } catch (\Exception $e) {
      $this->messenger()->addError($this->t('Error testing connection: @error', [
        '@error' => $e->getMessage(),
      ]));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    /** @var \Drupal\Core\Config\Config $config */
    $config = $this->configFactory->getEditable('dntrade.settings');

    $config->set('api_key', $form_state->getValue('api_key'));
    $config->set('sync_api_key', $form_state->getValue('sync_api_key'));
    
    if ($base_uri = $form_state->getValue('base_uri')) {
      if (preg_match("#https?://#", $base_uri) === 0) {
        $base_uri = 'https://' . $base_uri;
      }
      $config->set('base_uri', rtrim($base_uri, '/') . '/');
    }

    $config->save();

    parent::submitForm($form, $form_state);
  }

}
