<?php

namespace Drupal\dntrade\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure DNTrade synchronization settings.
 */
class SyncSettingsForm extends ConfigFormBase {
  
  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dntrade_sync_settings';
  }
  
  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['dntrade.settings'];
  }
  
  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('dntrade.settings');
    
    $form['status'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Synchronization Status'),
      '#description' => $this->t('Control whether synchronizations are allowed to run.'),
    ];
    
    $form['status']['sync_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Synchronization'),
      '#default_value' => $config->get('sync_enabled') ?? TRUE,
      '#description' => $this->t('Uncheck to temporarily disable all synchronizations. Useful for maintenance or development.'),
    ];
    
    $form['status']['allow_manual_sync'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow Manual Sync When Disabled'),
      '#default_value' => $config->get('allow_manual_sync') ?? FALSE,
      '#description' => $this->t('Allow manual synchronization even when automatic sync is disabled.'),
      '#states' => [
        'visible' => [
          ':input[name="sync_enabled"]' => ['checked' => FALSE],
        ],
      ],
    ];
    
    $form['intervals'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Synchronization Intervals'),
      '#description' => $this->t('Configure how often to synchronize products from DNTrade.'),
    ];
    
    $form['intervals']['full_sync_interval'] = [
      '#type' => 'select',
      '#title' => $this->t('Full Sync Interval'),
      '#default_value' => $config->get('full_sync_interval') ?? 86400,
      '#options' => [
        3600 => $this->t('Every hour'),
        21600 => $this->t('Every 6 hours'),
        43200 => $this->t('Every 12 hours'),
        86400 => $this->t('Every day'),
        604800 => $this->t('Every week'),
      ],
      '#description' => $this->t('How often to run full synchronization. Full sync processes all products.'),
    ];
    
    $form['intervals']['incremental_sync_interval'] = [
      '#type' => 'select',
      '#title' => $this->t('Incremental Sync Interval'),
      '#default_value' => $config->get('incremental_sync_interval') ?? 900,
      '#options' => [
        300 => $this->t('Every 5 minutes'),
        900 => $this->t('Every 15 minutes'),
        1800 => $this->t('Every 30 minutes'),
        3600 => $this->t('Every hour'),
      ],
      '#description' => $this->t('How often to run incremental synchronization. Incremental sync only processes modified products.'),
    ];
    
    $form['processing'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Processing Settings'),
    ];
    
    $form['processing']['batch_size'] = [
      '#type' => 'number',
      '#title' => $this->t('Batch Size'),
      '#default_value' => $config->get('batch_size') ?? 100,
      '#min' => 10,
      '#max' => 100,
      '#description' => $this->t('Number of products to process per batch (max 100, DNTrade API limit).'),
    ];
    
    $form['processing']['max_retries'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum Retries'),
      '#default_value' => $config->get('max_retries') ?? 3,
      '#min' => 1,
      '#max' => 10,
      '#description' => $this->t('Maximum number of retries for failed API requests.'),
    ];
    
    $form['content'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Content Processing'),
    ];
    
    $form['content']['download_images'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Download Product Images'),
      '#default_value' => $config->get('download_images') ?? TRUE,
      '#description' => $this->t('Download and attach product images from DNTrade.'),
    ];
    
    $form['content']['create_categories'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Create Missing Categories'),
      '#default_value' => $config->get('create_categories') ?? TRUE,
      '#description' => $this->t('Automatically create missing category terms in taxonomy.'),
    ];
    
    $form['content']['skip_rizne'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Skip "Різне" Category'),
      '#default_value' => $config->get('skip_rizne') ?? TRUE,
      '#description' => $this->t('Skip products in the "Різне" (Miscellaneous) category.'),
    ];
    
    $form['notifications'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Email Notifications'),
      '#description' => $this->t('Configure email notifications for synchronization events.'),
    ];
    
    $form['notifications']['enable_notifications'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Success Notifications'),
      '#default_value' => $config->get('enable_notifications') ?? FALSE,
      '#description' => $this->t('Send email notifications when synchronization completes successfully.'),
    ];
    
    $form['notifications']['notification_emails'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Notification Email Addresses'),
      '#default_value' => $config->get('notification_emails'),
      '#description' => $this->t('Enter email addresses (one per line) to receive success notifications.'),
      '#states' => [
        'visible' => [
          ':input[name="enable_notifications"]' => ['checked' => TRUE],
        ],
      ],
    ];
    
    $form['notifications']['enable_error_notifications'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Error Notifications'),
      '#default_value' => $config->get('enable_error_notifications') ?? TRUE,
      '#description' => $this->t('Send email notifications when synchronization fails.'),
    ];
    
    $form['notifications']['error_notification_emails'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Error Notification Email Addresses'),
      '#default_value' => $config->get('error_notification_emails'),
      '#description' => $this->t('Enter email addresses (one per line) to receive error notifications.'),
      '#states' => [
        'visible' => [
          ':input[name="enable_error_notifications"]' => ['checked' => TRUE],
        ],
      ],
    ];
    
    // Information about current schedule
    if ($this->currentUser()->hasPermission('administer dntrade')) {
      $form['schedule_info'] = [
        '#type' => 'details',
        '#title' => $this->t('Current Schedule Information'),
        '#open' => FALSE,
      ];
      
      try {
        /** @var \Drupal\dntrade\DntradeCron $cron */
        $cron = \Drupal::service('dntrade.cron');
        $schedule = $cron->getScheduleInfo();
        
        $form['schedule_info']['info'] = [
          '#markup' => '
            <p><strong>Last Full Sync:</strong> ' . $schedule['last_full_sync_human'] . '</p>
            <p><strong>Next Full Sync:</strong> ' . $schedule['next_full_sync_human'] . '</p>
            <p><strong>Last Incremental Sync:</strong> ' . $schedule['last_incremental_sync_human'] . '</p>
            <p><strong>Next Incremental Sync:</strong> ' . $schedule['next_incremental_sync_human'] . '</p>
          ',
        ];
      } catch (\Exception $e) {
        // Service might not be available yet
      }
    }

    return parent::buildForm($form, $form_state);
  }
  
  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);
    
    // Validate email addresses
    if ($form_state->getValue('enable_notifications')) {
      $emails = $form_state->getValue('notification_emails');
      if (!empty($emails)) {
        $invalid_emails = $this->validateEmailList($emails);
        if (!empty($invalid_emails)) {
          $form_state->setErrorByName('notification_emails', 
            $this->t('Invalid email addresses: @emails', ['@emails' => implode(', ', $invalid_emails)]));
        }
      }
    }
    
    if ($form_state->getValue('enable_error_notifications')) {
      $emails = $form_state->getValue('error_notification_emails');
      if (!empty($emails)) {
        $invalid_emails = $this->validateEmailList($emails);
        if (!empty($invalid_emails)) {
          $form_state->setErrorByName('error_notification_emails', 
            $this->t('Invalid email addresses: @emails', ['@emails' => implode(', ', $invalid_emails)]));
        }
      }
    }
  }
  
  /**
   * Validate list of email addresses.
   */
  private function validateEmailList(string $email_list): array {
    $invalid = [];
    $emails = preg_split('/\r\n|\r|\n/', $email_list);
    
    foreach ($emails as $email) {
      $email = trim($email);
      if (!empty($email) && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $invalid[] = $email;
      }
    }
    
    return $invalid;
  }
  
  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('dntrade.settings')
      ->set('sync_enabled', $form_state->getValue('sync_enabled'))
      ->set('allow_manual_sync', $form_state->getValue('allow_manual_sync'))
      ->set('full_sync_interval', $form_state->getValue('full_sync_interval'))
      ->set('incremental_sync_interval', $form_state->getValue('incremental_sync_interval'))
      ->set('max_retries', $form_state->getValue('max_retries'))
      ->set('batch_size', $form_state->getValue('batch_size'))
      ->set('download_images', $form_state->getValue('download_images'))
      ->set('create_categories', $form_state->getValue('create_categories'))
      ->set('skip_rizne', $form_state->getValue('skip_rizne'))
      ->set('enable_notifications', $form_state->getValue('enable_notifications'))
      ->set('notification_emails', $form_state->getValue('notification_emails'))
      ->set('enable_error_notifications', $form_state->getValue('enable_error_notifications'))
      ->set('error_notification_emails', $form_state->getValue('error_notification_emails'))
      ->save();
    
    parent::submitForm($form, $form_state);
    
    $this->messenger()->addStatus($this->t('Synchronization settings saved.'));
  }
}
