<?php

namespace Drupal\dntrade;

/**
 * Rate limiter for DNTrade API.
 */
class RateLimiter {
  
  /**
   * Number of requests made in the current minute.
   *
   * @var int
   */
  private $requestsThisMinute = 0;
  
  /**
   * Maximum requests allowed per minute.
   *
   * @var int
   */
  private $maxRequests = 95; // Safety margin
  
  /**
   * Time when rate limit resets.
   *
   * @var int
   */
  private $resetTime = 0;
  
  /**
   * Start time of current minute window.
   *
   * @var int
   */
  private $minuteStart = 0;
  
  /**
   * Check if we can make a request.
   */
  public function check(array $responseHeaders = null): void {
    $now = time();
    $currentMinute = floor($now / 60);
    
    // Reset if new minute
    if ($currentMinute !== $this->minuteStart) {
      $this->requestsThisMinute = 0;
      $this->minuteStart = $currentMinute;
      $this->resetTime = 0;
    }
    
    // Update from response headers if available
    if ($responseHeaders) {
      $this->updateFromHeaders($responseHeaders);
    }
    
    // Check if we've reached the limit
    if ($this->requestsThisMinute >= $this->maxRequests) {
      $this->waitForReset();
    }
    
    $this->requestsThisMinute++;
  }
  
  /**
   * Update rate limit status from API response headers.
   */
  private function updateFromHeaders(array $headers): void {
    // Parse API-Rate-Limit header (e.g., "1/100")
    if (isset($headers['API-Rate-Limit']) && preg_match('/(\d+)\/(\d+)/', $headers['API-Rate-Limit'], $matches)) {
      $this->requestsThisMinute = (int)$matches[1];
      
      // If we have reset time from headers
      if (isset($headers['X-RateLimit-Reset'])) {
        $this->resetTime = (int)$headers['X-RateLimit-Reset'];
      } elseif (isset($headers['Retry-After'])) {
        $this->resetTime = time() + (int)$headers['Retry-After'];
      }
    }
  }
  
  /**
   * Wait for rate limit reset.
   */
  private function waitForReset(): void {
    $now = time();
    
    if ($this->resetTime > 0) {
      $waitTime = $this->resetTime - $now + 1;
    } else {
      // Wait until next minute if no reset time provided
      $waitTime = 60 - ($now % 60) + 1;
    }
    
    if ($waitTime > 0) {
      \Drupal::logger('dntrade')->info('Rate limit reached, waiting @seconds seconds', [
        '@seconds' => $waitTime,
      ]);
      sleep($waitTime);
    }
    
    // Reset counters
    $this->requestsThisMinute = 0;
    $this->minuteStart = floor(time() / 60);
    $this->resetTime = 0;
  }
  
  /**
   * Handle 429 Too Many Requests error.
   */
  public function handleRateLimitError(): void {
    \Drupal::logger('dntrade')->warning('Rate limit error received, waiting 61 seconds');
    $this->resetTime = time() + 61;
    $this->waitForReset();
  }
  
  /**
   * Get current request count.
   */
  public function getRequestCount(): int {
    return $this->requestsThisMinute;
  }
  
  /**
   * Reset the rate limiter.
   */
  public function reset(): void {
    $this->requestsThisMinute = 0;
    $this->minuteStart = 0;
    $this->resetTime = 0;
  }
}
