<?php // для проверки статуса синхронизации – возможности временного отключения синхрониззации

namespace Drupal\dntrade;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\State\StateInterface;

/**
 * Service to manage synchronization status.
 */
class SyncStatus {
  
  /**
   * Config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;
  
  /**
   * State API.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;
  
  /**
   * Constructor.
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    StateInterface $state
  ) {
    $this->configFactory = $configFactory;
    $this->state = $state;
  }
  
  /**
   * Check if synchronization is enabled.
   */
  public function isEnabled(): bool {
    $config = $this->configFactory->get('dntrade.settings');
    return (bool) $config->get('sync_enabled');
  }
  
  /**
   * Enable synchronization.
   */
  public function enable(): void {
    $this->configFactory->getEditable('dntrade.settings')
      ->set('sync_enabled', TRUE)
      ->save();
    
    \Drupal::logger('dntrade')->info('Synchronization enabled.');
  }
  
  /**
   * Disable synchronization.
   */
  public function disable(): void {
    $this->configFactory->getEditable('dntrade.settings')
      ->set('sync_enabled', FALSE)
      ->save();
    
    \Drupal::logger('dntrade')->info('Synchronization disabled.');
  }
  
  /**
   * Check if manual sync is allowed.
   */
  public function isManualAllowed(): bool {
    $config = $this->configFactory->get('dntrade.settings');
    return (bool) $config->get('allow_manual_sync') ?? TRUE;
  }
  
  /**
   * Toggle synchronization status.
   */
  public function toggle(): bool {
    $current = $this->isEnabled();
    if ($current) {
      $this->disable();
      return FALSE;
    } else {
      $this->enable();
      return TRUE;
    }
  }
  
  /**
   * Get synchronization status with details.
   */
  public function getStatus(): array {
    $config = $this->configFactory->get('dntrade.settings');
    
    return [
      'enabled' => $this->isEnabled(),
      'last_enabled_change' => $this->state->get('dntrade.last_enabled_change'),
      'disabled_by' => $this->state->get('dntrade.disabled_by'),
      'disabled_reason' => $this->state->get('dntrade.disabled_reason'),
      'allow_manual_sync' => $this->isManualAllowed(),
      'next_sync_time' => $this->state->get('dntrade.next_sync_time'),
    ];
  }
  
  /**
   * Disable with reason.
   */
  public function disableWithReason(string $reason, string $user = null): void {
    $this->disable();
    
    $this->state->setMultiple([
      'dntrade.disabled_by' => $user ?: \Drupal::currentUser()->getAccountName(),
      'dntrade.disabled_reason' => $reason,
      'dntrade.last_enabled_change' => time(),
    ]);
    
    \Drupal::logger('dntrade')->warning('Synchronization disabled by @user. Reason: @reason', [
      '@user' => $user ?: 'unknown',
      '@reason' => $reason,
    ]);
  }
  
  /**
   * Enable with reason.
   */
  public function enableWithReason(string $reason, string $user = null): void {
    $this->enable();
    
    $this->state->setMultiple([
      'dntrade.enabled_by' => $user ?: \Drupal::currentUser()->getAccountName(),
      'dntrade.enabled_reason' => $reason,
      'dntrade.last_enabled_change' => time(),
    ]);
    
    \Drupal::logger('dntrade')->info('Synchronization enabled by @user. Reason: @reason', [
      '@user' => $user ?: 'unknown',
      '@reason' => $reason,
    ]);
  }
  
  /**
   * Check if sync should run now.
   */
  public function shouldRun(string $sync_type = 'incremental'): bool {
    if (!$this->isEnabled()) {
      return FALSE;
    }
    
    // Check maintenance mode
    if (\Drupal::state()->get('system.maintenance_mode')) {
      return FALSE;
    }
    
    // Check if we're in a batch process
    if (batch_get()) {
      return FALSE;
    }
    
    // Check for development environment
    if ($this->isDevelopmentEnvironment()) {
      \Drupal::logger('dntrade')->debug('Skipping sync in development environment');
      return FALSE;
    }
    
    return TRUE;
  }
  
  /**
   * Check if we're in a development environment.
   */
  private function isDevelopmentEnvironment(): bool {
    $env = getenv('APP_ENV') ?: getenv('ENVIRONMENT');
    
    if ($env && in_array(strtolower($env), ['dev', 'development', 'local', 'test'])) {
      return TRUE;
    }
    
    // Check Drupal environment
    if (\Drupal::hasService('kernel')) {
      $kernel = \Drupal::service('kernel');
      if (method_exists($kernel, 'getEnvironment') && $kernel->getEnvironment() === 'dev') {
        return TRUE;
      }
    }
    
    return FALSE;
  }
}
