<?php

namespace Drupal\doc_to_html\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for DOC to HTML basic settings.
 */
class BasicSettings extends ConfigFormBase
{

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array
  {
    return [
      'doc_to_html.basic_settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string
  {
    return 'doc_to_html_basic_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array
  {
    $config = $this->config('doc_to_html.basic_settings');

    // Folder configuration: controls where converted HTML will be stored
    // (relative to public://) before being injected into widgets/forms.
    $form['wrapper_folder'] = [
      '#type' => 'details',
      '#open' => TRUE,
      '#title' => $this->t('Path to save HTML files'),
    ];

    $form['wrapper_folder']['doc_to_html_folder'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Folder name'),
      '#description' => $this->t('This folder is relative to the <code>public://</code> stream wrapper.'),
      '#maxlength' => 100,
      '#size' => 60,
      '#required' => TRUE,
      '#default_value' => $config->get('doc_to_html_folder') ?: 'doc_to_html',
    ];

    // File type toggles let site builders restrict allowed uploads for both the
    // Test Wizard and the field widget.
    $form['wrapper_file'] = [
      '#type' => 'details',
      '#open' => TRUE,
      '#title' => $this->t('Supported file types'),
    ];

    $form['wrapper_file']['doc'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('DOC'),
      '#default_value' => (bool)$config->get('doc'),
    ];

    $form['wrapper_file']['docx'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('DOCX'),
      '#default_value' => (bool)$config->get('docx'),
    ];

    // HTML post-processing options: decide whether to force UTF-8 and apply
    // regex extractions to narrow the converted markup.
    $form['parse_data'] = [
      '#type' => 'details',
      '#open' => TRUE,
      '#title' => $this->t('Parse HTML output'),
    ];

    $form['parse_data']['utf_8_encode'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Force UTF-8 encoding'),
      '#description' => $this->t('Ensure that the HTML output is converted to UTF-8 if needed.'),
      '#default_value' => (bool)$config->get('utf_8_encode'),
    ];

    $form['parse_data']['regex_to_parse_body'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Regular expression to extract body content'),
      '#description' => $this->t('A regular expression used to extract body content from the generated HTML. If provided, it will be applied.'),
      '#maxlength' => 255,
      '#size' => 80,
      '#default_value' => $config->get('regex_to_parse_body') ?: '',
    ];

    $form['parse_data']['regex_body_match_index'] = [
      '#type' => 'number',
      '#title' => $this->t('Body regex match index'),
      '#description' => $this->t('Select which capture group to return from the body regex match. 0 = entire match.'),
      '#default_value' => is_numeric($config->get('regex_body_match_index')) ? (int) $config->get('regex_body_match_index') : 0,
      '#min' => 0,
      '#step' => 1,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void
  {
    // Ensure at least one file type is selected so conversions remain possible.
    $doc = (bool)$form_state->getValue('doc');
    $docx = (bool)$form_state->getValue('docx');

    if (!$doc && !$docx) {
      $form_state->setErrorByName('doc', $this->t('You must enable at least one supported file type (DOC or DOCX).'));
      $form_state->setErrorByName('docx', $this->t('You must enable at least one supported file type (DOC or DOCX).'));
    }

    // Optionally check that the regex is valid and require a match index when set.
    $regex = $form_state->getValue('regex_to_parse_body');
    if (!empty($regex)) {
      set_error_handler(static function () {
      }, E_WARNING);
      $is_valid = @preg_match($regex, '') !== FALSE;
      restore_error_handler();

      if (!$is_valid) {
        $form_state->setErrorByName('regex_to_parse_body', $this->t('The provided regular expression is not valid.'));
      }

      $match_index = $form_state->getValue('regex_body_match_index');
      if ($match_index === NULL || $match_index === '' || !is_numeric($match_index) || (int)$match_index < 0) {
        $form_state->setErrorByName('regex_body_match_index', $this->t('When a body regex is provided, the body regex match index is required and must be a non-negative integer.'));
      }
    }

    $match_index = $form_state->getValue('regex_body_match_index');
    if ($match_index !== NULL && $match_index !== '') {
      if (!is_numeric($match_index) || (int)$match_index < 0) {
        $form_state->setErrorByName('regex_body_match_index', $this->t('The match index must be a non-negative integer.'));
      }
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void
  {
    parent::submitForm($form, $form_state);

    $editable = $this->configFactory->getEditable('doc_to_html.basic_settings');

    // Persist toggles and folder settings.
    $editable
      ->set('doc_to_html_folder', (string)$form_state->getValue('doc_to_html_folder'))
      ->set('doc', (bool)$form_state->getValue('doc'))
      ->set('docx', (bool)$form_state->getValue('docx'))
      ->set('utf_8_encode', (bool)$form_state->getValue('utf_8_encode'));

    $regex = trim((string)$form_state->getValue('regex_to_parse_body'));
    if ($regex !== '') {
      $editable->set('regex_to_parse_body', $regex);
    }
    else {
      $editable->clear('regex_to_parse_body');
    }

    // Persist the body regex match index only when provided.
    $match_index = $form_state->getValue('regex_body_match_index');
    if ($match_index !== NULL && $match_index !== '') {
      $editable->set('regex_body_match_index', (int) $match_index);
    } else {
      $editable->clear('regex_body_match_index');
    }

    $editable->save();
  }

}
