<?php

namespace Drupal\doc_to_html\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\doc_to_html\Services\DefaultService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form to map imported HTML to entity fields.
 */
class ImportToField extends ConfigFormBase
{

  /**
   * Default service for DOC to HTML utilities.
   *
   * @var DefaultService
   */
  protected DefaultService $defaultService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static
  {
    /** @var static $instance */
    $instance = parent::create($container);
    $instance->defaultService = $container->get('doc_to_html.default_service');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array
  {
    return [
      'doc_to_html.importtofield',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string
  {
    return 'doc_to_html_import_to_field';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array
  {
    $config = $this->config('doc_to_html.importtofield');

    // Retrieve all "text_with_summary" fields grouped by entity and bundle.
    $field_config = $this->defaultService->getEntityBundleFieldBy('text_with_summary');

    $form['doc_to_html'] = [
      '#type' => 'container',
    ];

    foreach ($field_config as $key => $item) {
      $bundle_title = $item['bundle_title'] ?? $this->t('Unknown bundle');
      $field_label = $item['field_label'] ?? $item['field_name'];

      // Wrapper for each bundle/field combination.
      $form['doc_to_html'][$key] = [
        '#type' => 'details',
        '#open' => FALSE,
        '#title' => $this->t('@bundle', ['@bundle' => $bundle_title]),
      ];

      // Checkbox: enable import to this field.
      $form['doc_to_html'][$key][$key . '-field'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('@field_label', ['@field_label' => $field_label]),
        '#default_value' => (bool)$config->get($key . '-field'),
      ];

      // Optional regex filter for this field.
      $form['doc_to_html'][$key][$key . '-regex_filter'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Regex filter'),
        '#description' => $this->t('Optional. A regular expression applied to the HTML content to remove unwanted tags or attributes.'),
        '#default_value' => $config->get($key . '-regex_filter') ?: '',
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void
  {
    $field_config = $this->defaultService->getEntityBundleFieldBy('text_with_summary');

    // Validate each regex field if provided.
    foreach ($field_config as $key => $item) {
      $regex_value = $form_state->getValue($key . '-regex_filter');
      if (!empty($regex_value)) {
        set_error_handler(static function () {
        }, E_WARNING);
        $is_valid = @preg_match($regex_value, '') !== FALSE;
        $regex_error_message = '';
        if (!$is_valid) {
          $error_code = preg_last_error();
          if (function_exists('preg_last_error_msg')) {
            $regex_error_message = preg_last_error_msg();
          }
          else {
            $regex_error_message = (string) $error_code;
          }
        }
        restore_error_handler();

        if (!$is_valid) {
          $form_state->setErrorByName(
            $key . '-regex_filter',
            $this->t('The provided regular expression for @field is not valid.', [
              '@field' => $item['field_label'] ?? $item['field_name'],
              '@reason' => $regex_error_message ?: $this->t('PCRE error code @code', ['@code' => $error_code ?? 0]),
            ])
          );
        }
      }
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void
  {
    parent::submitForm($form, $form_state);

    $config = $this->configFactory->getEditable('doc_to_html.importtofield');
    $field_config = $this->defaultService->getEntityBundleFieldBy('text_with_summary');

    foreach ($field_config as $key => $item) {
      $config
        ->set($key . '-field', (bool)$form_state->getValue($key . '-field'))
        ->set($key . '-regex_filter', $form_state->getValue($key . '-regex_filter'));
    }

    $config->save();
  }

}
