<?php

namespace Drupal\doc_to_html\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\doc_to_html\Services\DefaultService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for LibreOffice settings used by DOC to HTML.
 */
class LibreOfficeSettings extends ConfigFormBase
{

  /**
   * Provides OS-specific defaults and helper logic.
   */
  protected DefaultService $defaultService;

  public function __construct(DefaultService $defaultService) {
    $this->defaultService = $defaultService;
  }

  public static function create(ContainerInterface $container): static
  {
    return new static(
      $container->get('doc_to_html.default_service'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array
  {
    return [
      'doc_to_html.libreoffice_settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string
  {
    return 'doc_to_html_libreoffice_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array
  {
    $config = $this->config('doc_to_html.libreoffice_settings');

    // Detect current OS and fetch suggested defaults.
    $os = PHP_OS_FAMILY === 'Darwin'
      ? 'Darwin'
      : (PHP_OS_FAMILY === 'Windows' ? 'WINNT' : 'Linux');

    $defaults = $this->defaultService->getSettingsForOS($os);

    $base_path_default = $config->get('base_path_application') ?? $defaults['base_path_application'];
    $command_default = $config->get('command') ?? $defaults['command'];

    $form['base_path_application'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Base path for LibreOffice'),
      '#description' => $this->t('Absolute base path where the LibreOffice executable is located. For example: <code>/usr/bin</code> or <code>/Applications/LibreOffice.app/Contents/MacOS</code>.'),
      '#maxlength' => 255,
      '#size' => 80,
      '#required' => FALSE,
      '#default_value' => $base_path_default,
    ];

    $form['command'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Command'),
      '#description' => $this->t('The executable name used to run LibreOffice (for example <code>soffice</code> or <code>soffice.exe</code>).'),
      '#maxlength' => 255,
      '#size' => 40,
      '#required' => TRUE,
      '#default_value' => $command_default,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void
  {
    // Get the command value from the form state.
    $command = trim((string)$form_state->getValue('command'));

    // Check if the command field is empty and set an error if so.
    if ($command === '') {
      $form_state->setErrorByName('command', $this->t('The command field is required.'));
    }

    // Basic sanity check: avoid path separators in the command name.
    // This ensures the command only contains the executable name, not the directory path.
    if (str_contains($command, '/') || str_contains($command, '\\')) {
      $form_state->setErrorByName('command', $this->t('The command should not contain path separators. Use the base path field for the directory and only the executable name here.'));
    }

    // Call the parent validateForm method to perform any additional validation.
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void
  {
    parent::submitForm($form, $form_state);

    $editable = $this->configFactory->getEditable('doc_to_html.libreoffice_settings');

    $base_path = trim((string)$form_state->getValue('base_path_application'));
    if ($base_path !== '') {
      $editable->set('base_path_application', $base_path);
    }
    else {
      $editable->clear('base_path_application');
    }

    $editable->set('command', trim((string)$form_state->getValue('command')));

    $editable->save();
  }

}
