<?php

namespace Drupal\doc_to_html\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\doc_to_html\Services\DefaultService;
use Drupal\doc_to_html\Services\CmdServiceInterface;
use Drupal\doc_to_html\Services\FileServiceInterface;
use Drupal\doc_to_html\Services\FileCleaner;
use Drupal\doc_to_html\Services\ConversionManager;
use Drupal\Component\Utility\Html;
use Drupal\file\Entity\File;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form controller for the DOC → HTML testing wizard.
 */
class TestWizard extends FormBase
{

  /**
   * The default service.
   */
  protected DefaultService $defaultService;

  /**
   * The command service.
   */
  protected CmdServiceInterface $cmdService;

  /**
   * The file service.
   */
  protected FileServiceInterface $fileService;

  /**
   * The file cleaner service.
   */
  protected FileCleaner $fileCleaner;

  /**
   * Shared conversion manager.
   */
  protected ConversionManager $conversionManager;

  public static function create(ContainerInterface $container): static
  {
    // Manual injection keeps the form portable and mirrors what happens when
    // Drupal builds the service-backed widget. Each collaborator encapsulates a
    // major slice of the conversion pipeline (default lookups, CLI execution,
    // file housekeeping, and the orchestrating ConversionManager).
    $instance = new static();
    $instance->defaultService = $container->get('doc_to_html.default_service');
    $instance->cmdService = $container->get('doc_to_html.cmd_service');
    $instance->fileService = $container->get('doc_to_html.file_service');
    $instance->fileCleaner = $container->get('doc_to_html.file_cleaner');
    $instance->conversionManager = $container->get('doc_to_html.conversion_manager');
    return $instance;
  }

  public function getFormId(): string
  {
    return 'doc_to_html_test_wizard';
  }

  public function buildForm(array $form, FormStateInterface $form_state): array
  {
    // Persist conversion output between AJAX rebuilds. Storing data in
    // form_state allows the wizard to act as a preview sandbox without touching
    // any actual entity field until the site builder copies the final HTML.
    $storage = $form_state->getStorage();

    // Ensure the CKEditor preview always starts from #default_value during
    // AJAX rebuilds triggered by wizard controls, otherwise the user input
    // (usually empty) would win and blank the editor.
    if ($trigger = $form_state->getTriggeringElement()) {
      if (isset($trigger['#parents'])) {
        $refreshParents = [
          ['actions', 'convert'],
          ['actions', 'clear'],
          ['regex', 'preg_match', 'body_regex'],
          ['regex', 'preg_match', 'body_match_index'],
          ['regex', 'dom_regex'],
        ];
        foreach ($refreshParents as $parents) {
          if ($trigger['#parents'] === $parents) {
            $userInput = $form_state->getUserInput();
            if (isset($userInput['previews']['final']['content'])) {
              unset($userInput['previews']['final']['content']);
              $form_state->setUserInput($userInput);
            }
            $form_state->unsetValue(['previews', 'final', 'content', 'value']);
            $form_state->unsetValue(['previews', 'final', 'content', 'format']);
            break;
          }
        }
      }
    }

    // Retrieve cached conversion output so repeated AJAX calls can show deltas
    // without re-running LibreOffice where unnecessary.
    $raw_html = $storage['raw_html'] ?? '';
    $body_only = $storage['body_html'] ?? '';
    $processed_override = $storage['final_html'] ?? '';
    $body_match_count = $storage['body_match_count'] ?? 0;
    $dom_match_count = $storage['dom_match_count'] ?? 0;
    $dom_replaced = $storage['dom_replaced'] ?? 0;

    $body_regex = $form_state->getValue(['regex', 'preg_match', 'body_regex']) ?? '';
    $body_match_index_val = $form_state->getValue(['regex', 'preg_match', 'body_match_index']);
    $body_match_index = is_numeric($body_match_index_val) ? (int)$body_match_index_val : NULL;
    $dom_regex = $form_state->getValue(['regex', 'dom_regex']) ?? '';

    $basic_config = \Drupal::config('doc_to_html.basic_settings');
    $upload_folder = $basic_config->get('doc_to_html_folder') ?? 'doc_to_html';
    // Pull defaults from configuration so the wizard mirrors the widget
    // behaviour. Site builders can experiment here before committing settings.
    $conf_body_regex = (string)($basic_config->get('regex_to_parse_body') ?? '');
    $conf_match_index = $basic_config->get('regex_body_match_index');
    $conf_match_index = is_numeric($conf_match_index) ? (int)$conf_match_index : 0;
    $conf_dom_regex = (string)($basic_config->get('dom_regex') ?? '');
    if ($body_regex === '' && $conf_body_regex !== '') {
      $body_regex = $conf_body_regex;
    }
    if ($body_match_index === NULL) {
      $body_match_index = $conf_match_index;
    }
    if ($dom_regex === '' && $conf_dom_regex !== '') {
      $dom_regex = $conf_dom_regex;
    }
    $dom_regex_active = is_string($dom_regex) && $dom_regex !== '';

    $upload_location = 'public://' . $upload_folder;

    // Clean stale file entities pointing to the working folder before rendering
    // to avoid growth of unused managed files when the wizard is reopened often.
    $this->fileCleaner->cleanMissingFiles($upload_location);

    $libre = $this->cmdService->getLibreOfficeVersion();

    $form['info'] = [
      '#type' => 'item',
      '#title' => $this->t('LibreOffice'),
      '#markup' => $libre ? $this->t('Detected: @v', ['@v' => $libre]) : $this->t('LibreOffice not detected. Configure it in settings.'),
    ];

    $supported_extensions = $this->defaultService->getSupportedFileExtensions();
    $extensions_string = implode(' ', $supported_extensions);

    // Ensure target directory exists before exposing the managed_file element so
    // editors receive immediate feedback if the configured folder is invalid.
    $this->fileService->prepareDirectory($upload_folder);

    $form['source'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('Upload DOC/DOCX'),
      '#upload_location' => $upload_location,
      '#upload_validators' => [
        'FileExtension' => ['extensions' => $extensions_string ?: 'doc docx'],
      ],
      '#description' => $this->t('Upload a file, then click Convert to preview.'),
    ];

    // Conversion controls: convert/clear for previews, save settings to persist
    // regex defaults back into configuration.
    $form['actions'] = [
      '#type' => 'actions',
    ];
    $form['actions']['convert'] = [
      '#type' => 'submit',
      '#value' => $this->t('Convert'),
      '#submit' => ['::convertSubmit'],
      '#ajax' => [
        'callback' => '::refreshPreview',
        'wrapper' => 'doc-to-html-previews',
      ],
    ];
    $form['actions']['clear'] = [
      '#type' => 'submit',
      '#value' => $this->t('Clear'),
      '#limit_validation_errors' => [],
      '#submit' => ['::clearSubmit'],
      '#ajax' => [
        'callback' => '::refreshPreview',
        'wrapper' => 'doc-to-html-previews',
      ],
    ];

    $form['actions']['save_settings'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save settings'),
      '#submit' => ['::saveBodySettingsSubmit'],
      '#limit_validation_errors' => [
        ['regex', 'preg_match', 'body_regex'],
        ['regex', 'preg_match', 'body_match_index'],
        ['regex', 'dom_regex'],
      ],
    ];

    // Live regex playground lets site builders adjust preg_match/preg_replace
    // parameters and instantly observe impact on the converted markup.
    $form['regex'] = [
      '#type' => 'details',
      '#title' => $this->t('Regex testing (optional)'),
      '#open' => TRUE,
      '#tree' => TRUE,
    ];
    $form['regex']['preg_match'] = [
      '#type' => 'details',
      '#title' => $this->t('Body extraction (preg_match)'),
      '#open' => TRUE,
    ];
    $form['regex']['preg_match']['body_regex'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Body extraction regex (preg_match)'),
      '#description' => $this->t('Regular expression used with preg_match to extract the HTML body. Include delimiters and modifiers, e.g. /<body[\s\S]*?<\/body>/is. If you use capture groups, select which match to return via Body match index.'),
      '#default_value' => $body_regex,
      '#ajax' => [
        'callback' => '::refreshPreview',
        'wrapper' => 'doc-to-html-previews',
        'event' => 'keyup',
      ],
    ];
    $form['regex']['dom_regex'] = [
      '#type' => 'textfield',
      '#title' => $this->t('DOM parsing regex (preg_replace)'),
      '#description' => $this->t('Example: /<span[^>]*>/i'),
      '#default_value' => $dom_regex,
      '#ajax' => [
        'callback' => '::refreshPreview',
        'wrapper' => 'doc-to-html-previews',
        'event' => 'keyup',
      ],
    ];
    $form['regex']['settings_hint'] = [
      '#type' => 'item',
      '#markup' => $this->t('Regexes saved here become default values during import. Field widgets can choose whether to apply them. Use the "Save settings" button at the bottom of this page to persist your changes.'),
    ];
    $form['regex']['preg_match']['body_match_index'] = [
      '#type' => 'number',
      '#title' => $this->t('Body match index'),
      '#description' => $this->t('Index of the match/group to return from preg_match. 0 returns the entire match; 1+ returns the corresponding capture group. Required when a body regex is provided.'),
      '#default_value' => $body_match_index,
      '#min' => 0,
      '#step' => 1,
      '#ajax' => [
        'callback' => '::refreshPreview',
        'wrapper' => 'doc-to-html-previews',
        'event' => 'change',
      ],
    ];

    // Compute intermediate and final previews with match counts so users can
    // gauge how many segments were kept or removed by each regex stage.
    if ($body_only === '') {
      $body_only = $raw_html;
    }

    $processed = $processed_override !== '' ? $processed_override : $body_only;

    if ($processed === '' && $dom_regex_active) {
      $processed = $body_only;
    }

    if (!$this->isSaveSettingsTrigger($form_state)) {
      // Set form values so the text_format element picks up the processed HTML
      // even when Drupal prefers form_state input over #default_value.
      $form_state->setValue(['previews', 'final', 'content'], [
        'value' => $processed,
        'format' => 'full_html',
      ]);
      // Also set raw user input for the widget so AJAX rebuilds render CKEditor
      // with the processed HTML instead of an empty value.
      $userInput = $form_state->getUserInput();
      if (!isset($userInput['previews'])) {
        $userInput['previews'] = [];
      }
      if (!isset($userInput['previews']['final'])) {
        $userInput['previews']['final'] = [];
      }
      $userInput['previews']['final']['content'] = [
        'value' => $processed,
        'format' => 'full_html',
      ];
      $form_state->setUserInput($userInput);
    }

    $form['previews'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'doc-to-html-previews'],
      '#tree' => TRUE,
    ];
    $form['previews']['raw'] = [
      '#type' => 'details',
      '#title' => $this->t('Raw HTML (converted, unprocessed)'),
      '#open' => FALSE,
      'content' => [
        '#type' => 'item',
        '#markup' => '<div class="doc-to-html-raw-preview">' . ($raw_html ?: '') . '</div>',
      ],
      'content_text' => [
        '#type' => 'details',
        '#title' => $this->t('Raw HTML (plain text)'),
        '#open' => FALSE,
        'pre' => [
          '#type' => 'item',
          '#markup' => '<pre style="white-space:pre-wrap;">' . Html::escape($raw_html ?: '') . '</pre>',
        ],
      ],
    ];
    $form['previews']['body'] = [
      '#type' => 'details',
      '#title' => $this->t('Extracted body') . ($body_regex ? ' (' . $this->t('match: @n, idx: @i', ['@n' => $body_match_count, '@i' => $body_match_index]) . ')' : ''),
      '#open' => FALSE,
      'content' => [
        '#type' => 'item',
        '#markup' => '<div class="doc-to-html-body-preview">' . ($body_only ?: '') . '</div>',
      ],
      'content_text' => [
        '#type' => 'details',
        '#title' => $this->t('Body (plain text)'),
        '#open' => FALSE,
        'pre' => [
          '#type' => 'item',
          '#markup' => '<pre style="white-space:pre-wrap;">' . Html::escape($body_only ?: '') . '</pre>',
        ],
      ],
    ];
    $form['previews']['final'] = [
      '#type' => 'details',
      '#title' => $this->t('Final preview (body + DOM regex)') . ($dom_regex_active ? ' (' . $this->t('match: @m, replaced: @r', ['@m' => $dom_match_count, '@r' => $dom_replaced]) . ')' : ''),
      '#open' => FALSE,
      'content' => [
        '#type' => 'text_format',
        '#title' => $this->t('Preview'),
        '#format' => 'full_html',
        '#allowed_formats' => ['full_html'],
        '#description' => $this->t('Processed preview (not saved).'),
        '#default_value' => $processed,
      ],
      'content_text' => [
        '#type' => 'details',
        '#title' => $this->t('Final (plain text)'),
        '#open' => FALSE,
        'pre' => [
          '#type' => 'item',
          '#markup' => '<pre style="white-space:pre-wrap;">' . Html::escape($processed ?: '') . '</pre>',
        ],
      ],
    ];

    return $form;
  }

  /**
   * @param array $form
   * @param FormStateInterface $form_state
   * @return void
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void
  {
    $body_regex = (string)$form_state->getValue(['regex', 'preg_match', 'body_regex']);
    if ($body_regex !== '') {
      $matchValidation = $this->conversionManager->validateRegexForMatch($body_regex);
      if (!$matchValidation->isSuccess()) {
        $form_state->setErrorByName('regex][preg_match][body_regex', $this->t('Invalid body regex for preg_match. @msg', ['@msg' => implode(' ', $matchValidation->getErrors())]));
      }
    }
    $idx = $form_state->getValue(['regex', 'preg_match', 'body_match_index']);
    if ($idx !== NULL && $idx !== '') {
      if (!is_numeric($idx) || (int)$idx < 0) {
        $form_state->setErrorByName('regex][preg_match][body_match_index', $this->t('Body match index must be a non-negative integer.'));
      }
    }
    $dom_regex = (string)$form_state->getValue(['regex', 'dom_regex']);
    if ($dom_regex !== '') {
      $matchValidation = $this->conversionManager->validateRegexForMatch($dom_regex);
      if (!$matchValidation->isSuccess()) {
        $form_state->setErrorByName('regex][dom_regex', $this->t('Invalid DOM regex for preg_match. @msg', ['@msg' => implode(' ', $matchValidation->getErrors())]));
      }
      else {
        $replaceValidation = $this->conversionManager->validateRegexForReplace($dom_regex);
        if (!$replaceValidation->isSuccess()) {
          $form_state->setErrorByName('regex][dom_regex', $this->t('Invalid DOM regex for preg_replace. @msg', ['@msg' => implode(' ', $replaceValidation->getErrors())]));
        }
      }
    }
  }

  /**
   * @param array $form
   * @param FormStateInterface $form_state
   * @return void
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void
  {
    // No-op. We use custom submit handlers.
  }

  /**
   * @param array $form
   * @param FormStateInterface $form_state
   * @return void
   */
  public function convertSubmit(array &$form, FormStateInterface $form_state): void
  {

    $fid_value = $form_state->getValue('source');
    $fid = is_array($fid_value) ? (reset($fid_value) ?: NULL) : $fid_value;
    if (!$fid) {
      $this->messenger()->addWarning($this->t('Please upload a file first.'));
      return;
    }

    $options = [
      'apply_body_regex' => TRUE,
      'body_regex' => $form_state->getValue(['regex', 'preg_match', 'body_regex']) ?? '',
      'body_match_index' => $form_state->getValue(['regex', 'preg_match', 'body_match_index']) ?? NULL,
      'dom_regex' => $form_state->getValue(['regex', 'dom_regex']) ?? '',
    ];

    $result = $this->conversionManager->convert((int) $fid, $options);

    if (!$result->isSuccess()) {
      foreach ($result->getErrors() as $error) {
        $this->messenger()->addError($error);
      }
      return;
    }

    foreach ($result->getWarnings() as $warning) {
      $this->messenger()->addWarning($warning);
    }

    $storage = $form_state->getStorage();
    $storage['raw_html'] = $result->get('raw_html', '');
    $storage['body_html'] = $result->get('body_html', '');
    $storage['final_html'] = $result->get('final_html', '');
    $storage['body_match_count'] = $result->get('body_match_count', 0);
    $storage['dom_match_count'] = $result->get('dom_match_count', 0);
    $storage['dom_replaced'] = $result->get('dom_replaced', 0);
    $form_state->setStorage($storage);

    $form_state->unsetValue(['previews', 'final', 'content', 'value']);
    $form_state->unsetValue(['previews', 'final', 'content', 'format']);
    $form_state->setRebuild();
  }

  /**
   * @param array $form
   * @param FormStateInterface $form_state
   * @return void
   */
  public function clearSubmit(array &$form, FormStateInterface $form_state): void
  {
    $form_state->setStorage([]);
    $form_state->setValue('source', []);
    $form_state->setValue(['regex', 'preg_match', 'body_regex'], '');
    $form_state->setValue(['regex', 'dom_regex'], '');
    $form_state->setValue(['regex', 'preg_match', 'body_match_index'], 0);
    $this->fileService->cleanFolder();
    // Clear submitted children so CKEditor uses #default_value on rebuild.
    $form_state->unsetValue(['previews', 'final', 'content', 'value']);
    $form_state->unsetValue(['previews', 'final', 'content', 'format']);
    $form_state->setRebuild();
  }

  /**
   * @param array $form
   * @param FormStateInterface $form_state
   * @return array
   */
  public function refreshPreview(array &$form, FormStateInterface $form_state): array
  {
    return $form['previews'];
  }

  /**
   * @param array $form
   * @param FormStateInterface $form_state
   * @return void
   */
  public function saveBodySettingsSubmit(array &$form, FormStateInterface $form_state): void
  {
    if ($form_state->hasAnyErrors()) {
      $form_state->setRebuild();
      return;
    }

    $input = $form_state->getUserInput();
    $regexInput = $input['regex']['preg_match']['body_regex'] ?? $form_state->getValue(['regex', 'preg_match', 'body_regex']);
    $idx = $input['regex']['preg_match']['body_match_index'] ?? $form_state->getValue(['regex', 'preg_match', 'body_match_index']);
    $domRegexInput = $input['regex']['dom_regex'] ?? $form_state->getValue(['regex', 'dom_regex']);

    $regex = trim((string)($regexInput ?? ''));
    $dom_regex = trim((string)($domRegexInput ?? ''));
    $validationFailed = FALSE;
    if ($regex !== '') {
      $validation = $this->conversionManager->validateRegexForMatch($regex);
      if (!$validation->isSuccess()) {
        $form_state->setErrorByName('regex][preg_match][body_regex', $this->t('Invalid body regex for preg_match. @msg', ['@msg' => implode(' ', $validation->getErrors())]));
        $validationFailed = TRUE;
      }
    }
    if ($dom_regex !== '') {
      $matchValidation = $this->conversionManager->validateRegexForMatch($dom_regex);
      if (!$matchValidation->isSuccess()) {
        $form_state->setErrorByName('regex][dom_regex', $this->t('Invalid DOM regex for preg_match. @msg', ['@msg' => implode(' ', $matchValidation->getErrors())]));
        $validationFailed = TRUE;
      }
      else {
        $replaceValidation = $this->conversionManager->validateRegexForReplace($dom_regex);
        if (!$replaceValidation->isSuccess()) {
          $form_state->setErrorByName('regex][dom_regex', $this->t('Invalid DOM regex for preg_replace. @msg', ['@msg' => implode(' ', $replaceValidation->getErrors())]));
          $validationFailed = TRUE;
        }
      }
    }
    if ($validationFailed) {
      $form_state->setRebuild();
      return;
    }
    $editable = \Drupal::configFactory()->getEditable('doc_to_html.basic_settings');
    if ($regex !== '') {
      $editable->set('regex_to_parse_body', $regex);
    } else {
      $editable->clear('regex_to_parse_body');
    }
    if ($idx !== NULL && $idx !== '' && is_numeric($idx) && (int)$idx >= 0) {
      $editable->set('regex_body_match_index', (int)$idx);
    }
    if ($dom_regex !== '') {
      $editable->set('dom_regex', $dom_regex);
    } else {
      $editable->clear('dom_regex');
    }
    $editable->save();

    \Drupal::configFactory()->reset('doc_to_html.basic_settings');

    // Clean working directory and stale entities to reflect the saved defaults.
    try {
      $this->fileService->cleanFolder();

      $config = \Drupal::config('doc_to_html.basic_settings');
      $folder = (string) ($config->get('doc_to_html_folder') ?: 'doc_to_html');
      $directory = 'public://' . $folder;

      $this->fileCleaner->cleanMissingFiles($directory);
    } catch (\Throwable $throwable) {
      \Drupal::logger('doc_to_html')->error('Unable to clean directory after saving defaults: @message', [
        '@message' => $throwable->getMessage(),
      ]);
    }

    $form_state->setValue(['regex', 'preg_match', 'body_regex'], $regex);
    if ($idx !== NULL && $idx !== '' && is_numeric($idx)) {
      $form_state->setValue(['regex', 'preg_match', 'body_match_index'], (int) $idx);
    }
    $form_state->setValue(['regex', 'dom_regex'], $dom_regex);

    $userInput = $form_state->getUserInput();
    if (!is_array($userInput)) {
      $userInput = [];
    }
    if (!isset($userInput['regex'])) {
      $userInput['regex'] = [];
    }
    if (!isset($userInput['regex']['preg_match'])) {
      $userInput['regex']['preg_match'] = [];
    }
    $userInput['regex']['preg_match']['body_regex'] = $regex;
    if ($idx !== NULL && $idx !== '' && is_numeric($idx)) {
      $userInput['regex']['preg_match']['body_match_index'] = (int) $idx;
    }
    $userInput['regex']['dom_regex'] = $dom_regex;
    $form_state->setUserInput($userInput);

    $messenger = $this->messenger();
    $messenger->deleteByType('status');
    $messenger->addStatus($this->t('Defaults saved.'));
    $form_state->setRebuild();
  }

  /**
   * @param string $pattern
   * @param string|null $message
   * @return bool
   */
  private function isRegexValidForMatch(string $pattern, ?string &$message = NULL): bool
  {
    set_error_handler(static function () {
    }, E_WARNING);
    $result = @preg_match($pattern, '') !== false;
    $message = ($result || !function_exists('preg_last_error_msg')) ? '' : preg_last_error_msg();
    restore_error_handler();
    return $result;
  }

  /**
   * @param string $pattern
   * @param string|null $message
   * @return bool
   */
  private function isRegexValidForReplace(string $pattern, ?string &$message = NULL): bool
  {
    set_error_handler(static function () {
    }, E_WARNING);
    $result = @preg_replace($pattern, '', '') !== NULL;
    $message = ($result || !function_exists('preg_last_error_msg')) ? '' : preg_last_error_msg();
    restore_error_handler();
    return $result;
  }

  /**
   * Checks if the save settings trigger has been activated.
   *
   * @param FormStateInterface $form_state
   *   The form state object containing the triggering element.
   *
   * @return bool
   *   TRUE if the save settings trigger is activated, FALSE otherwise.
   */
  private function isSaveSettingsTrigger(FormStateInterface $form_state): bool
  {
    $trigger = $form_state->getTriggeringElement();
    return isset($trigger['#parents']) && $trigger['#parents'] === ['actions', 'save_settings'];
  }
}
