<?php

namespace Drupal\doc_to_html\Plugin\Field\FieldWidget;

use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Field\Annotation\FieldWidget;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\doc_to_html\Services\FileServiceInterface;
use Drupal\doc_to_html\Services\ConversionManagerInterface;
use Drupal\doc_to_html\Services\DefaultService;
use Drupal\text\Plugin\Field\FieldWidget\TextareaWithSummaryWidget;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Routing\TrustedRedirectResponse;

/**
 * @FieldWidget(
 *   id = "doc_to_html_widget",
 *   label = @Translation("DOC to HTML"),
 *   field_types = {
 *     "text_with_summary",
 *     "text_long"
 *   }
 * )
 */
class DocToHtmlWidget extends TextareaWithSummaryWidget implements ContainerFactoryPluginInterface
{

  protected $basicConfig;

  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    array $third_party_settings,
    protected ConfigFactoryInterface $configFactory,
    protected FileServiceInterface $fileService,
    protected ConversionManagerInterface $conversionManager,
    protected $messenger,
    protected DefaultService $defaultService
  )
  {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
    $this->basicConfig = $this->configFactory->get('doc_to_html.basic_settings');
  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static
  {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('config.factory'),
      $container->get('doc_to_html.file_service'),
      $container->get('doc_to_html.conversion_manager'),
      $container->get('messenger'),
      $container->get('doc_to_html.default_service')
    );
  }

  public static function defaultSettings(): array
  {
    return [
        'default_apply_body_regex' => FALSE,
        'default_dom_regex_override' => '',
      ] + parent::defaultSettings();
  }

  /**
   * -------------------------------------------------------
   * FORM WIDGET
   * -------------------------------------------------------
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state): array
  {

    // 1. Render the original textarea/summary widget to preserve Drupal defaults.
    $original = parent::formElement($items, $delta, $element, $form, $form_state);

    if (isset($original['value'])) {
      $original['value']['#format'] = 'full_html';
      $original['value']['#allowed_formats'] = ['full_html'];
    }

    // 2. Wrap the original widget so we can append conversion controls.
    $wrapper = [
      '#type' => 'container',
      '#tree' => TRUE,
    ];

    // Copy the original widget structure into the wrapper.
    foreach ($original as $key => $value) {
      $wrapper[$key] = $value;
    }

    // 3. Determine the parent path for this field delta: #field_parents + field name + delta.
    $field_parents = $element['#field_parents'] ?? [];
    $field_name = $this->fieldDefinition->getName();
    $containerParents = array_merge($field_parents, [$field_name, $delta]);

    // Parents for the text_format value element.
    $valueParents = array_merge($containerParents, ['value']);

    // -----------------------------------------------------
    // File upload + conversion controls.
    // -----------------------------------------------------
    $wrapper['doc_to_html'] = [
      '#type' => 'details',
      '#title' => $this->t('DOC to HTML file'),
      '#open' => FALSE,
      '#tree' => TRUE,
      '#weight' => 100,
    ];

    $uploadLocation = $this->prepareUploadDirectory();
    $extensions = implode(' ', $this->defaultService->getSupportedFileExtensions()) ?: 'doc docx';

    $wrapper['doc_to_html']['doc_source'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('Document to import'),
      '#upload_location' => $uploadLocation,
      '#upload_validators' => [
        'FileExtension' => ['extensions' => $extensions],
      ],
      '#weight' => -10,
    ];

    // Store the path to the managed_file element for submit/AJAX handlers.
    // Example: ['body', 0, 'doc_to_html', 'doc_source'].
    $docSourceParents = array_merge($containerParents, ['doc_to_html', 'doc_source']);

    $wrapper['doc_to_html']['actions'] = [
      '#type' => 'actions',
      '#weight' => -9,
    ];

    $wrapper['doc_to_html']['actions']['convert'] = [
      '#type' => 'submit',
      '#value' => $this->t('Convert'),
      // Only our submit handler should run.
      '#submit' => [[$this, 'convertSubmit']],
      // Validate only the managed_file path so other field constraints do not block conversions.
      '#limit_validation_errors' => [
        $docSourceParents,
      ],
      // Metadata used by the submit/AJAX handlers to locate elements.
      '#doc_to_html_source_parents' => $docSourceParents,
      '#doc_to_html_value_parents' => $valueParents,
      '#doc_to_html_container_parents' => $containerParents,
    ];

    return $wrapper;
  }




  public function convertSubmit(array &$form, FormStateInterface $form_state): void
  {
    $trigger = $form_state->getTriggeringElement();

    $docSourceParents = $trigger['#doc_to_html_source_parents'] ?? NULL;
    $valueParents = $trigger['#doc_to_html_value_parents'] ?? NULL;

    if (!$docSourceParents || !$valueParents) {
      $this->messenger->addError($this->t('Internal error: widget parents not found.'));
      return;
    }

    // 1) Recupero FID
    $fidValue = $form_state->getValue($docSourceParents);
    $fid = is_array($fidValue) ? (reset($fidValue) ?: NULL) : $fidValue;

    if (!$fid) {
      $this->messenger->addWarning($this->t('Upload a DOC/DOCX file before converting.'));
      return;
    }

    // 2) Conversione
    $options = $this->buildConversionOptions();
    $result = $this->conversionManager->convert((int)$fid, $options);

    if (!$result->isSuccess()) {
      foreach ($result->getErrors() as $error) {
        $this->messenger->addError($error);
      }
      return;
    }

    $converted = $result->get('final_html');
    if ($converted === NULL) {
      $this->messenger->addError($this->t('Conversion produced no output.'));
      return;
    }

    // 3) Prendo l’entità da modificare
    $form_object = $form_state->getFormObject();

    $entity = NULL;
    if (method_exists($form_object, 'getEntity')) {
      $entity = $form_object->getEntity();
    }
    //$entity = $form_state->getFormObject()->getEntity();
    $field_name = $this->fieldDefinition->getName();

    // 4) Aggiorno il valore del campo **direttamente sull'entità**
    $entity->set($field_name, [
      'value' => $converted,
      'format' => 'full_html',
    ]);

    // 5) Salvo l'entità subito
    $entity->save();

    $this->messenger->addStatus($this->t('Document converted and saved successfully.'));

    // 6) Redirect allo stesso nodo (edit)
    $url = $entity->toUrl('edit-form');
    $response = new TrustedRedirectResponse($url->toString());

    $form_state->setResponse($response);
  }


  /**
   * Salva html convertito nel valore finale.
   *
   * This method updates both FormState values and raw user input,
   * so rebuilt form widgets (including CKEditor) will use the new data.
   */
  private function applyConvertedHtml(FormStateInterface $form_state, array $value_parents, string $converted): void
  {
    // Current value of the text_format (value + format + optional summary).
    $current = $form_state->getValue($value_parents) ?? [];
    $format = $current['format'] ?? 'full_html';
    $summary = $current['summary'] ?? '';

    $newValue = [
      'value' => $converted,
      'format' => $format,
      'summary' => $summary,
    ];

    // 1) Update FormState values.
    $form_state->setValue($value_parents, $newValue);

    // 2) Update raw user input so widgets take this value on rebuild.
    $userInput = $form_state->getUserInput();
    if (!is_array($userInput)) {
      $userInput = [];
    }
    NestedArray::setValue($userInput, $value_parents, $newValue);
    $form_state->setUserInput($userInput);

  }


  private function clearManagedFile(FormStateInterface $form_state, array $source_parents): void
  {
    $form_state->setValue($source_parents, []);
    $userInput = $form_state->getUserInput();
    NestedArray::setValue($userInput, $source_parents, []);
    $form_state->setUserInput($userInput);
  }

  /**
   * Conversion settings
   */
  private function buildConversionOptions(): array
  {
    $defaults = [
      'apply_body_regex' => (bool)$this->getSetting('default_apply_body_regex'),
      'body_regex' => $this->basicConfig->get('regex_to_parse_body') ?? '',
      'body_match_index' => $this->basicConfig->get('regex_body_match_index'),
      'dom_regex' => $this->basicConfig->get('dom_regex') ?? '',
    ];

    if ($defaults['apply_body_regex'] === FALSE) {
      $defaults['body_regex'] = '';
      $defaults['body_match_index'] = NULL;
    }

    $override = (string)$this->getSetting('default_dom_regex_override');
    if ($override !== '') {
      $defaults['dom_regex'] = $override;
    }

    return $defaults;
  }

  /**
   * Directory upload sicura
   */
  private function prepareUploadDirectory(): string
  {

    $folder = trim((string)($this->basicConfig->get('doc_to_html_folder') ?? 'doc_to_html'));

    // Normalizzazione: impedisce doppio "public://"
    $folder = str_replace('public://', '', $folder);
    $folder = ltrim($folder, '/');

    $this->fileService->prepareDirectory($folder);

    return 'public://' . $folder;
  }
}
