<?php

namespace Drupal\doc_to_html\Services;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Psr\Log\LoggerInterface;

/**
 * Executes LibreOffice commands and performs DOC/DOCX → HTML conversions.
 */
class CmdService implements CmdServiceInterface
{

  public function __construct(
    protected ConfigFactoryInterface     $configFactory,
    protected FileServiceInterface       $fileService,
    protected MarkupServiceInterface     $markupService,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected LoggerInterface            $logger,
  )
  {
  }

  /**
   * {@inheritdoc}
   */
  public function getLibreOfficeVersion(): ?string
  {
    $config = $this->configFactory->get('doc_to_html.libreoffice_settings');
    $basePath = rtrim((string) ($config->get('base_path_application') ?? ''), DIRECTORY_SEPARATOR);
    $command = $config->get('command');

    if (empty($command)) {
      $this->logger->error('LibreOffice command is not configured.');
      return NULL;
    }

    // Build the executable path; fall back to PATH lookups when no base path is
    // stored. Consumers rely on the version string to expose helpful diagnostics
    // in the Test Wizard and the status report.
    $executable = $basePath ? "{$basePath}/{$command}" : $command;

    $output = [];
    $code = 0;

    @exec(escapeshellcmd($executable) . ' --version 2>&1', $output, $code);

    return $code === 0 && !empty($output) ? implode(' | ', $output) : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function convertFile(string $sourceUri, bool $applyRegex = FALSE, ?string $overrideBodyRegex = NULL, ?int $overrideBodyMatchIndex = NULL): ?string
  {
    $config = $this->configFactory->get('doc_to_html.basic_settings');

    $folder = $config->get('doc_to_html_folder') ?? 'doc_to_html';
    $destinationDir = $this->fileService->prepareDirectory($folder);

    $sourcePath = $this->fileService->escapeRealPath($sourceUri);
    $destinationPath = $destinationDir;

    // Build command.
    $libConfig = $this->configFactory->get('doc_to_html.libreoffice_settings');
    $basePath = rtrim((string) ($libConfig->get('base_path_application') ?? ''), DIRECTORY_SEPARATOR);
    $command = $libConfig->get('command');

    if (empty($command)) {
      $this->logger->error('LibreOffice command is missing.');
      return NULL;
    }

    $executable = $basePath ? "{$basePath}/{$command}" : $command;

    // Run LibreOffice in headless mode to produce an HTML rendition next to the
    // source document. Output is redirected so logging captures potential
    // failures and users get actionable feedback.
    $cmd = escapeshellcmd($executable)
      . ' --headless --convert-to html '
      . escapeshellarg($sourcePath)
      . ' --outdir ' . escapeshellarg($destinationPath)
      . ' 2>&1';

    $output = shell_exec($cmd);

    if (!$output) {
      $this->logger->error('LibreOffice conversion failed: @cmd', ['@cmd' => $cmd]);
      return NULL;
    }

    // Let the markup service extract final HTML.
    return $this->markupService->parseConvertedHtml($sourceUri, $applyRegex, $overrideBodyRegex, $overrideBodyMatchIndex);
  }

}
