<?php

namespace Drupal\doc_to_html\Services;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\file\Entity\File;
use Psr\Log\LoggerInterface;

/**
 * Centralizes DOC/DOCX → HTML conversion logic and related helpers.
 */
class ConversionManager implements ConversionManagerInterface
{

  public function __construct(
    protected CmdServiceInterface $cmdService,
    protected FileServiceInterface $fileService,
    protected ConfigFactoryInterface $configFactory,
    protected LoggerInterface $logger,
  ) {
  }

  /**
   * Performs the DOC/DOCX → HTML conversion workflow.
   */
  public function convert(int $fid, array $options = []): ConversionResult
  {
    $file = File::load($fid);
    if (!$file) {
      return ConversionResult::failure(['Unable to load the uploaded file.']);
    }

    $config = $this->configFactory->get('doc_to_html.basic_settings');
    $bodyRegex = $options['body_regex'] ?? $config->get('regex_to_parse_body') ?? '';
    $applyBody = (bool) ($options['apply_body_regex'] ?? FALSE);
    $matchIndex = $options['body_match_index'] ?? $config->get('regex_body_match_index');
    $matchIndex = is_numeric($matchIndex) ? max(0, (int) $matchIndex) : 0;
    $domRegex = $options['dom_regex'] ?? $config->get('dom_regex') ?? '';

    $conversionWarnings = [];

    try {
      $html = $this->cmdService->convertFile(
        $file->getFileUri(),
        FALSE,
        $applyBody && $bodyRegex !== '' ? $bodyRegex : NULL,
        $applyBody && $bodyRegex !== '' ? $matchIndex : NULL
      );
    }
    catch (\Throwable $throwable) {
      $this->logger->error('DOC to HTML conversion failed: @message', ['@message' => $throwable->getMessage()]);
      $this->cleanupSourceFile($file);
      return ConversionResult::failure(['Conversion failed. Check server logs for details.']);
    }

    if ($html === NULL) {
      $this->cleanupSourceFile($file);
      return ConversionResult::failure(['Conversion failed. Check LibreOffice configuration.']);
    }

    $rawHtml = $this->readConvertedFile($file) ?? $html;

    $bodyHtml = $rawHtml;
    $bodyMatches = 0;
    if ($applyBody && $bodyRegex !== '') {
      $result = $this->applyBodyExtraction($rawHtml, $bodyRegex, $matchIndex);
      if ($result === NULL) {
        $this->cleanupSourceFile($file);
        return ConversionResult::failure(['The body extraction regex could not be applied.']);
      }
      $bodyHtml = $result['markup'];
      $bodyMatches = $result['matches'];
      if ($result['warning']) {
        $conversionWarnings[] = $result['warning'];
      }
    }

    $finalHtml = $bodyHtml;
    $domMatches = 0;
    $domReplaced = 0;
    if ($domRegex !== '') {
      $domResult = $this->applyDomRegex($bodyHtml, $domRegex);
      if ($domResult === NULL) {
        $this->cleanupSourceFile($file);
        return ConversionResult::failure(['The DOM regex could not be applied.']);
      }
      $finalHtml = $domResult['markup'];
      $domMatches = $domResult['matches'];
      $domReplaced = $domResult['replaced'];
    }

    $this->cleanupSourceFile($file);

    return ConversionResult::success([
      'raw_html' => trim($rawHtml),
      'body_html' => trim($bodyHtml),
      'final_html' => trim($finalHtml),
      'body_match_count' => $bodyMatches,
      'dom_match_count' => $domMatches,
      'dom_replaced' => $domReplaced,
    ], $conversionWarnings);
  }

  /**
   * Validates that a regex is usable with preg_match.
   */
  public function validateRegexForMatch(string $pattern): ConversionResult
  {
    set_error_handler(static function () {
    }, E_WARNING);
    $isValid = @preg_match($pattern, '') !== FALSE;
    $message = function_exists('preg_last_error_msg') ? preg_last_error_msg() : '';
    restore_error_handler();

    if (!$isValid) {
      return ConversionResult::failure([
        $message !== '' ? $message : 'Invalid regular expression.',
      ]);
    }

    return ConversionResult::success([]);
  }

  /**
   * Validates that a regex is usable with preg_replace.
   */
  public function validateRegexForReplace(string $pattern): ConversionResult
  {
    set_error_handler(static function () {
    }, E_WARNING);
    $result = @preg_replace($pattern, '', '') !== NULL;
    $message = function_exists('preg_last_error_msg') ? preg_last_error_msg() : '';
    restore_error_handler();

    if (!$result) {
      return ConversionResult::failure([
        $message !== '' ? $message : 'Invalid regular expression.',
      ]);
    }

    return ConversionResult::success([]);
  }

  private function readConvertedFile(File $file): ?string
  {
    $htmlUri = $this->fileService->convertUriToHtml($file->getFileUri());
    $realPath = $this->fileService->realPath($htmlUri);

    if (!$realPath || !is_file($realPath)) {
      $this->logger->warning('Converted HTML file not found: @file', ['@file' => $htmlUri]);
      return NULL;
    }

    $contents = file_get_contents($realPath);
    if ($contents === FALSE) {
      $this->logger->warning('Unable to read converted HTML file: @file', ['@file' => $htmlUri]);
      return NULL;
    }

    return $contents;
  }

  private function applyBodyExtraction(string $html, string $regex, int $matchIndex): ?array
  {
    set_error_handler(static function () {
    }, E_WARNING);
    $matches = [];
    $resultAll = @preg_match_all($regex, $html, $matches) ?: 0;
    $single = [];
    $resultSingle = @preg_match($regex, $html, $single);
    $error = function_exists('preg_last_error_msg') ? preg_last_error_msg() : '';
    restore_error_handler();

    if ($resultSingle === FALSE) {
      $this->logger->warning('Body extraction regex failed: @message', ['@message' => $error]);
      return NULL;
    }

    $body = $html;
    $warning = NULL;
    if ($resultSingle === 1) {
      if (array_key_exists($matchIndex, $single)) {
        $body = (string) $single[$matchIndex];
      }
      elseif (isset($single[0])) {
        $body = (string) $single[0];
        $warning = 'Configured match index not found; using full match.';
      }
    }

    return [
      'markup' => trim($body),
      'matches' => (int) $resultAll,
      'warning' => $warning,
    ];
  }

  private function applyDomRegex(string $html, string $regex): ?array
  {
    set_error_handler(static function () {
    }, E_WARNING);
    $matches = @preg_match_all($regex, $html, $tmp) ?: 0;
    $replacedCount = 0;
    $processed = @preg_replace($regex, '', $html, -1, $replacedCount);
    $error = function_exists('preg_last_error_msg') ? preg_last_error_msg() : '';
    restore_error_handler();

    if ($processed === NULL) {
      $this->logger->warning('DOM regex failed: @message', ['@message' => $error]);
      return NULL;
    }

    return [
      'markup' => (string) $processed,
      'matches' => (int) $matches,
      'replaced' => (int) $replacedCount,
    ];
  }

  private function cleanupSourceFile(File $file): void
  {
    try {
      $file->delete();
    }
    catch (\Throwable $throwable) {
      $this->logger->warning('Unable to delete temporary file @id: @message', [
        '@id' => $file->id(),
        '@message' => $throwable->getMessage(),
      ]);
    }

    $this->fileService->cleanFolder();
  }

}
